"""Stripe Connect integration for developer account management."""

from typing import Optional, Dict, Any
from agenticlypay.client import AgenticlyPayClient


class ConnectManager:
    """Manages Stripe Connect accounts for developers (wrapper around AgenticlyPayClient)."""

    def __init__(self, base_url: str = "https://api.agenticlypay.com"):
        """Initialize the Connect manager."""
        self.client = AgenticlyPayClient(base_url=base_url)

    def create_developer_account(
        self,
        email: str,
        country: str = "US",
        metadata: Optional[Dict[str, str]] = None,
    ) -> Dict[str, Any]:
        """
        Create a new Express account for a developer.

        Args:
            email: Developer email address (must be Zelle-compatible)
            country: Country code (default: US)
            metadata: Additional metadata to store

        Returns:
            Dictionary with account information
        """
        result = self.client.create_account(email=email, country=country, metadata=metadata)
        return result.get("account", {})

    def create_onboarding_link(
        self,
        account_id: str,
        refresh_url: str,
        return_url: str,
        email: str,
    ) -> Dict[str, Any]:
        """
        Create an onboarding link for a developer account.

        Args:
            account_id: Stripe Connect account ID
            refresh_url: URL to redirect if link expires
            return_url: URL to redirect after onboarding completion
            email: Developer email address

        Returns:
            Dictionary with onboarding link URL
        """
        result = self.client.create_onboarding_link(
            account_id=account_id,
            refresh_url=refresh_url,
            return_url=return_url,
            email=email,
        )
        return result.get("onboarding_link", {})

    def get_account_status(self, account_id: str, email: str) -> Dict[str, Any]:
        """
        Get the status of a developer account.

        Args:
            account_id: Stripe Connect account ID
            email: Developer email address

        Returns:
            Dictionary with account status information
        """
        result = self.client.get_account(account_id=account_id, email=email)
        return result.get("account", {})

    def update_account_metadata(
        self, account_id: str, metadata: Dict[str, str], email: str
    ) -> Dict[str, Any]:
        """
        Update metadata for a developer account.

        Args:
            account_id: Stripe Connect account ID
            metadata: Metadata to update
            email: Developer email address

        Returns:
            Updated account information
        """
        # Note: This endpoint may not exist in the API, keeping for backward compatibility
        # In practice, this would need to be implemented in the backend API
        raise NotImplementedError("update_account_metadata is not available via API client")

    def collect_tax_information(
        self, account_id: str, tax_info: Dict[str, Any], email: str
    ) -> Dict[str, Any]:
        """
        Collect and store tax information for a developer.

        Args:
            account_id: Stripe Connect account ID
            tax_info: Tax information dictionary
            email: Developer email address

        Returns:
            Confirmation of tax information storage
        """
        # Note: This endpoint may not exist in the API, keeping for backward compatibility
        # In practice, this would need to be implemented in the backend API
        raise NotImplementedError("collect_tax_information is not available via API client")

