"""Firestore storage utilities for console data."""

from typing import Any, Dict, List, Optional
from datetime import datetime
from google.cloud import firestore
from agenticlypay.config import config

# Initialize Firestore client
db = firestore.Client(project=config.project_id or "agenticlypay-com")


class FirestoreStorage:
    """Firestore-based storage for console data."""

    @staticmethod
    def get_collection(name: str):
        """Get a Firestore collection reference."""
        return db.collection(name)

    @staticmethod
    def create_project(user_id: str, name: str) -> Dict[str, Any]:
        """Create a new project."""
        project_ref = db.collection("projects").document()
        project_data = {
            "id": project_ref.id,
            "user_id": user_id,
            "name": name,
            "created_at": firestore.SERVER_TIMESTAMP,
            "updated_at": firestore.SERVER_TIMESTAMP,
        }
        project_ref.set(project_data)
        return {**project_data, "created_at": datetime.now().isoformat(), "updated_at": datetime.now().isoformat()}

    @staticmethod
    def get_projects(user_id: str) -> List[Dict[str, Any]]:
        """Get all projects for a user."""
        projects = db.collection("projects").where("user_id", "==", user_id).stream()
        result = []
        for doc in projects:
            data = doc.to_dict()
            if data:
                # Convert Firestore timestamps to ISO strings
                if "created_at" in data and hasattr(data["created_at"], "isoformat"):
                    data["created_at"] = data["created_at"].isoformat()
                if "updated_at" in data and hasattr(data["updated_at"], "isoformat"):
                    data["updated_at"] = data["updated_at"].isoformat()
                result.append(data)
        return result

    @staticmethod
    def get_project(project_id: str) -> Optional[Dict[str, Any]]:
        """Get a project by ID."""
        doc = db.collection("projects").document(project_id).get()
        if doc.exists:
            data = doc.to_dict()
            if data:
                if "created_at" in data and hasattr(data["created_at"], "isoformat"):
                    data["created_at"] = data["created_at"].isoformat()
                if "updated_at" in data and hasattr(data["updated_at"], "isoformat"):
                    data["updated_at"] = data["updated_at"].isoformat()
            return data
        return None

    @staticmethod
    def log_usage(email: str, endpoint: str, status_code: int) -> None:
        """Log API usage by email."""
        usage_ref = db.collection("usage").document()
        usage_data = {
            "email": email,
            "endpoint": endpoint,
            "status_code": status_code,
            "timestamp": firestore.SERVER_TIMESTAMP,
        }
        usage_ref.set(usage_data)

    @staticmethod
    def get_usage_by_email(email: str, last_days: int = 30) -> List[Dict[str, Any]]:
        """Get usage data for an email address."""
        from datetime import timedelta

        cutoff = datetime.now() - timedelta(days=last_days)
        usage = (
            db.collection("usage")
            .where("email", "==", email)
            .where("timestamp", ">=", cutoff)
            .order_by("timestamp", direction=firestore.Query.DESCENDING)
            .limit(1000)
            .stream()
        )
        result = []
        for doc in usage:
            data = doc.to_dict()
            if data:
                if "timestamp" in data and hasattr(data["timestamp"], "isoformat"):
                    data["timestamp"] = data["timestamp"].isoformat()
                result.append(data)
        return result

    @staticmethod
    def create_user(user_id: str, email: str, provider: str) -> Dict[str, Any]:
        """Create or update a user."""
        user_ref = db.collection("users").document(user_id)
        user_data = {
            "id": user_id,
            "email": email,
            "provider": provider,
            "created_at": firestore.SERVER_TIMESTAMP,
            "updated_at": firestore.SERVER_TIMESTAMP,
        }
        user_ref.set(user_data, merge=True)
        return user_data

    @staticmethod
    def get_user(user_id: str) -> Optional[Dict[str, Any]]:
        """Get a user by ID."""
        doc = db.collection("users").document(user_id).get()
        if doc.exists:
            return doc.to_dict()
        return None

    @staticmethod
    def get_1099_consent(email: str) -> Optional[Dict[str, Any]]:
        """Get 1099 electronic delivery consent status for an email."""
        doc = db.collection("1099_consents").document(email).get()
        if doc.exists:
            data = doc.to_dict()
            if data:
                if "consent_date" in data and hasattr(data["consent_date"], "isoformat"):
                    data["consent_date"] = data["consent_date"].isoformat()
                if "withdrawal_date" in data and hasattr(data["withdrawal_date"], "isoformat"):
                    data["withdrawal_date"] = data["withdrawal_date"].isoformat()
            return data
        return None

    @staticmethod
    def set_1099_consent(email: str, consented: bool, consent_date: datetime, method: str) -> None:
        """Store 1099 electronic delivery consent."""
        consent_ref = db.collection("1099_consents").document(email)
        consent_data = {
            "email": email,
            "consented": consented,
            "consent_date": consent_date,
            "method": method,  # e.g., "console", "email_link"
            "updated_at": firestore.SERVER_TIMESTAMP,
        }
        consent_ref.set(consent_data, merge=True)

    @staticmethod
    def withdraw_1099_consent(email: str) -> None:
        """Withdraw 1099 electronic delivery consent."""
        consent_ref = db.collection("1099_consents").document(email)
        consent_ref.update({
            "consented": False,
            "withdrawal_date": firestore.SERVER_TIMESTAMP,
            "updated_at": firestore.SERVER_TIMESTAMP,
        })

    @staticmethod
    def store_tax_info(email: str, tax_info: Dict[str, Any]) -> None:
        """Store tax information (SSN/EIN, address, name) for a developer."""
        tax_ref = db.collection("tax_information").document(email)
        tax_data = {
            "email": email,
            **tax_info,
            "updated_at": firestore.SERVER_TIMESTAMP,
        }
        tax_ref.set(tax_data, merge=True)

    @staticmethod
    def get_tax_info(email: str) -> Optional[Dict[str, Any]]:
        """Get tax information for a developer."""
        doc = db.collection("tax_information").document(email).get()
        if doc.exists:
            return doc.to_dict()
        return None

