"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AiInstructionsFile = exports.AiInstructions = exports.AiAgent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("./component");
const file_1 = require("./file");
/**
 * Supported AI coding assistants and their instruction file locations.
 */
var AiAgent;
(function (AiAgent) {
    /**
     * GitHub Copilot - .github/copilot-instructions.md
     */
    AiAgent["GITHUB_COPILOT"] = "GitHub Copilot";
    /**
     * Cursor IDE - .cursor/rules/project.md
     */
    AiAgent["CURSOR"] = "Cursor";
    /**
     * Claude Code - CLAUDE.md
     */
    AiAgent["CLAUDE"] = "Claude";
    /**
     * Amazon Q - .amazonq/rules/project.md
     */
    AiAgent["AMAZON_Q"] = "Amazon Q";
    /**
     * Kiro - .kiro/steering/project.md
     */
    AiAgent["KIRO"] = "Kiro";
})(AiAgent || (exports.AiAgent = AiAgent = {}));
/**
 * Generates instruction files for AI coding assistants with projen-specific guidance.
 *
 * This component creates configuration files that help AI tools like GitHub Copilot,
 * Cursor IDE, Claude Code, and Amazon Q understand that the project is managed by projen
 * and should follow projen conventions.
 *
 * @example
 * const project = new TypeScriptProject({
 *   name: "my-project",
 *   defaultReleaseBranch: "main",
 * });
 *
 * // Basic usage - generates files for all supported AI agents
 * new AiInstructions(project);
 *
 * // Custom usage - specify which agents and add custom instructions
 * new AiInstructions(project, {
 *   agents: [AiAgent.GITHUB_COPILOT, AiAgent.CURSOR],
 *   agentSpecificInstructions: {
 *     [AiAgent.GITHUB_COPILOT]: ["Always use descriptive commit messages."],
 *   },
 * });
 *
 * // Add more instructions after instantiation
 * const ai = new AiInstructions(project);
 * ai.addInstructions("Use functional programming patterns.");
 * ai.addInstructions("Always write comprehensive tests.");
 */
class AiInstructions extends component_1.Component {
    /**
     * Returns projen-specific instructions for AI agents.
     */
    static projen(project) {
        return projenInstructions(project);
    }
    /**
     * Returns development best practices instructions for AI agents.
     */
    static bestPractices(project) {
        return bestPracticesInstructions(project);
    }
    constructor(project, options = {}) {
        super(project);
        this.files = new Map();
        this.agents =
            options.agents ??
                Object.values(AiAgent).filter((v) => typeof v === "string");
        // Assert files for declared agents
        for (const agent of this.agents) {
            this.ensureInstructionsFile(agent);
        }
        if (options.includeDefaultInstructions ?? true) {
            this.addInstructions(AiInstructions.projen(project), AiInstructions.bestPractices(project));
        }
        if (options.instructions) {
            this.addInstructions(...options.instructions);
        }
        if (options.agentSpecificInstructions) {
            for (const [agent, instructions] of Object.entries(options.agentSpecificInstructions)) {
                this.addAgentSpecificInstructions(agent, ...instructions);
            }
        }
    }
    /**
     * Create or return the instructions file.
     */
    ensureInstructionsFile(agent) {
        if (this.files.has(agent)) {
            return this.files.get(agent);
        }
        const filePath = this.getAgentFilePath(agent);
        const file = new AiInstructionsFile(this.project, filePath, {
            committed: true,
            readonly: true,
        });
        this.files.set(agent, file);
        this.project.addPackageIgnore(file.path);
        return file;
    }
    /**
     * Adds instructions that will be included for all selected AI agents.
     *
     * @param instructions The instructions to add.
     * @example
     * aiInstructions.addInstructions("Always use TypeScript strict mode.");
     * aiInstructions.addInstructions("Prefer functional programming.", "Avoid mutations.");
     */
    addInstructions(...instructions) {
        for (const agent of this.files.keys()) {
            this.addAgentSpecificInstructions(agent, ...instructions);
        }
    }
    /**
     * Add instructions for a specific AI agent.
     *
     * This can also be used to add instructions for an AI agent that was previously not enabled.
     *
     * @param agent The AI agent to add instructions for
     * @param instructions The instruction(s) to add
     * @example
     * aiInstructions.addAgentSpecificInstructions(AiAgent.GITHUB_COPILOT, "Use descriptive commit messages.");
     */
    addAgentSpecificInstructions(agent, ...instructions) {
        const file = this.ensureInstructionsFile(agent);
        file.addInstructions(...instructions);
    }
    /**
     * Get the file path for a given AI agent.
     */
    getAgentFilePath(agent) {
        switch (agent) {
            case AiAgent.GITHUB_COPILOT:
                return ".github/copilot-instructions.md";
            case AiAgent.CURSOR:
                return ".cursor/rules/project.md";
            case AiAgent.CLAUDE:
                return "CLAUDE.md";
            case AiAgent.AMAZON_Q:
                return ".amazonq/rules/project.md";
            case AiAgent.KIRO:
                return ".kiro/steering/project.md";
            default:
                throw new Error(`Unknown AI agent: ${agent}`);
        }
    }
}
exports.AiInstructions = AiInstructions;
_a = JSII_RTTI_SYMBOL_1;
AiInstructions[_a] = { fqn: "projen.AiInstructions", version: "0.98.9" };
class AiInstructionsFile extends file_1.FileBase {
    constructor() {
        super(...arguments);
        this.instructions = [];
    }
    /**
     * Adds instructions to the instruction file.
     */
    addInstructions(...instructions) {
        this.instructions.push(...instructions);
    }
    synthesizeContent(resolver) {
        return resolver.resolve(this.instructions).join("\n\n") + "\n";
    }
}
exports.AiInstructionsFile = AiInstructionsFile;
_b = JSII_RTTI_SYMBOL_1;
AiInstructionsFile[_b] = { fqn: "projen.AiInstructionsFile", version: "0.98.9" };
function bestPracticesInstructions(project) {
    const projenCommand = project.projenCommand;
    return `# Development Best Practices

- **Always run build after changes**: After modifying any source or test file, run \`${projenCommand} build\` to ensure your changes compile and pass all tests.
- **Task completion criteria**: A task is not considered complete until:
  - All tests pass (\`${projenCommand} test\`)
  - There are no compilation errors (\`${projenCommand} compile\`)
  - There are no linting errors (usually part of the build, if not, run the linter defined in tasks.json)
  - The full build succeeds (\`${projenCommand} build\`)`;
}
function projenInstructions(project) {
    const projenCommand = project.projenCommand;
    return `# Projen-managed Project Instructions

This project is managed by [projen](https://github.com/projen/projen), a project configuration management tool.

## Important Guidelines

### Task Execution

- **Always use projen for task execution**: Run tasks using \`${projenCommand} <task-name>\` instead of directly using npm, yarn, or other package managers.
- **Check available tasks**: Look in \`.projen/tasks.json\` to see all available tasks, their descriptions, and steps.
- **Common tasks**:
  - \`${projenCommand}\` - Synthesize project configuration files
  - \`${projenCommand} build\` - Builds the project, including running tests
  - \`${projenCommand} test\` - Runs tests only
  - \`${projenCommand} compile\` - Compiles the source code only

### File Modifications

- **DO NOT manually edit generated files**: Files marked with a comment like "~~ Generated by projen. To modify..." should never be edited directly.
- **Modify configuration in .projenrc**: To change project configuration, always edit the \`.projenrc.ts\`, \`.projenrc.py\` or \`.projenrc.json\` etc. file and then run \`${projenCommand}\` to regenerate the project files.
- **Check .projenrc first**: Before suggesting changes to package.json, tsconfig.json, or other configuration files, always check if these are managed by projen and suggest changes to .projenrc instead.

### Dependencies

- **Add dependencies through projen**: Use the projen configuration to add dependencies instead of manually editing package.json or using npm/yarn install directly.
- **Example**: In .projenrc, use methods like \`addDeps()\`, \`addDevDeps()\`, or \`addPeerDeps()\` to add dependencies.

### Workflow

1. Make changes to .projenrc configuration file
2. Run \`${projenCommand}\` to synthesize and update generated files
3. Review the changes
4. Commit both .projenrc and the generated files

## Projen Configuration

This project's configuration is defined in the .projenrc file at the root of the repository. All project metadata, dependencies, scripts, and tooling configuration should be managed through this file.

## Additional Resources

- [Projen Documentation](https://projen.io)
- [Projen GitHub Repository](https://github.com/projen/projen)`;
}
//# sourceMappingURL=data:application/json;base64,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