"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerCompose = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const docker_compose_port_1 = require("./docker-compose-port");
const docker_compose_render_1 = require("./docker-compose-render");
const docker_compose_service_1 = require("./docker-compose-service");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * Create a docker-compose YAML file.
 */
class DockerCompose extends component_1.Component {
    /**
     * Depends on a service name.
     */
    static serviceName(serviceName) {
        return {
            serviceName,
        };
    }
    /**
     * Create a port mapping.
     * @param publishedPort Published port number
     * @param targetPort Container's port number
     * @param options Port mapping options
     */
    static portMapping(publishedPort, targetPort, options) {
        const protocol = options?.protocol ?? docker_compose_port_1.DockerComposeProtocol.TCP;
        return {
            target: targetPort,
            published: publishedPort,
            protocol: protocol,
            mode: "host",
        };
    }
    /**
     * Create a bind volume that binds a host path to the target path in the container.
     * @param sourcePath Host path name
     * @param targetPath Target path name
     */
    static bindVolume(sourcePath, targetPath) {
        return {
            bind(_volumeInfo) {
                return {
                    type: "bind",
                    source: sourcePath,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named volume and mount it to the target path. If you use this
     * named volume in several services, the volume will be shared. In this
     * case, the volume configuration of the first-provided options are used.
     *
     * @param volumeName Name of the volume
     * @param targetPath Target path
     * @param options volume configuration (default: docker compose defaults)
     */
    static namedVolume(volumeName, targetPath, options = {}) {
        return {
            bind(volumeInfo) {
                volumeInfo.addVolumeConfiguration(volumeName, options);
                return {
                    type: "volume",
                    source: volumeName,
                    target: targetPath,
                };
            },
        };
    }
    /**
     * Create a named network and mount it to the target path. If you use this
     * named network in several services, the network will be shared. In this
     * case, the network configuration of the first-provided options are used.
     *
     * @param networkName Name of the network
     * @param options network configuration
     */
    static network(networkName, options = {}) {
        return {
            bind(networkInfo) {
                networkInfo.addNetworkConfiguration(networkName, options);
                return networkName;
            },
        };
    }
    constructor(project, props) {
        super(project);
        const nameSuffix = props?.nameSuffix ? `${props.nameSuffix}.yml` : "yml";
        this.file = new yaml_1.YamlFile(project, `docker-compose.${nameSuffix}`, {
            committed: true,
            readonly: true,
            obj: () => this._synthesizeDockerCompose(),
        });
        if (props?.schemaVersion && !parseFloat(props.schemaVersion)) {
            throw Error("Version tag needs to be a number");
        }
        this.version = props?.schemaVersion;
        this.services = {};
        // Add the services provided via the constructor argument.
        const initialServices = props?.services ?? {};
        for (const [name, serviceDescription] of Object.entries(initialServices)) {
            this.addService(name, serviceDescription);
        }
    }
    /**
     * Add a service to the docker-compose file.
     * @param serviceName name of the service
     * @param description a service description
     */
    addService(serviceName, description) {
        const service = new docker_compose_service_1.DockerComposeService(serviceName, description);
        this.services[serviceName] = service;
        return service;
    }
    /**
     * @internal
     */
    _synthesizeDockerCompose() {
        if (Object.keys(this.services).length === 0) {
            throw new Error("DockerCompose requires at least one service");
        }
        return (0, docker_compose_render_1.renderDockerComposeFile)(this.services, this.version);
    }
}
exports.DockerCompose = DockerCompose;
_a = JSII_RTTI_SYMBOL_1;
DockerCompose[_a] = { fqn: "projen.DockerCompose", version: "0.98.9" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9ja2VyLWNvbXBvc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZG9ja2VyLWNvbXBvc2UvZG9ja2VyLWNvbXBvc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFLQSwrREFJK0I7QUFDL0IsbUVBQWtFO0FBQ2xFLHFFQUlrQztBQU9sQyw0Q0FBeUM7QUFFekMsa0NBQW1DO0FBMkJuQzs7R0FFRztBQUNILE1BQWEsYUFBYyxTQUFRLHFCQUFTO0lBQzFDOztPQUVHO0lBQ0gsTUFBTSxDQUFDLFdBQVcsQ0FBQyxXQUFtQjtRQUNwQyxPQUFPO1lBQ0wsV0FBVztTQUNaLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxNQUFNLENBQUMsV0FBVyxDQUNoQixhQUFxQixFQUNyQixVQUFrQixFQUNsQixPQUF5QztRQUV6QyxNQUFNLFFBQVEsR0FBRyxPQUFPLEVBQUUsUUFBUSxJQUFJLDJDQUFxQixDQUFDLEdBQUcsQ0FBQztRQUVoRSxPQUFPO1lBQ0wsTUFBTSxFQUFFLFVBQVU7WUFDbEIsU0FBUyxFQUFFLGFBQWE7WUFDeEIsUUFBUSxFQUFFLFFBQVE7WUFDbEIsSUFBSSxFQUFFLE1BQU07U0FDYixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsVUFBVSxDQUNmLFVBQWtCLEVBQ2xCLFVBQWtCO1FBRWxCLE9BQU87WUFDTCxJQUFJLENBQUMsV0FBdUM7Z0JBQzFDLE9BQU87b0JBQ0wsSUFBSSxFQUFFLE1BQU07b0JBQ1osTUFBTSxFQUFFLFVBQVU7b0JBQ2xCLE1BQU0sRUFBRSxVQUFVO2lCQUNuQixDQUFDO1lBQ0osQ0FBQztTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSCxNQUFNLENBQUMsV0FBVyxDQUNoQixVQUFrQixFQUNsQixVQUFrQixFQUNsQixVQUFxQyxFQUFFO1FBRXZDLE9BQU87WUFDTCxJQUFJLENBQUMsVUFBc0M7Z0JBQ3pDLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxVQUFVLEVBQUUsT0FBTyxDQUFDLENBQUM7Z0JBRXZELE9BQU87b0JBQ0wsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsTUFBTSxFQUFFLFVBQVU7b0JBQ2xCLE1BQU0sRUFBRSxVQUFVO2lCQUNuQixDQUFDO1lBQ0osQ0FBQztTQUNGLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNILE1BQU0sQ0FBQyxPQUFPLENBQ1osV0FBbUIsRUFDbkIsVUFBc0MsRUFBRTtRQUV4QyxPQUFPO1lBQ0wsSUFBSSxDQUFDLFdBQXdDO2dCQUMzQyxXQUFXLENBQUMsdUJBQXVCLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO2dCQUUxRCxPQUFPLFdBQVcsQ0FBQztZQUNyQixDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFVRCxZQUFZLE9BQWdCLEVBQUUsS0FBMEI7UUFDdEQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRWYsTUFBTSxVQUFVLEdBQUcsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFNLENBQUMsVUFBVSxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQztRQUMxRSxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZUFBUSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsVUFBVSxFQUFFLEVBQUU7WUFDaEUsU0FBUyxFQUFFLElBQUk7WUFDZixRQUFRLEVBQUUsSUFBSTtZQUNkLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsd0JBQXdCLEVBQUU7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxLQUFLLEVBQUUsYUFBYSxJQUFJLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDO1lBQzdELE1BQU0sS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDbEQsQ0FBQztRQUNELElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxFQUFFLGFBQWEsQ0FBQztRQUNwQyxJQUFJLENBQUMsUUFBUSxHQUFHLEVBQUUsQ0FBQztRQUVuQiwwREFBMEQ7UUFDMUQsTUFBTSxlQUFlLEdBQUcsS0FBSyxFQUFFLFFBQVEsSUFBSSxFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDO1lBQ3pFLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDNUMsQ0FBQztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksVUFBVSxDQUNmLFdBQW1CLEVBQ25CLFdBQTRDO1FBRTVDLE1BQU0sT0FBTyxHQUFHLElBQUksNkNBQW9CLENBQUMsV0FBVyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUcsT0FBTyxDQUFDO1FBQ3JDLE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFRDs7T0FFRztJQUNILHdCQUF3QjtRQUN0QixJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxDQUFDLENBQUM7UUFDakUsQ0FBQztRQUVELE9BQU8sSUFBQSwrQ0FBdUIsRUFBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUM5RCxDQUFDOztBQXpKSCxzQ0EwSkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xuICBEb2NrZXJDb21wb3NlTmV0d29ya0NvbmZpZyxcbiAgSURvY2tlckNvbXBvc2VOZXR3b3JrQmluZGluZyxcbiAgSURvY2tlckNvbXBvc2VOZXR3b3JrQ29uZmlnLFxufSBmcm9tIFwiLi9kb2NrZXItY29tcG9zZS1uZXR3b3JrXCI7XG5pbXBvcnQge1xuICBEb2NrZXJDb21wb3NlUG9ydE1hcHBpbmdPcHRpb25zLFxuICBEb2NrZXJDb21wb3NlUHJvdG9jb2wsXG4gIERvY2tlckNvbXBvc2VTZXJ2aWNlUG9ydCxcbn0gZnJvbSBcIi4vZG9ja2VyLWNvbXBvc2UtcG9ydFwiO1xuaW1wb3J0IHsgcmVuZGVyRG9ja2VyQ29tcG9zZUZpbGUgfSBmcm9tIFwiLi9kb2NrZXItY29tcG9zZS1yZW5kZXJcIjtcbmltcG9ydCB7XG4gIERvY2tlckNvbXBvc2VTZXJ2aWNlLFxuICBEb2NrZXJDb21wb3NlU2VydmljZURlc2NyaXB0aW9uLFxuICBJRG9ja2VyQ29tcG9zZVNlcnZpY2VOYW1lLFxufSBmcm9tIFwiLi9kb2NrZXItY29tcG9zZS1zZXJ2aWNlXCI7XG5pbXBvcnQge1xuICBEb2NrZXJDb21wb3NlVm9sdW1lQ29uZmlnLFxuICBEb2NrZXJDb21wb3NlVm9sdW1lTW91bnQsXG4gIElEb2NrZXJDb21wb3NlVm9sdW1lQmluZGluZyxcbiAgSURvY2tlckNvbXBvc2VWb2x1bWVDb25maWcsXG59IGZyb20gXCIuL2RvY2tlci1jb21wb3NlLXZvbHVtZVwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4uL2NvbXBvbmVudFwiO1xuaW1wb3J0IHsgUHJvamVjdCB9IGZyb20gXCIuLi9wcm9qZWN0XCI7XG5pbXBvcnQgeyBZYW1sRmlsZSB9IGZyb20gXCIuLi95YW1sXCI7XG5cbi8qKlxuICogUHJvcHMgZm9yIERvY2tlckNvbXBvc2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VyQ29tcG9zZVByb3BzIHtcbiAgLyoqXG4gICAqIEEgbmFtZSB0byBhZGQgdG8gdGhlIGRvY2tlci1jb21wb3NlLnltbCBmaWxlbmFtZS5cbiAgICogQGV4YW1wbGUgJ215bmFtZScgeWllbGRzICdkb2NrZXItY29tcG9zZS5teW5hbWUueW1sJ1xuICAgKiBAZGVmYXVsdCAtIG5vIG5hbWUgaXMgYWRkZWRcbiAgICovXG4gIHJlYWRvbmx5IG5hbWVTdWZmaXg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERvY2tlciBDb21wb3NlIHNjaGVtYSB2ZXJzaW9uIGRvIGJlIHVzZWRcbiAgICogQGRlZmF1bHQgLSBubyB2ZXJzaW9uIGlzIHByb3ZpZGVkXG4gICAqIEBkZXByZWNhdGVkIC0gVGhlIHRvcCBsZXZlbCBgdmVyc2lvbmAgZmllbGQgaXMgb2Jzb2xldGUgcGVyIHRoZSBDb21wb3NlIFNwZWNpZmljYXRpb24uXG4gICAqIHtAbGluayBodHRwczovL2dpdGh1Yi5jb20vY29tcG9zZS1zcGVjL2NvbXBvc2Utc3BlYy9ibG9iL21hc3Rlci9zcGVjLm1kI3ZlcnNpb24tYW5kLW5hbWUtdG9wLWxldmVsLWVsZW1lbnRzIENvbXBvc2UgU3BlY2lmaWNhdGlvbn1cbiAgICovXG4gIHJlYWRvbmx5IHNjaGVtYVZlcnNpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNlcnZpY2UgZGVzY3JpcHRpb25zLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZXM/OiBSZWNvcmQ8c3RyaW5nLCBEb2NrZXJDb21wb3NlU2VydmljZURlc2NyaXB0aW9uPjtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBkb2NrZXItY29tcG9zZSBZQU1MIGZpbGUuXG4gKi9cbmV4cG9ydCBjbGFzcyBEb2NrZXJDb21wb3NlIGV4dGVuZHMgQ29tcG9uZW50IHtcbiAgLyoqXG4gICAqIERlcGVuZHMgb24gYSBzZXJ2aWNlIG5hbWUuXG4gICAqL1xuICBzdGF0aWMgc2VydmljZU5hbWUoc2VydmljZU5hbWU6IHN0cmluZyk6IElEb2NrZXJDb21wb3NlU2VydmljZU5hbWUge1xuICAgIHJldHVybiB7XG4gICAgICBzZXJ2aWNlTmFtZSxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHBvcnQgbWFwcGluZy5cbiAgICogQHBhcmFtIHB1Ymxpc2hlZFBvcnQgUHVibGlzaGVkIHBvcnQgbnVtYmVyXG4gICAqIEBwYXJhbSB0YXJnZXRQb3J0IENvbnRhaW5lcidzIHBvcnQgbnVtYmVyXG4gICAqIEBwYXJhbSBvcHRpb25zIFBvcnQgbWFwcGluZyBvcHRpb25zXG4gICAqL1xuICBzdGF0aWMgcG9ydE1hcHBpbmcoXG4gICAgcHVibGlzaGVkUG9ydDogbnVtYmVyLFxuICAgIHRhcmdldFBvcnQ6IG51bWJlcixcbiAgICBvcHRpb25zPzogRG9ja2VyQ29tcG9zZVBvcnRNYXBwaW5nT3B0aW9uc1xuICApOiBEb2NrZXJDb21wb3NlU2VydmljZVBvcnQge1xuICAgIGNvbnN0IHByb3RvY29sID0gb3B0aW9ucz8ucHJvdG9jb2wgPz8gRG9ja2VyQ29tcG9zZVByb3RvY29sLlRDUDtcblxuICAgIHJldHVybiB7XG4gICAgICB0YXJnZXQ6IHRhcmdldFBvcnQsXG4gICAgICBwdWJsaXNoZWQ6IHB1Ymxpc2hlZFBvcnQsXG4gICAgICBwcm90b2NvbDogcHJvdG9jb2wsXG4gICAgICBtb2RlOiBcImhvc3RcIixcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIGJpbmQgdm9sdW1lIHRoYXQgYmluZHMgYSBob3N0IHBhdGggdG8gdGhlIHRhcmdldCBwYXRoIGluIHRoZSBjb250YWluZXIuXG4gICAqIEBwYXJhbSBzb3VyY2VQYXRoIEhvc3QgcGF0aCBuYW1lXG4gICAqIEBwYXJhbSB0YXJnZXRQYXRoIFRhcmdldCBwYXRoIG5hbWVcbiAgICovXG4gIHN0YXRpYyBiaW5kVm9sdW1lKFxuICAgIHNvdXJjZVBhdGg6IHN0cmluZyxcbiAgICB0YXJnZXRQYXRoOiBzdHJpbmdcbiAgKTogSURvY2tlckNvbXBvc2VWb2x1bWVCaW5kaW5nIHtcbiAgICByZXR1cm4ge1xuICAgICAgYmluZChfdm9sdW1lSW5mbzogSURvY2tlckNvbXBvc2VWb2x1bWVDb25maWcpOiBEb2NrZXJDb21wb3NlVm9sdW1lTW91bnQge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIHR5cGU6IFwiYmluZFwiLFxuICAgICAgICAgIHNvdXJjZTogc291cmNlUGF0aCxcbiAgICAgICAgICB0YXJnZXQ6IHRhcmdldFBhdGgsXG4gICAgICAgIH07XG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgbmFtZWQgdm9sdW1lIGFuZCBtb3VudCBpdCB0byB0aGUgdGFyZ2V0IHBhdGguIElmIHlvdSB1c2UgdGhpc1xuICAgKiBuYW1lZCB2b2x1bWUgaW4gc2V2ZXJhbCBzZXJ2aWNlcywgdGhlIHZvbHVtZSB3aWxsIGJlIHNoYXJlZC4gSW4gdGhpc1xuICAgKiBjYXNlLCB0aGUgdm9sdW1lIGNvbmZpZ3VyYXRpb24gb2YgdGhlIGZpcnN0LXByb3ZpZGVkIG9wdGlvbnMgYXJlIHVzZWQuXG4gICAqXG4gICAqIEBwYXJhbSB2b2x1bWVOYW1lIE5hbWUgb2YgdGhlIHZvbHVtZVxuICAgKiBAcGFyYW0gdGFyZ2V0UGF0aCBUYXJnZXQgcGF0aFxuICAgKiBAcGFyYW0gb3B0aW9ucyB2b2x1bWUgY29uZmlndXJhdGlvbiAoZGVmYXVsdDogZG9ja2VyIGNvbXBvc2UgZGVmYXVsdHMpXG4gICAqL1xuICBzdGF0aWMgbmFtZWRWb2x1bWUoXG4gICAgdm9sdW1lTmFtZTogc3RyaW5nLFxuICAgIHRhcmdldFBhdGg6IHN0cmluZyxcbiAgICBvcHRpb25zOiBEb2NrZXJDb21wb3NlVm9sdW1lQ29uZmlnID0ge31cbiAgKTogSURvY2tlckNvbXBvc2VWb2x1bWVCaW5kaW5nIHtcbiAgICByZXR1cm4ge1xuICAgICAgYmluZCh2b2x1bWVJbmZvOiBJRG9ja2VyQ29tcG9zZVZvbHVtZUNvbmZpZyk6IERvY2tlckNvbXBvc2VWb2x1bWVNb3VudCB7XG4gICAgICAgIHZvbHVtZUluZm8uYWRkVm9sdW1lQ29uZmlndXJhdGlvbih2b2x1bWVOYW1lLCBvcHRpb25zKTtcblxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIHR5cGU6IFwidm9sdW1lXCIsXG4gICAgICAgICAgc291cmNlOiB2b2x1bWVOYW1lLFxuICAgICAgICAgIHRhcmdldDogdGFyZ2V0UGF0aCxcbiAgICAgICAgfTtcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuYW1lZCBuZXR3b3JrIGFuZCBtb3VudCBpdCB0byB0aGUgdGFyZ2V0IHBhdGguIElmIHlvdSB1c2UgdGhpc1xuICAgKiBuYW1lZCBuZXR3b3JrIGluIHNldmVyYWwgc2VydmljZXMsIHRoZSBuZXR3b3JrIHdpbGwgYmUgc2hhcmVkLiBJbiB0aGlzXG4gICAqIGNhc2UsIHRoZSBuZXR3b3JrIGNvbmZpZ3VyYXRpb24gb2YgdGhlIGZpcnN0LXByb3ZpZGVkIG9wdGlvbnMgYXJlIHVzZWQuXG4gICAqXG4gICAqIEBwYXJhbSBuZXR3b3JrTmFtZSBOYW1lIG9mIHRoZSBuZXR3b3JrXG4gICAqIEBwYXJhbSBvcHRpb25zIG5ldHdvcmsgY29uZmlndXJhdGlvblxuICAgKi9cbiAgc3RhdGljIG5ldHdvcmsoXG4gICAgbmV0d29ya05hbWU6IHN0cmluZyxcbiAgICBvcHRpb25zOiBEb2NrZXJDb21wb3NlTmV0d29ya0NvbmZpZyA9IHt9XG4gICk6IElEb2NrZXJDb21wb3NlTmV0d29ya0JpbmRpbmcge1xuICAgIHJldHVybiB7XG4gICAgICBiaW5kKG5ldHdvcmtJbmZvOiBJRG9ja2VyQ29tcG9zZU5ldHdvcmtDb25maWcpOiBzdHJpbmcge1xuICAgICAgICBuZXR3b3JrSW5mby5hZGROZXR3b3JrQ29uZmlndXJhdGlvbihuZXR3b3JrTmFtZSwgb3B0aW9ucyk7XG5cbiAgICAgICAgcmV0dXJuIG5ldHdvcmtOYW1lO1xuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBEb2NrZXIgQ29tcG9zZSBmaWxlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZmlsZTogWWFtbEZpbGU7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzZXJ2aWNlczogUmVjb3JkPHN0cmluZywgRG9ja2VyQ29tcG9zZVNlcnZpY2U+O1xuICBwcml2YXRlIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IocHJvamVjdDogUHJvamVjdCwgcHJvcHM/OiBEb2NrZXJDb21wb3NlUHJvcHMpIHtcbiAgICBzdXBlcihwcm9qZWN0KTtcblxuICAgIGNvbnN0IG5hbWVTdWZmaXggPSBwcm9wcz8ubmFtZVN1ZmZpeCA/IGAke3Byb3BzIS5uYW1lU3VmZml4fS55bWxgIDogXCJ5bWxcIjtcbiAgICB0aGlzLmZpbGUgPSBuZXcgWWFtbEZpbGUocHJvamVjdCwgYGRvY2tlci1jb21wb3NlLiR7bmFtZVN1ZmZpeH1gLCB7XG4gICAgICBjb21taXR0ZWQ6IHRydWUsXG4gICAgICByZWFkb25seTogdHJ1ZSxcbiAgICAgIG9iajogKCkgPT4gdGhpcy5fc3ludGhlc2l6ZURvY2tlckNvbXBvc2UoKSxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcz8uc2NoZW1hVmVyc2lvbiAmJiAhcGFyc2VGbG9hdChwcm9wcy5zY2hlbWFWZXJzaW9uKSkge1xuICAgICAgdGhyb3cgRXJyb3IoXCJWZXJzaW9uIHRhZyBuZWVkcyB0byBiZSBhIG51bWJlclwiKTtcbiAgICB9XG4gICAgdGhpcy52ZXJzaW9uID0gcHJvcHM/LnNjaGVtYVZlcnNpb247XG4gICAgdGhpcy5zZXJ2aWNlcyA9IHt9O1xuXG4gICAgLy8gQWRkIHRoZSBzZXJ2aWNlcyBwcm92aWRlZCB2aWEgdGhlIGNvbnN0cnVjdG9yIGFyZ3VtZW50LlxuICAgIGNvbnN0IGluaXRpYWxTZXJ2aWNlcyA9IHByb3BzPy5zZXJ2aWNlcyA/PyB7fTtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCBzZXJ2aWNlRGVzY3JpcHRpb25dIG9mIE9iamVjdC5lbnRyaWVzKGluaXRpYWxTZXJ2aWNlcykpIHtcbiAgICAgIHRoaXMuYWRkU2VydmljZShuYW1lLCBzZXJ2aWNlRGVzY3JpcHRpb24pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBzZXJ2aWNlIHRvIHRoZSBkb2NrZXItY29tcG9zZSBmaWxlLlxuICAgKiBAcGFyYW0gc2VydmljZU5hbWUgbmFtZSBvZiB0aGUgc2VydmljZVxuICAgKiBAcGFyYW0gZGVzY3JpcHRpb24gYSBzZXJ2aWNlIGRlc2NyaXB0aW9uXG4gICAqL1xuICBwdWJsaWMgYWRkU2VydmljZShcbiAgICBzZXJ2aWNlTmFtZTogc3RyaW5nLFxuICAgIGRlc2NyaXB0aW9uOiBEb2NrZXJDb21wb3NlU2VydmljZURlc2NyaXB0aW9uXG4gICk6IERvY2tlckNvbXBvc2VTZXJ2aWNlIHtcbiAgICBjb25zdCBzZXJ2aWNlID0gbmV3IERvY2tlckNvbXBvc2VTZXJ2aWNlKHNlcnZpY2VOYW1lLCBkZXNjcmlwdGlvbik7XG4gICAgdGhpcy5zZXJ2aWNlc1tzZXJ2aWNlTmFtZV0gPSBzZXJ2aWNlO1xuICAgIHJldHVybiBzZXJ2aWNlO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgX3N5bnRoZXNpemVEb2NrZXJDb21wb3NlKCk6IG9iamVjdCB7XG4gICAgaWYgKE9iamVjdC5rZXlzKHRoaXMuc2VydmljZXMpLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiRG9ja2VyQ29tcG9zZSByZXF1aXJlcyBhdCBsZWFzdCBvbmUgc2VydmljZVwiKTtcbiAgICB9XG5cbiAgICByZXR1cm4gcmVuZGVyRG9ja2VyQ29tcG9zZUZpbGUodGhpcy5zZXJ2aWNlcywgdGhpcy52ZXJzaW9uKTtcbiAgfVxufVxuXG4vKipcbiAqIEJ1aWxkIGFyZ3VtZW50cyBmb3IgY3JlYXRpbmcgYSBkb2NrZXIgaW1hZ2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VyQ29tcG9zZUJ1aWxkIHtcbiAgLyoqXG4gICAqIERvY2tlciBidWlsZCBjb250ZXh0IGRpcmVjdG9yeS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbnRleHQ6IHN0cmluZztcblxuICAvKipcbiAgICogQSBkb2NrZXJmaWxlIHRvIGJ1aWxkIGZyb20uXG4gICAqIEBkZWZhdWx0IFwiRG9ja2VyZmlsZVwiXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJmaWxlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBCdWlsZCBhcmdzLlxuICAgKiBAZGVmYXVsdCAtIG5vbmUgYXJlIHByb3ZpZGVkXG4gICAqL1xuICByZWFkb25seSBhcmdzPzogUmVjb3JkPHN0cmluZywgc3RyaW5nPjtcbn1cbiJdfQ==