"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const projenrc_1 = require("./projenrc");
const util_1 = require("./util");
const constructs_1 = require("./util/constructs");
class FileBase extends component_1.Component {
    /**
     * The projen marker, used to identify files as projen-generated.
     *
     * Value is undefined if the project is being ejected.
     */
    get marker() {
        if (this.project.ejected || !this.shouldAddMarker) {
            return undefined;
        }
        // `marker` is empty if project is being ejected or if explicitly disabled
        const projenrc = projenrc_1.ProjenrcFile.of(this.project)?.filePath ?? common_1.DEFAULT_PROJEN_RC_JS_FILENAME;
        return `${common_1.PROJEN_MARKER}. To modify, edit ${projenrc} and run "${this.project.projenCommand}".`;
    }
    constructor(scope, filePath, options = {}) {
        const project = (0, constructs_1.findClosestProject)(scope, new.target.name);
        const root = project.root;
        const normalizedPath = path.normalize(filePath);
        const projectPath = (0, util_1.normalizePersistedPath)(normalizedPath);
        const absolutePath = path.resolve(project.outdir, projectPath);
        const relativeProjectPath = path.relative(root.outdir, absolutePath);
        const rootProjectPath = (0, util_1.normalizePersistedPath)(relativeProjectPath);
        const autoId = `${new.target.name}@${projectPath}`;
        // Before actually creating the file, ensure the file path is unique within the full project tree
        // This is required because projects can create files inside their subprojects
        if (root.tryFindFile(absolutePath) || scope.node.tryFindChild(autoId)) {
            throw new Error(`There is already a file under ${rootProjectPath}`);
        }
        super(scope, autoId);
        this.node.addMetadata("type", "file");
        this.node.addMetadata("path", rootProjectPath);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = projectPath;
        this.absolutePath = absolutePath;
        this.shouldAddMarker = options.marker ?? true;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? project.commitGenerated ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.annotateGenerated(`/${filePath}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => (0, _resolve_1.resolve)(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            (0, fs_1.rmSync)(filePath, { force: true, recursive: true });
            return;
        }
        // check if the file was changed.
        const prev = (0, util_1.tryReadFileSync)(filePath);
        const prevReadonly = !(0, util_1.isWritable)(filePath);
        const successfulExecutableAssertion = (0, util_1.assertExecutablePermissions)(filePath, this.executable);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            successfulExecutableAssertion) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        (0, util_1.writeFile)(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = (0, util_1.tryReadFileSync)(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.98.9" };
//# sourceMappingURL=data:application/json;base64,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