"use strict";
// @see https://docs.gitlab.com/ee/ci/yaml/
// Schema from: https://gitlab.com/gitlab-org/gitlab/-/raw/master/app/assets/javascripts/editor/schema/ci.json
// Initial TypeScript Interfaces from: https://app.quicktype.io/#
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowWhen = exports.Strategy = exports.LinkType = exports.DefaultElement = exports.KubernetesEnum = exports.DeploymentTier = exports.Action = exports.JobWhen = exports.PullPolicy = exports.CacheWhen = exports.CachePolicy = void 0;
/**
 * Configure the upload and download behaviour of a cache.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachepolicy
 */
var CachePolicy;
(function (CachePolicy) {
    /** Only download the cache when the job starts, but never upload changes when the job finishes. */
    CachePolicy["PULL"] = "pull";
    /** Only upload a cache when the job finishes, but never download the cache when the job starts. */
    CachePolicy["PUSH"] = "push";
    /** The job downloads the cache when the job starts, and uploads changes to the cache when the job ends. */
    CachePolicy["PULL_PUSH"] = "pull-push";
})(CachePolicy || (exports.CachePolicy = CachePolicy = {}));
/**
 * Configure when artifacts are uploaded depended on job status.
 * @see https://docs.gitlab.com/ee/ci/yaml/#cachewhen
 */
var CacheWhen;
(function (CacheWhen) {
    /** Upload artifacts regardless of job status. */
    CacheWhen["ALWAYS"] = "always";
    /** Upload artifacts only when the job fails. */
    CacheWhen["ON_FAILURE"] = "on_failure";
    /** Upload artifacts only when the job succeeds (this is the default). */
    CacheWhen["ON_SUCCESS"] = "on_success";
})(CacheWhen || (exports.CacheWhen = CacheWhen = {}));
/**
 * Describes the conditions for when to pull an image.
 * @see https://docs.gitlab.com/ee/ci/yaml/#servicepull_policy
 */
var PullPolicy;
(function (PullPolicy) {
    PullPolicy["ALWAYS"] = "always";
    PullPolicy["NEVER"] = "never";
    PullPolicy["IF_NOT_PRESENT"] = "if-not-present";
})(PullPolicy || (exports.PullPolicy = PullPolicy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * @see https://docs.gitlab.com/ee/ci/yaml/#when
 */
var JobWhen;
(function (JobWhen) {
    JobWhen["ALWAYS"] = "always";
    JobWhen["DELAYED"] = "delayed";
    JobWhen["MANUAL"] = "manual";
    JobWhen["NEVER"] = "never";
    JobWhen["ON_FAILURE"] = "on_failure";
    JobWhen["ON_SUCCESS"] = "on_success";
})(JobWhen || (exports.JobWhen = JobWhen = {}));
/**
 * Specifies what this job will do. 'start' (default) indicates the job will start the
 * deployment. 'prepare' indicates this will not affect the deployment. 'stop' indicates
 * this will stop the deployment.
 */
var Action;
(function (Action) {
    Action["PREPARE"] = "prepare";
    Action["START"] = "start";
    Action["STOP"] = "stop";
})(Action || (exports.Action = Action = {}));
/**
 * Explicitly specifies the tier of the deployment environment if non-standard environment
 * name is used.
 */
var DeploymentTier;
(function (DeploymentTier) {
    DeploymentTier["DEVELOPMENT"] = "development";
    DeploymentTier["OTHER"] = "other";
    DeploymentTier["PRODUCTION"] = "production";
    DeploymentTier["STAGING"] = "staging";
    DeploymentTier["TESTING"] = "testing";
})(DeploymentTier || (exports.DeploymentTier = DeploymentTier = {}));
/**
 * Filter job based on if Kubernetes integration is active.
 */
var KubernetesEnum;
(function (KubernetesEnum) {
    KubernetesEnum["ACTIVE"] = "active";
})(KubernetesEnum || (exports.KubernetesEnum = KubernetesEnum = {}));
var DefaultElement;
(function (DefaultElement) {
    DefaultElement["AFTER_SCRIPT"] = "after_script";
    DefaultElement["ARTIFACTS"] = "artifacts";
    DefaultElement["BEFORE_SCRIPT"] = "before_script";
    DefaultElement["CACHE"] = "cache";
    DefaultElement["IMAGE"] = "image";
    DefaultElement["INTERRUPTIBLE"] = "interruptible";
    DefaultElement["RETRY"] = "retry";
    DefaultElement["SERVICES"] = "services";
    DefaultElement["TAGS"] = "tags";
    DefaultElement["TIMEOUT"] = "timeout";
})(DefaultElement || (exports.DefaultElement = DefaultElement = {}));
/**
 * The content kind of what users can download via url.
 */
var LinkType;
(function (LinkType) {
    LinkType["IMAGE"] = "image";
    LinkType["OTHER"] = "other";
    LinkType["PACKAGE"] = "package";
    LinkType["RUNBOOK"] = "runbook";
})(LinkType || (exports.LinkType = LinkType = {}));
/**
 * You can mirror the pipeline status from the triggered pipeline to the source bridge job
 * by using strategy: depend
 * @see https://docs.gitlab.com/ee/ci/yaml/#triggerstrategy
 */
var Strategy;
(function (Strategy) {
    Strategy["DEPEND"] = "depend";
})(Strategy || (exports.Strategy = Strategy = {}));
/**
 * Describes the conditions for when to run the job. Defaults to 'on_success'.
 * The value can only be 'always' or 'never' when used with workflow.
 * @see https://docs.gitlab.com/ee/ci/yaml/#workflowrules
 */
var WorkflowWhen;
(function (WorkflowWhen) {
    WorkflowWhen["ALWAYS"] = "always";
    WorkflowWhen["NEVER"] = "never";
})(WorkflowWhen || (exports.WorkflowWhen = WorkflowWhen = {}));
//# sourceMappingURL=data:application/json;base64,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