import { Artifacts, Cache, Default, DefaultHooks, IDToken, Image, Include, Job, Retry, Service, VariableConfig, Workflow } from "./configuration-model";
import { Component } from "../component";
import { Project } from "../project";
import { YamlFile } from "../yaml";
/**
 * Options for `CiConfiguration`.
 */
export interface CiConfigurationOptions {
    /**
     * Default settings for the CI Configuration. Jobs that do not define one or more of the listed keywords use the value defined in the default section.
     */
    readonly default?: Default;
    /**
     * A special job used to upload static sites to Gitlab pages. Requires a `public/` directory
     * with `artifacts.path` pointing to it.
     */
    readonly pages?: Job;
    /**
     * Used to control pipeline behavior.
     */
    readonly workflow?: Workflow;
    /**
     * Groups jobs into stages. All jobs in one stage must complete before next stage is
     * executed. If no stages are specified. Defaults to ['build', 'test', 'deploy'].
     */
    readonly stages?: string[];
    /**
     * Global variables that are passed to jobs.
     * If the job already has that variable defined, the job-level variable takes precedence.
     */
    readonly variables?: Record<string, any>;
    /**
     * An initial set of jobs to add to the configuration.
     */
    readonly jobs?: Record<string, Job>;
    /**
     * The path of the file to generate.
     */
    readonly path?: string;
}
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
export declare class CiConfiguration extends Component {
    /**
     * The name of the configuration.
     */
    readonly name: string;
    /**
     * Path to CI file generated by the configuration.
     */
    readonly path: string;
    /**
     * The workflow YAML file.
     */
    readonly file: YamlFile;
    /**
     * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
     */
    readonly defaultAfterScript: string[];
    /**
     * Default list of files and directories that should be attached to the job if it succeeds. Artifacts are sent to Gitlab where they can be downloaded.
     */
    readonly defaultArtifacts?: Artifacts;
    /**
     * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
     */
    readonly defaultBeforeScript: string[];
    /**
     * A default list of cache definitions (máx. 4) with the files and directories to cache between jobs. You can only use paths that are in the local working copy.
     */
    private _defaultCache?;
    get defaultCache(): Cache[] | undefined;
    /**
     * Specifies the default docker image to use globally for all jobs.
     */
    readonly defaultImage?: Image;
    /**
     * The default behavior for whether a job should be canceled when a newer pipeline starts before the job completes (Default: false).
     */
    readonly defaultInterruptible?: boolean;
    /**
     * How many times a job is retried if it fails. If not defined, defaults to 0 and jobs do not retry.
     */
    readonly defaultRetry?: Retry;
    /**
     * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
     */
    private defaultServices;
    /**
     * Used to select a specific runner from the list of all runners that are available for the project.
     */
    readonly defaultTags: string[];
    /**
     * A default timeout job written in natural language (Ex. one hour, 3600 seconds, 60 minutes).
     */
    readonly defaultTimeout?: string;
    /**
     * Default ID tokens (JSON Web Tokens) that are used for CI/CD authentication to use globally for all jobs.
     */
    readonly defaultIdTokens?: Record<string, IDToken>;
    /**
     * Specify a list of commands to execute on the runner before cloning the Git repository and any submodules https://docs.gitlab.com/ci/yaml/#hookspre_get_sources_script
     */
    private defaultHooks?;
    /**
     * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
     * object with properties for the method if including external YAML file. The external
     * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
     */
    private include;
    /**
     * A special job used to upload static sites to Gitlab pages. Requires a `public/` directory
     * with `artifacts.path` pointing to it.
     */
    readonly pages?: Job;
    /**
     * Groups jobs into stages. All jobs in one stage must complete before next stage is
     * executed. Defaults to ['build', 'test', 'deploy'].
     */
    readonly stages: string[];
    /**
     * Global variables that are passed to jobs.
     * If the job already has that variable defined, the job-level variable takes precedence.
     */
    readonly variables: Record<string, number | VariableConfig | string>;
    /**
     * Used to control pipeline behavior.
     */
    readonly workflow?: Workflow;
    /**
     * The jobs in the CI configuration.
     */
    readonly jobs: Record<string, Job>;
    constructor(project: Project, name: string, options?: CiConfigurationOptions);
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes: Include[]): void;
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    private assertIsValidInclude;
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    private areEqualIncludes;
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services: Service[]): void;
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables: Record<string, any>): void;
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages: string[]): void;
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs: Record<string, Job>): void;
    private isValidCacheSetup;
    private assertIsValidCacheSetup;
    /**
     * Adds up to 4 default caches configuration to the CI configuration.
     * @param caches Caches to add.
     */
    addDefaultCaches(caches: Cache[]): void;
    /**
     * Specify a list of commands to execute on the runner before cloning the Git repository and any submodules
     * https://docs.gitlab.com/ci/yaml/#hookspre_get_sources_script
     * @param hooks
     */
    addDefaultHooks(hooks: DefaultHooks): void;
    private renderCI;
    private renderDefault;
}
