"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    get defaultCache() {
        return this._defaultCache;
    }
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.name = path.parse(name).name;
        const derivedPath = this.name === "gitlab-ci"
            ? ".gitlab-ci.yml"
            : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.path = options?.path ?? derivedPath;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            defaults.cache && this.addDefaultCaches(defaults.cache);
            this.defaultIdTokens = defaults.idTokens;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addGlobalVariables(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
            if (value.cache) {
                this.assertIsValidCacheSetup(value.cache);
            }
        }
    }
    isValidCacheSetup(caches) {
        const MAX_CONFIGURABLE_CACHES = 4;
        return caches.length <= MAX_CONFIGURABLE_CACHES;
    }
    assertIsValidCacheSetup(caches) {
        if (!this.isValidCacheSetup(caches)) {
            throw new Error(`${this.name}: GitLab CI can only define up to 4 caches, got: ${caches.length}`);
        }
    }
    /**
     * Adds up to 4 default caches configuration to the CI configuration.
     * @param caches Caches to add.
     */
    addDefaultCaches(caches) {
        this.assertIsValidCacheSetup(caches);
        this._defaultCache = caches;
    }
    /**
     * Specify a list of commands to execute on the runner before cloning the Git repository and any submodules
     * https://docs.gitlab.com/ci/yaml/#hookspre_get_sources_script
     * @param hooks
     */
    addDefaultHooks(hooks) {
        this.defaultHooks = hooks;
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            // we do not want to change job names
            // as they can be hidden (https://docs.gitlab.com/ee/ci/jobs/index.html#hide-jobs)
            // or referenced in extends
            ...snakeCaseKeys(this.jobs, true),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            idTokens: this.defaultIdTokens,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
            hooks: this.defaultHooks,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.98.9" };
function snakeCaseKeys(obj, skipTopLevel = false) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((o) => snakeCaseKeys(o));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" &&
            v != null &&
            k !== "variables" &&
            k !== "idTokens") {
            v = snakeCaseKeys(v);
        }
        result[skipTopLevel ? k : (0, case_1.snake)(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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