"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gitpod = exports.GitpodPortVisibility = exports.GitpodOnOpen = exports.GitpodOpenIn = exports.GitpodOpenMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("./component");
const yaml_1 = require("./yaml");
/**
 * Relative path of Gitpod file to generate
 */
const GITPOD_FILE = ".gitpod.yml";
/**
 * https://www.gitpod.io/docs/configuration/
 * https://www.gitpod.io/docs/config-start-tasks/
 * https://www.gitpod.io/docs/prebuilds/#configure-the-github-app
 * https://www.gitpod.io/docs/vscode-extensions/
 */
/**
 * Configure how the terminal should be opened relative to the previous task.
 */
var GitpodOpenMode;
(function (GitpodOpenMode) {
    /**
     * Opens in the same tab group right after the previous tab
     */
    GitpodOpenMode["TAB_AFTER"] = "tab-after";
    /**
     * Opens in the same tab group left before the previous tab
     */
    GitpodOpenMode["TAB_BEFORE"] = "tab-before";
    /**
     * Splits and adds the terminal to the right
     */
    GitpodOpenMode["SPLIT_RIGHT"] = "split-right";
    /**
     * Splits and adds the terminal to the left
     */
    GitpodOpenMode["SPLIT_LEFT"] = "split-left";
    /**
     * Splits and adds the terminal to the top
     */
    GitpodOpenMode["SPLIT_TOP"] = "split-top";
    /**
     * Splits and adds the terminal to the bottom
     */
    GitpodOpenMode["SPLIT_BOTTOM"] = "split-bottom";
})(GitpodOpenMode || (exports.GitpodOpenMode = GitpodOpenMode = {}));
/**
 * Configure where in the IDE the terminal should be opened.
 */
var GitpodOpenIn;
(function (GitpodOpenIn) {
    /**
     * the bottom panel (default)
     */
    GitpodOpenIn["BOTTOM"] = "bottom";
    /**
     * the left panel
     */
    GitpodOpenIn["LEFT"] = "left";
    /**
     * the right panel
     */
    GitpodOpenIn["RIGHT"] = "right";
    /**
     * the main editor area
     */
    GitpodOpenIn["MAIN"] = "main";
})(GitpodOpenIn || (exports.GitpodOpenIn = GitpodOpenIn = {}));
/**
 * What to do when a service on a port is detected.
 */
var GitpodOnOpen;
(function (GitpodOnOpen) {
    /**
     * Open a new browser tab
     */
    GitpodOnOpen["OPEN_BROWSER"] = "open-browser";
    /**
     * Open a preview on the right side of the IDE
     */
    GitpodOnOpen["OPEN_PREVIEW"] = "open-preview";
    /**
     * Show a notification asking the user what to do (default)
     */
    GitpodOnOpen["NOTIFY"] = "notify";
    /**
     * Do nothing.
     */
    GitpodOnOpen["IGNORE"] = "ignore";
})(GitpodOnOpen || (exports.GitpodOnOpen = GitpodOnOpen = {}));
/**
 * Whether the port visibility should be private or public
 */
var GitpodPortVisibility;
(function (GitpodPortVisibility) {
    /**
     * Allows everyone with the port URL to access the port (default)
     */
    GitpodPortVisibility["PUBLIC"] = "public";
    /**
     * Only allows users with workspace access to access the port
     */
    GitpodPortVisibility["PRIVATE"] = "private";
})(GitpodPortVisibility || (exports.GitpodPortVisibility = GitpodPortVisibility = {}));
/**
 * The Gitpod component which emits .gitpod.yml
 */
class Gitpod extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.tasks = new Array();
        this.ports = new Array();
        this.vscodeExtensions = new Array();
        this.dockerImage = options?.dockerImage;
        if (options?.tasks) {
            for (const task of options.tasks) {
                this.addTasks(task);
            }
        }
        if (options?.prebuilds) {
            this.addPrebuilds(options.prebuilds);
        }
        if (options?.ports) {
            this.addPorts(...options.ports);
        }
        if (options?.vscodeExtensions) {
            this.addVscodeExtensions(...options.vscodeExtensions);
        }
        this.config = {
            image: () => this.renderDockerImage(),
            tasks: this.tasks,
            github: () => this.renderPrebuilds(),
            ports: this.ports,
            vscode: {
                extensions: this.vscodeExtensions,
            },
        };
        new yaml_1.YamlFile(this.project, GITPOD_FILE, {
            obj: this.config,
            omitEmpty: true,
            // GitPod needs to read the file from the repository in order to work.
            committed: true,
        });
    }
    /**
     * Add a custom Docker image or Dockerfile for the container.
     *
     * @param image The Docker image
     */
    addDockerImage(image) {
        if (this.dockerImage) {
            throw new Error("dockerImage cannot be redefined.");
        }
        this.dockerImage = image;
    }
    /**
     * Add tasks to run when gitpod starts.
     *
     * By default, all tasks will be run in parallel. To run tasks in sequence,
     * create a new `Task` and specify the other tasks as subtasks.
     *
     * @param tasks The new tasks
     */
    addTasks(...tasks) {
        this.tasks.push(...tasks.map((task) => ({
            name: task.name,
            command: `npx projen ${task.name}`,
        })));
    }
    /**
     * Add a prebuilds configuration for the Gitpod App
     * @param config The configuration
     */
    addPrebuilds(config) {
        this.prebuilds = config;
    }
    /**
     * Add a task with more granular options.
     *
     * By default, all tasks will be run in parallel. To run tasks in sequence,
     * create a new `Task` and set the other tasks as subtasks.
     *
     * @param options The task parameters
     */
    addCustomTask(options) {
        this.tasks.push({
            name: options.name,
            command: options.command,
            openMode: options.openMode,
            openIn: options.openIn,
            before: options.before,
            init: options.init,
            prebuild: options.prebuild,
        });
    }
    /**
     * Add ports that should be exposed (forwarded) from the container.
     *
     * @param ports The new ports
     */
    addPorts(...ports) {
        this.ports.push(...ports.map((port) => ({ port: port })));
    }
    /**
     * Add a list of VSCode extensions that should be automatically installed
     * in the container.
     *
     * These must be in the format defined in the Open VSX registry.
     * @example 'scala-lang.scala@0.3.9:O5XmjwY5Gz+0oDZAmqneJw=='
     * @see https://www.gitpod.io/docs/vscode-extensions/
     *
     * @param extensions The extension IDs
     */
    addVscodeExtensions(...extensions) {
        this.vscodeExtensions.push(...extensions);
    }
    renderDockerImage() {
        if (this.dockerImage?.image) {
            return this.dockerImage.image;
        }
        else if (this.dockerImage?.dockerFile) {
            return {
                file: this.dockerImage.dockerFile,
            };
        }
        else {
            return undefined;
        }
    }
    renderPrebuilds() {
        if (this.prebuilds) {
            return {
                prebuilds: this.prebuilds,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Gitpod = Gitpod;
_a = JSII_RTTI_SYMBOL_1;
Gitpod[_a] = { fqn: "projen.Gitpod", version: "0.98.9" };
//# sourceMappingURL=data:application/json;base64,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