"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const runner_options_1 = require("../runner-options");
const version_1 = require("../version");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.18.0" } },
    dotnet: { dotnet: { version: "6.x" } },
};
const PUBLISH_JOB_PREFIX = "release_";
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this._gitHubPostPublishing = [];
        // List of publish jobs added to the publisher
        // Maps between the basename and the jobname
        this.publishJobs = {};
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "lts/*";
        this.workflowContainerImage = options.workflowContainerImage;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.publishTasks = options.publishTasks ?? false;
        this.runsOn = options.workflowRunsOn;
        this.runsOnGroup = options.workflowRunsOnGroup;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Adds post publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPostPublishingSteps(...steps) {
        this._gitHubPostPublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: version_1.CHANGES_SINCE_LAST_RELEASE,
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const jobName = "github";
        this.addPublishJob(jobName, (_branch, branchOptions) => {
            return {
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                postPublishSteps: options.postPublishSteps ?? this._gitHubPostPublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                needs: Object.entries(this.publishJobs)
                    .filter(([name, _]) => name != jobName)
                    .map(([_, job]) => job),
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.githubReleaseCommand(options, branchOptions),
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                },
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        if (options.trustedPublishing && options.npmTokenSecret) {
            throw new Error("Cannot use npmTokenSecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of NPM tokens. " +
                "Remove the npmTokenSecret option to use trusted publishing.");
        }
        const trustedPublisher = options.trustedPublishing ? "true" : undefined;
        const npmProvenance = options.npmProvenance ? "true" : undefined;
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const needsIdTokenWrite = isAwsCodeArtifactWithOidc || trustedPublisher || npmProvenance;
        const npmToken = trustedPublisher
            ? undefined
            : (0, node_package_1.defaultNpmToken)(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v5",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob("npm", (_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            let publishTools = PUBLIB_TOOLCHAIN.js;
            if (options.trustedPublishing && this.workflowNodeVersion == "lts/*") {
                // trusted publishing requires node 24.x and above
                // lts/* is currently 22.x
                // @todo remove once node 24.x is lts
                publishTools = {
                    node: {
                        version: "24.x",
                    },
                };
            }
            return {
                publishTools,
                prePublishSteps,
                postPublishSteps: options.postPublishSteps ?? [],
                environment: options.githubEnvironment ?? branchOptions.environment,
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                    NPM_CONFIG_PROVENANCE: npmProvenance,
                    NPM_TRUSTED_PUBLISHER: trustedPublisher,
                },
                permissions: {
                    idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        if (options.trustedPublishing && options.nugetApiKeySecret) {
            throw new Error("Cannot use nugetApiKeySecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of API keys. " +
                "Remove the nugetApiKeySecret option to use trusted publishing.");
        }
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        const needsIdTokenWrite = options.trustedPublishing;
        this.addPublishJob("nuget", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            env: {
                NUGET_TRUSTED_PUBLISHER: options.trustedPublishing
                    ? "true"
                    : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: options.trustedPublishing
                    ? undefined
                    : secret(isGitHubPackages
                        ? "GITHUB_TOKEN"
                        : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
                NUGET_USERNAME: options.trustedPublishing
                    ? secret(options.nugetUsernameSecret ?? "NUGET_USERNAME")
                    : undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        if (mavenServerId === "central-ossrh" && options.mavenEndpoint != null) {
            throw new Error('Custom endpoints are not supported when publishing to Maven Central (mavenServerId: "central-ossrh"). Please remove "mavenEndpoint" from the options.');
        }
        this.addPublishJob("maven", (_branch, branchOptions) => ({
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        if (options.trustedPublishing &&
            (options.twineUsernameSecret || options.twinePasswordSecret)) {
            throw new Error("Cannot use twineUsernameSecret and twinePasswordSecret when trustedPublishing is enabled. " +
                "Trusted publishing uses OIDC tokens for authentication instead of username/password credentials. " +
                "Remove the twineUsernameSecret and twinePasswordSecret options to use trusted publishing.");
        }
        let permissions = { contents: workflows_model_1.JobPermission.READ };
        const prePublishSteps = options.prePublishSteps ?? [];
        let workflowEnv = {};
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.twineRegistryUrl);
        if (isAwsCodeArtifact) {
            const { domain, account, region } = awsCodeArtifactInfoFromUrl(options.twineRegistryUrl);
            const { authProvider, roleToAssume, accessKeyIdSecret, secretAccessKeySecret, } = options.codeArtifactOptions ?? {};
            const useOidcAuth = authProvider === CodeArtifactAuthProvider.GITHUB_OIDC;
            if (useOidcAuth) {
                if (!roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
                permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
                prePublishSteps.push({
                    name: "Configure AWS Credentials via GitHub OIDC Provider",
                    uses: "aws-actions/configure-aws-credentials@v5",
                    with: {
                        "role-to-assume": roleToAssume,
                        "aws-region": region,
                    },
                });
            }
            prePublishSteps.push({
                name: "Generate CodeArtifact Token",
                run: `echo "TWINE_PASSWORD=$(aws codeartifact get-authorization-token --domain ${domain} --domain-owner ${account} --region ${region} --query authorizationToken --output text)" >> $GITHUB_ENV`,
                env: useOidcAuth
                    ? undefined
                    : {
                        AWS_ACCESS_KEY_ID: secret(accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID"),
                        AWS_SECRET_ACCESS_KEY: secret(secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY"),
                    },
            });
            workflowEnv = { TWINE_USERNAME: "aws" };
        }
        else if (options.trustedPublishing) {
            permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
            workflowEnv = {
                PYPI_TRUSTED_PUBLISHER: "true",
            };
            // attestations default to true, only disable when explicitly requested
            if (options.attestations === false) {
                workflowEnv.PYPI_DISABLE_ATTESTATIONS = "true";
            }
        }
        else {
            workflowEnv = {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            };
        }
        this.addPublishJob("pypi", (_branch, branchOptions) => ({
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            permissions,
            prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv,
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob("golang", (_branch, branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            environment: options.githubEnvironment ?? branchOptions.environment,
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(
    /**
     * The basename of the publish job (should be lowercase).
     * Will be extended with a prefix.
     */
    basename, factory) {
        const jobname = `${PUBLISH_JOB_PREFIX}${basename}`;
        this.publishJobs[basename] = jobname;
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [env, expression] of workflowEnvEntries) {
                requiredEnv.push(env);
                jobEnv[env] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${basename.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v5",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR, // this must be "dist" for publib
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
                ...opts.postPublishSteps,
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            const container = this.workflowContainerImage
                ? {
                    image: this.workflowContainerImage,
                }
                : undefined;
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        shell: "bash",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    ...(opts.environment ? { environment: opts.environment } : {}),
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId, ...(opts.needs ?? [])],
                    ...(0, runner_options_1.filteredRunsOnOptions)(this.runsOn, this.runsOnGroup),
                    container,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_SHA",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.98.9" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
/**
 * Parses info about code artifact domain from given AWS code artifact url
 * @param url Of code artifact domain
 * @returns domain, account, and region of code artifact domain
 */
function awsCodeArtifactInfoFromUrl(url) {
    const captureRegex = /([a-z0-9-]+)-(.+)\.d\.codeartifact\.(.+)\.amazonaws\.com/;
    const matches = url?.match(captureRegex) ?? [];
    const [_, domain, account, region] = matches;
    return { domain, account, region };
}
//# sourceMappingURL=data:application/json;base64,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