"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Task = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const logging_1 = require("./logging");
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
class Task {
    constructor(name, props = {}) {
        this.name = name;
        this._description = props.description;
        this._conditions = props.condition ? [props.condition] : [];
        this._cwd = props.cwd;
        this._locked = false;
        this._env = props.env ?? {};
        this._steps = props.steps ?? [];
        this.requiredEnv = props.requiredEnv;
        if (props.exec && props.steps) {
            throw new Error("cannot specify both exec and steps");
        }
        if (props.exec) {
            this.exec(props.exec, { receiveArgs: props.receiveArgs });
        }
    }
    /**
     * Forbid additional changes to this task.
     */
    lock() {
        this._locked = true;
    }
    /**
     * Returns the working directory for this task.
     */
    get cwd() {
        return this._cwd;
    }
    /**
     * Sets the working directory for this task.
     */
    set cwd(cwd) {
        this._cwd = cwd;
    }
    /**
     * Returns the description of this task.
     */
    get description() {
        return this._description;
    }
    /**
     * Sets the description of this task.
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * A command to execute which determines if the task should be skipped. If it
     * returns a zero exit code, the task will not be executed.
     */
    get condition() {
        if (this._conditions?.length) {
            return this._conditions.join(" && ");
        }
        return undefined;
    }
    /**
     * Add a command to execute which determines if the task should be skipped.
     *
     * If a condition already exists, the new condition will be appended with ` && ` delimiter.
     * @param condition The command to execute.
     * @see {@link Task.condition}
     */
    addCondition(...condition) {
        this._conditions.push(...condition);
    }
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("reset");
            return;
        }
        while (this._steps.length) {
            this._steps.shift();
        }
        if (command) {
            this.exec(command, options);
        }
    }
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("add exec to");
            return;
        }
        this._steps.push({ exec: command, ...options });
    }
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("add builtin to");
            return;
        }
        this._steps.push({ builtin: name });
    }
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("add say to");
            return;
        }
        this._steps.push({ say: message, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell, options = {}) {
        this.assertUnlocked();
        this.prependExec(shell, options);
    }
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("add spawn to");
            return;
        }
        this._steps.push({ spawn: subtask.name, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("prependExec to");
            return;
        }
        this._steps.unshift({
            exec: shell,
            ...options,
        });
    }
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("prependSpawn to");
            return;
        }
        this._steps.unshift({
            spawn: subtask.name,
            ...options,
        });
    }
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message, options = {}) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("prependSay to");
            return;
        }
        this._steps.unshift({
            say: message,
            ...options,
        });
    }
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name, value) {
        this.assertUnlocked();
        this._env[name] = value;
    }
    /**
     * Returns all environment variables in the task level
     */
    get envVars() {
        return this._env;
    }
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps() {
        // If the list of steps is a Lazy value, we can't know what the steps
        // are until synthesis occurs, so just return an empty array.
        if (!Array.isArray(this._steps)) {
            return [];
        }
        return [...this._steps];
    }
    /**
     * Insert one or more steps at a given index
     *
     * @param index Steps will be inserted before this index. May be negative to
     * count backwards from the end, or may be `== steps().length` to insert at the end.
     * @param steps The steps to insert
     */
    insertStep(index, ...steps) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("insert steps into");
            return;
        }
        if (index < -this._steps.length || index > this.steps.length) {
            throw new Error(`Cannot insert steps at index ${index} for task ${this.name} because the index is out of bounds for size ${this.steps.length}`);
        }
        this._steps.splice(index, 0, ...steps);
    }
    /**
     *
     * @param index The index of the step to edit
     * @param step The new step to replace the old one entirely, it is not merged with the old step
     */
    updateStep(index, step) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("update step for");
            return;
        }
        const existingStep = this._steps[index];
        if (!existingStep) {
            throw new Error(`Cannot update step at index ${index} for task ${this.name} because it does not exist`);
        }
        this._steps[index] = step;
    }
    /**
     *
     * @param index The index of the step to remove
     */
    removeStep(index) {
        this.assertUnlocked();
        if (!Array.isArray(this._steps)) {
            this.warnForLazyValue("remove step from");
            return;
        }
        const existingStep = this._steps[index];
        if (!existingStep) {
            throw new Error(`Cannot remove step at index ${index} for task ${this.name} because it does not exist`);
        }
        this._steps.splice(index, 1);
    }
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec() {
        // Ensure task-level env vars are strings
        const env = Object.keys(this._env).reduce((prev, curr) => ({
            ...prev,
            [curr]: this.getEnvString(curr, this._env[curr]),
        }), {});
        // Ensure step-level env vars are strings
        const steps = Array.isArray(this._steps)
            ? [...this._steps].map((s) => {
                return s.env
                    ? {
                        ...s,
                        env: Object.keys(s.env).reduce((prev, curr) => ({
                            ...prev,
                            [curr]: this.getEnvString(curr, s.env[curr]),
                        }), {}),
                    }
                    : s;
            })
            : this._steps;
        return {
            name: this.name,
            description: this.description,
            env: env,
            requiredEnv: this.requiredEnv,
            steps: steps,
            condition: this.condition,
            cwd: this._cwd,
        };
    }
    assertUnlocked() {
        if (this._locked) {
            throw new Error(`Task "${this.name}" is locked for changes`);
        }
    }
    warnForLazyValue(actionBeingUndertaken) {
        (0, logging_1.warn)(`Cannot ${actionBeingUndertaken} task "${this.name}" because it is a lazy value, try using the preSynthesize phase.`);
    }
    /**
     * Ensure that environment variables are persisted as strings
     * to prevent type errors when parsing from tasks.json in future
     */
    getEnvString(name, value) {
        if (typeof value !== "string" && value !== undefined) {
            (0, logging_1.warn)(`Received non-string value for environment variable ${name}. Value will be stringified.`);
            return String(value);
        }
        else {
            return value;
        }
    }
}
exports.Task = Task;
_a = JSII_RTTI_SYMBOL_1;
Task[_a] = { fqn: "projen.Task", version: "0.98.9" };
//# sourceMappingURL=data:application/json;base64,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