"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.COMPONENT_SYMBOL = exports.PROJECT_SYMBOL = void 0;
exports.tryFindClosest = tryFindClosest;
exports.findClosestProject = findClosestProject;
exports.closestProjectMustBe = closestProjectMustBe;
exports.isProject = isProject;
exports.isComponent = isComponent;
exports.tagAsProject = tagAsProject;
exports.tagAsComponent = tagAsComponent;
exports.PROJECT_SYMBOL = Symbol.for("projen.Project");
exports.COMPONENT_SYMBOL = Symbol.for("projen.Component");
/**
 * Create a function to find the closest construct matching a predicate
 * @param predicate
 * @returns A function to find the closest construct matching the predicate
 */
function tryFindClosest(predicate) {
    return (construct) => construct?.node.scopes.reverse().find(predicate);
}
/**
 * Find the closest project, searching upwards from a construct.
 *
 * @param construct the construct to start searching from
 * @param constructInCreation the name of the construct being created
 * @returns the closest project, if any
 */
function findClosestProject(construct, constructInCreation) {
    if (isComponent(construct)) {
        return construct.project;
    }
    const project = tryFindClosest(isProject)(construct);
    if (!project) {
        throw new Error(`${constructInCreation} at '${construct.node.path}' must be created in the scope of a Project, but no Project was found`);
    }
    return project;
}
/**
 * Find the closest project of a given type, searching upwards from a construct.
 *
 * This function should be used within a class constructor.
 * If not, you must provide a name as second argument or the call will fail.
 *
 * @param construct the construct to start searching from
 * @param constructInCreation the name of the construct being created
 * @returns the closest project of the expected type, if any
 */
function closestProjectMustBe(construct, projectType, constructInCreation) {
    const project = tryFindClosest(isProject)(construct);
    const error = `${constructInCreation} at '${construct.node.path}' must be created within a ${projectType.name}`;
    if (!project) {
        throw new Error(`${error}, but no Project was found`);
    }
    if (!(project instanceof projectType)) {
        throw new Error(`${error}, but found: ${project.constructor.name}`);
    }
    return project;
}
function isProject(x) {
    return x !== null && typeof x === "object" && exports.PROJECT_SYMBOL in x;
}
function isComponent(x) {
    return x !== null && typeof x === "object" && exports.COMPONENT_SYMBOL in x;
}
function tagAs(scope, tag) {
    Object.defineProperty(scope, tag, { value: true });
}
function tagAsProject(scope) {
    tagAs(scope, exports.PROJECT_SYMBOL);
}
function tagAsComponent(scope) {
    tagAs(scope, exports.COMPONENT_SYMBOL);
}
//# sourceMappingURL=data:application/json;base64,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