# Tags Management in Must CDK

Must CDK provides a comprehensive tagging system that automatically applies tags to all AWS resources created by the constructs, including the CloudFormation stack itself. This system is designed to support both global environment-based tags and construct-specific tags, with a clear precedence model.

## Table of Contents

- [Overview](#overview)
- [Environment Tags](#environment-tags)
- [Construct-Specific Tags](#construct-specific-tags)
- [Stack-Level Tagging](#stack-level-tagging)
- [Tag Precedence](#tag-precedence)
- [Implementation Details](#implementation-details)
- [Examples](#examples)
- [Troubleshooting](#troubleshooting)

## Overview

The Must CDK tagging system provides:

- **Global Environment Tags**: Set via `TAGS` environment variable
- **Construct-Specific Tags**: Set via the `tags` property on individual constructs
- **Stack-Level Tagging**: Automatic application of tags to the CloudFormation stack
- **Automatic Tag Inheritance**: All child resources inherit tags from their parent constructs
- **Precedence Model**: Environment tags override construct-specific tags
- **Flexible Format**: Support for complex values including URLs and configuration strings

## Environment Tags

Environment tags are set using the `TAGS` environment variable and are applied globally to all Must CDK constructs and the CloudFormation stack.

### Format

```bash
TAGS="key1=value1,key2=value2,key3=value3"
```

### Setting Environment Tags

```bash
# Basic example
export TAGS="Product=MyApp,Owner=TeamName,Environment=production"

# Complex values with special characters
export TAGS="Product=MyApp,URL=https://api.example.com/v1,Config=key1=val1&key2=val2"

# Multi-line for readability (bash)
export TAGS="Product=MyApp,\
Owner=TeamName,\
Environment=production,\
CostCenter=Engineering,\
Project=WebApplication"
```

### CI/CD Integration

```yaml
# GitHub Actions
env:
  TAGS: "Product=MyApp,Environment=production,Owner=TeamName,BuildNumber=${{ github.run_number }}"

# AWS CodeBuild
environment:
  variables:
    TAGS: "Product=MyApp,Environment=production,Owner=TeamName"
```

## Construct-Specific Tags

Each Must CDK construct accepts a `tags` property for construct-specific tagging.

### TypeScript

```typescript
import { AmplifyApp } from 'must-cdk';

new AmplifyApp(this, 'MyApp', {
  appName: 'my-application',
  repository: 'https://github.com/user/repo',
  tags: {
    Team: 'frontend',
    Version: 'v1.0.0',
    Component: 'web-app',
    Maintainer: 'john.doe@company.com'
  }
});
```

### Python

```python
from must_cdk import AmplifyApp

AmplifyApp(self, 'MyApp',
  app_name='my-application',
  repository='https://github.com/user/repo',
  tags={
    'Team': 'frontend',
    'Version': 'v1.0.0',
    'Component': 'web-app',
    'Maintainer': 'john.doe@company.com'
  }
)
```

## Stack-Level Tagging

**New Feature**: Must CDK automatically applies tags to the CloudFormation stack itself, providing stack-level visibility and management capabilities.

### Benefits

- **Cost Tracking**: Track costs at the stack level in AWS Cost Explorer
- **Stack Organization**: Filter and organize stacks in the AWS Console
- **Governance**: Apply policies to entire stacks based on tags
- **Ownership**: Identify stack ownership and purpose
- **Automation**: Enable stack-level automation based on tags

### What Gets Tagged

When you deploy with Must CDK, tags are applied to:

1. **Individual AWS Resources**: EC2 instances, S3 buckets, Lambda functions, etc.
2. **CloudFormation Stack**: The stack itself in the AWS Console
3. **Cost Reports**: Stack-level cost allocation in billing reports

### Viewing Stack Tags

```bash
# View stack tags via AWS CLI
aws cloudformation describe-stacks --stack-name MyStack --query 'Stacks[0].Tags'

# View in AWS Console
# Navigate to CloudFormation → Stacks → [Your Stack] → Tags tab
```

### Stack Tag Inheritance

All Must CDK constructs automatically apply their final merged tags (environment + construct-specific) to both:
- The construct and its child resources
- The parent CloudFormation stack

This ensures consistent tagging across all levels of your infrastructure.

## Tag Precedence

Environment tags take precedence over construct-specific tags. This allows for global overrides while maintaining construct-level defaults.

### Example

```bash
# Environment variable
export TAGS="Environment=production,Team=platform,Owner=OpsTeam"
```

```typescript
// Construct configuration
new AmplifyApp(this, 'MyApp', {
  appName: 'my-app',
  repository: 'https://github.com/user/repo',
  tags: {
    Team: 'frontend',        // Will be overridden by environment
    Component: 'web-app',    // Will be preserved
    Version: 'v1.0.0'       // Will be preserved
  }
});
```

**Final tags applied:**
- `Environment=production` (from environment)
- `Team=platform` (from environment, overrides construct tag)
- `Owner=OpsTeam` (from environment)
- `Component=web-app` (from construct)
- `Version=v1.0.0` (from construct)

## Implementation Details

### Tag Processing Flow

1. **Environment Tag Parsing**: The `TAGS` environment variable is parsed into key-value pairs
2. **Tag Merging**: Construct-specific tags are merged with environment tags
3. **Precedence Application**: Environment tags override construct tags with the same key
4. **Resource Tagging**: Final tags are applied to all created resources
5. **Stack Tagging**: Final tags are applied to the CloudFormation stack
6. **Inheritance**: Child resources inherit tags through CDK's `Tags.of()` mechanism

### Stack Tagging Implementation

Must CDK uses the `applyTags()` function that:

```typescript
// Applies tags to both construct and stack
export function applyTags(
  construct: Construct,
  userTags?: Record<string, string>,
): Record<string, string> {
  const finalTags = mergeTags(userTags);

  // Apply tags to the construct (inherits to child resources)
  Object.entries(finalTags).forEach(([key, value]) => {
    Tags.of(construct).add(key, value);
  });

  // Apply tags to the CloudFormation stack
  const stack = Stack.of(construct);
  Object.entries(finalTags).forEach(([key, value]) => {
    Tags.of(stack).add(key, value);
  });

  return finalTags;
}
```

This ensures that every Must CDK construct automatically tags both its resources and the parent CloudFormation stack.

### Tag Format Conversion

Must CDK automatically converts tags between different AWS resource formats:

```typescript
// Internal format
{ Product: 'MyApp', Owner: 'TeamName' }

// CloudFormation format (for resources like Amplify)
[
  { Key: 'Product', Value: 'MyApp' },
  { Key: 'Owner', Value: 'TeamName' }
]

// Direct property format (for resources like Lambda)
{ Product: 'MyApp', Owner: 'TeamName' }
```

### Error Handling

The tagging system includes robust error handling:

- **Malformed Environment Variables**: Invalid key-value pairs are skipped with warnings
- **Empty Values**: Empty tag values are preserved if explicitly set
- **Special Characters**: URLs, JSON, and other complex values are supported
- **Missing Environment Variable**: System gracefully handles missing `TAGS` variable

## Examples

### Development Environment

```bash
export TAGS="Environment=development,Owner=DevTeam,CostCenter=R&D"

# Deploy with development-specific tags
cdk deploy MyStack
```

### Staging Environment

```bash
export TAGS="Environment=staging,Owner=QATeam,CostCenter=Testing,AutoShutdown=true"

# Deploy with staging-specific tags
cdk deploy MyStack
```

### Production Environment

```bash
export TAGS="Environment=production,Owner=OpsTeam,CostCenter=Production,Backup=required,Monitoring=enhanced"

# Deploy with production-specific tags
cdk deploy MyStack
```

### Stack-Level Cost Tracking

```bash
# Set cost tracking tags
export TAGS="Product=ECommerce,Environment=production,CostCenter=Engineering,BudgetOwner=john.doe@company.com"

# Deploy stack
cdk deploy ECommerceStack

# View stack costs in AWS Cost Explorer
# Filter by: Tag Key = "Product", Tag Value = "ECommerce"
# Group by: Tag Key = "Environment"
```

### Stack Organization and Governance

```bash
# Development environment
export TAGS="Environment=development,Owner=DevTeam,AutoShutdown=true,Backup=none"
cdk deploy MyAppDev

# Production environment  
export TAGS="Environment=production,Owner=OpsTeam,AutoShutdown=false,Backup=daily"
cdk deploy MyAppProd

# In AWS Console, filter stacks by:
# - Environment tag to see all dev vs prod stacks
# - Owner tag to see team ownership
# - AutoShutdown tag for automation policies
```

### Multi-Service Application

```typescript
// Shared environment tags
// TAGS="Product=ECommerce,Environment=production,Owner=Platform"

// Frontend service
new AmplifyApp(this, 'Frontend', {
  appName: 'ecommerce-frontend',
  repository: 'https://github.com/company/frontend',
  tags: {
    Component: 'frontend',
    Framework: 'nextjs',
    Team: 'frontend-team'
  }
});

// API service
new ApiGatewayToLambda(this, 'API', {
  lambdaFunction: apiFunction,
  apiName: 'ecommerce-api',
  tags: {
    Component: 'api',
    Runtime: 'nodejs18',
    Team: 'backend-team'
  }
});

// CDN
new CloudFrontToOrigins(this, 'CDN', {
  s3Origins: [{ id: 'assets', bucket: assetsBucket }],
  tags: {
    Component: 'cdn',
    CacheStrategy: 'aggressive',
    Team: 'platform-team'
  }
});

// Result: All resources AND the CloudFormation stack will have:
// - Product=ECommerce (from environment)
// - Environment=production (from environment)  
// - Owner=Platform (from environment)
// - Component=frontend/api/cdn (from construct)
// - Team=frontend-team/backend-team/platform-team (from construct)
```

## Troubleshooting

### Common Issues

#### 1. Tags Not Appearing on Resources

**Problem**: Tags are not visible on AWS resources in the console.

**Solutions**:
- Verify the `TAGS` environment variable is set correctly
- Check that the environment variable is available during CDK deployment
- Ensure tag keys and values don't contain invalid characters
- Verify the construct supports tagging (all Must CDK constructs do)

#### 2. Environment Tags Not Overriding Construct Tags

**Problem**: Construct-specific tags are not being overridden by environment tags.

**Solutions**:
- Verify the `TAGS` environment variable format: `key1=value1,key2=value2`
- Check for typos in tag keys between environment and construct tags
- Ensure the environment variable is set before running CDK commands

#### 3. Special Characters in Tag Values

**Problem**: Tag values with special characters are not parsed correctly.

**Solutions**:
- Avoid using commas in tag values (they're used as separators)
- Use URL encoding for complex values if necessary
- Test tag parsing with simple values first

#### 4. Tags Missing from Some Resources

**Problem**: Some resources within a construct don't have tags applied.

**Solutions**:
- This is expected behavior for some AWS resources that don't support tagging
- Must CDK applies tags to all taggable resources automatically
- Check AWS documentation for resource-specific tagging support

### Debugging Tag Application

#### 1. Enable Debug Logging

```bash
# Enable CDK debug logging
export CDK_DEBUG=true
cdk deploy
```

#### 2. Verify Environment Variable Parsing

```typescript
// Add temporary logging to verify tag parsing
import { getTagsFromEnvironment } from 'must-cdk/lib/common/tags';

console.log('Environment tags:', getTagsFromEnvironment());
```

#### 3. Check CloudFormation Template

```bash
# Generate CloudFormation template to verify tags
cdk synth > template.yaml

# Search for tags in the template
grep -A 5 -B 5 "Tags" template.yaml
```

#### 4. AWS CLI Verification

```bash
# Check tags on deployed resources
aws amplify get-app --app-id <app-id> --query 'app.tags'
aws cloudfront get-distribution --id <distribution-id> --query 'Distribution.DistributionConfig.Tags'
aws apigateway get-tags --resource-arn <api-arn>
```

### Performance Considerations

#### Tag Limits

Be aware of AWS tagging limits:
- Maximum 50 tags per resource (varies by service)
- Tag key length: 1-128 characters
- Tag value length: 0-256 characters
- Case-sensitive keys and values

#### Best Practices for Performance

1. **Minimize Tag Count**: Use only necessary tags to stay within limits
2. **Consistent Formatting**: Use consistent tag formats to avoid confusion
3. **Avoid Dynamic Tags**: Don't use frequently changing values (like timestamps) in tags
4. **Cache Environment Tags**: Environment tags are parsed once per construct

## Migration Guide

### From Manual Tagging

If you're migrating from manual tagging to Must CDK's unified system:

1. **Audit Existing Tags**: Document current tagging patterns
2. **Standardize Tag Keys**: Ensure consistent naming across resources
3. **Set Environment Variables**: Move common tags to `TAGS` environment variable
4. **Update Construct Props**: Remove redundant tags from construct properties
5. **Test Deployment**: Verify tags are applied correctly after migration

### Example Migration

**Before (Manual Tagging)**:
```typescript
const app = new amplify.CfnApp(this, 'App', {
  name: 'my-app',
  tags: [
    { key: 'Product', value: 'MyApp' },
    { key: 'Environment', value: 'production' },
    { key: 'Owner', value: 'TeamName' }
  ]
});
```

**After (Must CDK Unified Tagging)**:
```bash
# Set environment tags
export TAGS="Product=MyApp,Environment=production,Owner=TeamName"
```

```typescript
// Simplified construct usage
new AmplifyApp(this, 'MyApp', {
  appName: 'my-app',
  repository: 'https://github.com/user/repo'
  // Tags applied automatically from environment
});
```

## API Reference

### Functions

#### `getTagsFromEnvironment(): Record<string, string>`

Parses the `TAGS` environment variable and returns tags as an object.

#### `mergeTags(userTags?: Record<string, string>): Record<string, string>`

Merges user-provided tags with environment tags, with environment tags taking precedence.

#### `formatTagsForCfn(tags: Record<string, string>): Array<{ key: string; value: string }>`

Converts tags object to AWS CloudFormation tag format.

### Interfaces

#### `TaggableProps`

```typescript
interface TaggableProps {
  readonly tags?: Record<string, string>;
}
```

Base interface for all Must CDK constructs that support tagging.

## Contributing

To contribute to the tagging system:

1. **Follow Existing Patterns**: Use the established tagging utilities
2. **Test Tag Application**: Ensure tags are applied to all relevant resources
3. **Update Documentation**: Document any new tagging behavior
4. **Add Tests**: Include tests for tag application in your constructs

For more information, see the [main project documentation](../README.md).
