# ECS CodeDeploy Examples

This directory contains examples for deploying containerized applications using Amazon ECS with AWS CodeDeploy for Blue/Green deployments.

## Example

### ECS with CodeDeploy
- **Files**: `typescript/ecs-codedeploy.ts` / `python/ecs_codedeploy.py`
- **Description**: Production-ready ECS service with comprehensive container configuration
- **Features**:
  - Environment variables support
  - Secrets management (AWS Secrets Manager & SSM Parameter Store)
  - Custom container names
  - Container access after creation
  - Comprehensive container configuration options
  - Blue/Green deployment with CodeDeploy
  - Application Load Balancer integration
  - Multi-container support (sidecar pattern)
  - SSL/TLS termination with ACM certificates
  - Advanced auto-scaling policies
  - Comprehensive tagging strategy
  - Health checks and monitoring
  - VPC and security group configuration
  - Container insights enabled

## Key Components

### 1. VPC Configuration
- Multi-AZ setup with public and private subnets
- NAT gateways for outbound internet access
- Security groups with proper ingress/egress rules

### 2. ECS Cluster
- Container insights enabled for monitoring
- Optimized for Fargate tasks

### 3. Blue/Green Deployment
- Zero-downtime deployments
- Automatic rollback on failure
- Configurable deployment timeouts
- Traffic shifting strategies

### 4. Auto Scaling
- CPU and memory-based scaling policies
- Configurable min/max capacity
- Scale-in and scale-out cooldowns

## Features

### Environment Variables Support
```typescript
containers: [{
  name: 'web-api',
  image: ecs.ContainerImage.fromRegistry('my-app:latest'),
  containerPort: 3000,
  environment: {
    NODE_ENV: 'production',
    LOG_LEVEL: 'info',
    DATABASE_URL: 'postgresql://...',
    CACHE_TTL: '3600'
  }
}]
```

### Secrets Management
```typescript
// Create secrets and parameters
const dbSecret = new secretsmanager.Secret(this, 'DatabaseSecret');
const apiKeyParam = new ssm.StringParameter(this, 'ApiKey', {
  parameterName: '/myapp/api-key',
  stringValue: 'your-api-key'
});

// Use in containers
containers: [{
  name: 'web-api',
  // ... other config
  secrets: {
    DB_PASSWORD: ecs.Secret.fromSecretsManager(dbSecret, 'password'),
    API_KEY: ecs.Secret.fromSsmParameter(apiKeyParam)
  }
}]
```

### Custom Container Names
```typescript
containers: [
  {
    name: 'web-server',
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    containerPort: 80
  },
  {
    name: 'log-router',
    image: ecs.ContainerImage.fromRegistry('fluent/fluent-bit:latest'),
    containerPort: 24224
  }
]
```

### Container Access After Creation
```typescript
const ecsService = new EcsCodeDeploy(this, 'MyService', {
  // ... configuration
});

// Access containers after creation
const webContainer = ecsService.containers[0];
const logContainer = ecsService.containers[1];

// Use container references for additional configuration
console.log(`Web container: ${webContainer.containerName}`);
```

### Comprehensive Container Configuration
```typescript
containers: [{
  name: 'web-api',
  image: ecs.ContainerImage.fromRegistry('my-app:latest'),
  containerPort: 3000,
  
  // Resource allocation
  cpu: 512,
  memoryReservation: 1024,
  
  // Container behavior
  essential: true,
  workingDirectory: '/app',
  user: 'appuser',
  entryPoint: ['/docker-entrypoint.sh'],
  command: ['node', 'server.js'],
  
  // Environment and secrets
  environment: {
    NODE_ENV: 'production',
    LOG_LEVEL: 'info'
  },
  secrets: {
    DB_PASSWORD: ecs.Secret.fromSecretsManager(dbSecret)
  },
  
  // Health check
  healthCheck: {
    command: ['CMD-SHELL', 'curl -f http://localhost:3000/health || exit 1'],
    interval: Duration.seconds(30),
    timeout: Duration.seconds(5),
    retries: 3,
    startPeriod: Duration.seconds(60)
  }
}]
```

## Deployment

```bash
# TypeScript
npm install
cdk deploy EnhancedEcsCodeDeployStack

# Python
pip install aws-cdk-lib must-cdk
cdk deploy EnhancedEcsCodeDeployStack
```

## Monitoring

The examples include:
- CloudWatch Container Insights
- Application Load Balancer health checks
- ECS service metrics
- Auto-scaling metrics
- Container-level environment variable monitoring

## Security Best Practices

- Tasks run in private subnets
- Security groups with minimal required access
- IAM roles with least privilege
- Secrets stored in AWS Secrets Manager/SSM Parameter Store
- Environment variables for non-sensitive configuration
- Encryption in transit and at rest
- VPC Flow Logs (optional)

## Troubleshooting

### Environment Variables Not Applied
- Ensure you're using the `environment` property in container configuration
- Check that the container definition includes the environment variables in CloudFormation
- Verify IAM permissions for accessing secrets

### Container Access Issues
- Use the `containers` property to access container definitions after creation
- Container names are available via `containerName` property
- Index-based access: `ecsService.containers[0]` for first container

### Secrets Not Loading
- Verify IAM permissions for the task execution role
- Check that secrets exist in the specified AWS region
- Ensure parameter names are correct for SSM parameters
