#!/usr/bin/env python3
import os
import aws_cdk as cdk
import aws_cdk.aws_lambda as lambda_
import aws_cdk.aws_s3 as s3
from constructs import Construct
from must_cdk import AmplifyApp, ApiGatewayToLambda, CloudFrontToOrigins


class TaggingExampleStack(cdk.Stack):
    """
    Example demonstrating Must CDK's unified tagging system
    
    Environment tags are set via TAGS environment variable:
    export TAGS="Product=MyApp,Environment=production,Owner=TeamName,CostCenter=Engineering"
    
    These tags will be applied to all resources, with environment tags taking precedence
    over construct-specific tags.
    """

    def __init__(self, scope: Construct, construct_id: str, **kwargs) -> None:
        super().__init__(scope, construct_id, **kwargs)

        # Example 1: Amplify App with construct-specific tags
        # Environment tags will be merged with these tags
        AmplifyApp(self, "MyAmplifyApp",
            app_name="my-tagged-app",
            repository="https://github.com/user/my-app",
            access_token="your-github-token",  # In production, use AWS Secrets Manager
            tags={
                "Component": "frontend",
                "Framework": "nextjs",
                "Team": "frontend-team",
                "Version": "v1.0.0",
                # Note: If TAGS contains "Team=platform-team", it will override "frontend-team"
            }
        )

        # Example 2: API Gateway with Lambda
        api_function = lambda_.Function(self, "ApiFunction",
            runtime=lambda_.Runtime.NODEJS_18_X,
            handler="index.handler",
            code=lambda_.Code.from_inline("""
                exports.handler = async (event) => {
                    return {
                        statusCode: 200,
                        body: JSON.stringify({ message: 'Hello from tagged API!' })
                    };
                };
            """),
        )

        ApiGatewayToLambda(self, "MyAPI",
            lambda_function=api_function,
            api_name="my-tagged-api",
            tags={
                "Component": "api",
                "Runtime": "nodejs18",
                "Team": "backend-team",
                "APIType": "rest",
                # Environment tags will be merged with these
            }
        )

        # Example 3: CloudFront Distribution
        assets_bucket = s3.Bucket(self, "AssetsBucket",
            bucket_name="my-tagged-assets-bucket",
            public_read_access=False,
        )

        CloudFrontToOrigins(self, "MyCDN",
            s3_origins=[
                {
                    "id": "assets-origin",
                    "bucket": assets_bucket,
                    "origin_path": "/static",
                }
            ],
            tags={
                "Component": "cdn",
                "CacheStrategy": "aggressive",
                "Team": "platform-team",
                "ContentType": "static-assets",
                # These will be merged with environment tags
            }
        )

        # Output information about tagging
        cdk.CfnOutput(self, "TaggingInfo",
            value="Check AWS Console to see tags applied to all resources",
            description="Tags are applied automatically from TAGS environment variable and construct props"
        )


# Example usage with different environments
app = cdk.App()

# Development stack
TaggingExampleStack(app, "TaggingExampleDev",
    env=cdk.Environment(
        account=os.getenv('CDK_DEFAULT_ACCOUNT'),
        region=os.getenv('CDK_DEFAULT_REGION'),
    ),
    # Additional stack-level tags can be added here
    tags={
        "StackType": "example",
        "Purpose": "demonstration"
    }
)

# Production stack (would typically be in separate deployment)
# TaggingExampleStack(app, "TaggingExampleProd",
#     env=cdk.Environment(
#         account="your-prod-account",
#         region="us-east-1",
#     ),
#     tags={
#         "StackType": "example",
#         "Purpose": "demonstration",
#         "Environment": "production"  # This would be overridden by TAGS env var if set
#     }
# )

app.synth()

"""
Usage Examples:

1. Development Environment:
   export TAGS="Product=MyApp,Environment=development,Owner=DevTeam,AutoShutdown=true"
   cdk deploy TaggingExampleDev

2. Staging Environment:
   export TAGS="Product=MyApp,Environment=staging,Owner=QATeam,AutoShutdown=true"
   cdk deploy TaggingExampleDev

3. Production Environment:
   export TAGS="Product=MyApp,Environment=production,Owner=OpsTeam,Backup=required,Monitoring=enhanced"
   cdk deploy TaggingExampleProd

4. Complex Tags with Special Characters:
   export TAGS="Product=MyApp,URL=https://api.example.com/v1,Config=key1=val1&key2=val2"
   cdk deploy TaggingExampleDev

Final tags on resources will be a merge of:
- Environment tags (from TAGS variable) - highest precedence
- Construct-specific tags (from construct props)
- Stack-level tags (from stack props) - lowest precedence

Example final tags for AmplifyApp with TAGS="Product=MyApp,Environment=production,Team=platform":
- Product=MyApp (from environment)
- Environment=production (from environment)  
- Team=platform (from environment, overrides construct's "frontend-team")
- Component=frontend (from construct)
- Framework=nextjs (from construct)
- Version=v1.0.0 (from construct)
- StackType=example (from stack)
- Purpose=demonstration (from stack)
"""
