import { TaggableProps } from "../common/tags";
/**
 * Properties for creating an AWS Amplify application.
 */
export interface AmplifyAppProps extends TaggableProps {
    /**
     * The name for the Amplify app.
     */
    readonly appName: string;
    /**
     * The Git repository URL for the Amplify app.
     * Format: https://github.com/user/repo or similar.
     */
    readonly repository: string;
    /**
     * GitHub personal access token for repository access.
     * If not provided, will attempt to use GitHub CLI token or MUFIN_PUSH_TOKEN environment variable.
     *
     * Note: For production deployments, consider migrating to GitHub Apps for better security,
     * organization support, and higher rate limits after initial setup.
     */
    readonly accessToken?: string;
    /**
     * Environment variables for the Amplify app.
     * These will be available during the build process.
     */
    readonly environmentVariables?: Record<string, string>;
    /**
     * Build specification for the Amplify app.
     * Defines the build commands and output artifacts.
     */
    readonly buildSpec?: string;
    /**
     * Custom domain configuration for the Amplify app.
     */
    readonly customDomain?: CustomDomainOptions;
    /**
     * Branch configurations for the Amplify app.
     * If not provided, a default 'main' branch will be created.
     */
    readonly branches?: BranchOptions[];
    /**
     * Platform for the Amplify app.
     * @default "WEB"
     */
    readonly platform?: "WEB" | "WEB_COMPUTE";
    /**
     * Build settings for the Amplify app.
     */
    readonly buildSettings?: BuildSettings;
    /**
     * Custom rules for the Amplify app.
     * Used for redirects, rewrites, and other routing rules.
     */
    readonly customRules?: CustomRule[];
    /**
     * Basic authentication configuration for the Amplify app.
     */
    readonly basicAuth?: BasicAuthConfig;
}
/**
 * Build settings for an Amplify application.
 */
export interface BuildSettings {
    /**
     * Compute type for the build.
     * @default "STANDARD"
     */
    readonly computeType?: "STANDARD" | "STANDARD_8GB";
    /**
     * Cache type for the build.
     * @default "AMPLIFY_MANAGED"
     */
    readonly cacheType?: "NO_CACHE" | "AMPLIFY_MANAGED" | "AMPLIFY_MANAGED_NO_COOKIES" | "S3";
    /**
     * Whether to enable automatic branch creation when a new branch is pushed to the repository.
     * @default false
     */
    readonly enableAutoBranchCreation?: boolean;
    /**
     * Whether to enable automatic branch deletion when a branch is deleted from the repository.
     * @default false
     */
    readonly enableAutoBranchDeletion?: boolean;
    /**
     * Whether to enable automatic builds when code is pushed to a branch.
     * @default true
     */
    readonly enableBranchAutoBuild?: boolean;
}
/**
 * Custom routing rule for an Amplify application.
 */
export interface CustomRule {
    /**
     * Source pattern to match in the URL.
     */
    readonly source: string;
    /**
     * Target URL to redirect or rewrite to.
     */
    readonly target: string;
    /**
     * HTTP status code for the redirect.
     * @default "200"
     */
    readonly status?: string;
    /**
     * Condition to apply the rule.
     */
    readonly condition?: string;
}
/**
 * Custom domain configuration for an Amplify application.
 */
export interface CustomDomainOptions {
    /**
     * The custom domain name for the Amplify app.
     */
    readonly domainName: string;
    /**
     * Subdomain configurations for the custom domain.
     */
    readonly subDomains?: SubDomainOptions[];
    /**
     * Whether to enable automatic subdomain creation.
     * @default false
     */
    readonly enableAutoSubdomain?: boolean;
    /**
     * Patterns for automatic subdomain creation.
     */
    readonly autoSubdomainCreationPatterns?: string[];
}
/**
 * Subdomain configuration for a custom domain in an Amplify application.
 */
export interface SubDomainOptions {
    /**
     * The branch name to map to this subdomain.
     */
    readonly branchName: string;
    /**
     * The prefix for the subdomain.
     * Use empty string for the root domain.
     */
    readonly prefix: string;
}
/**
 * Basic authentication configuration for an Amplify application.
 */
export interface BasicAuthConfig {
    /**
     * Username for basic authentication.
     */
    readonly username: string;
    /**
     * Password for basic authentication.
     */
    readonly password: string;
    /**
     * Whether to enable basic authentication.
     * @default true
     */
    readonly enableBasicAuth?: boolean;
}
/**
 * Branch configuration for an Amplify application.
 */
export interface BranchOptions {
    /**
     * The name of the branch.
     */
    readonly branchName: string;
    /**
     * The stage for the branch (e.g., PRODUCTION, BETA, DEVELOPMENT).
     * @default "PRODUCTION"
     */
    readonly stage?: string;
    /**
     * Environment variables specific to this branch.
     */
    readonly environmentVariables?: Record<string, string>;
    /**
     * Whether to enable pull request previews for this branch.
     * @default false
     */
    readonly pullRequestPreview?: boolean;
    /**
     * Whether to enable automatic builds for this branch.
     * @default true
     */
    readonly enableAutoBuild?: boolean;
    /**
     * The framework for this branch.
     */
    readonly framework?: string;
    /**
     * Branch-specific build specification.
     */
    readonly buildSpec?: string;
}
