"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const api = require("aws-cdk-lib/aws-apigateway");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const utils_1 = require("./utils");
const tags_1 = require("../common/tags");
class ApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = props.lambdaFunction;
        // Apply tags to both construct and CloudFormation stack
        (0, tags_1.applyTags)(this, props.tags);
        // Determine if we should use proxy mode
        const useProxy = props.proxy !== false && !props.customRoutes;
        if (useProxy) {
            // Prepare LambdaRestApi props with binary media types if specified
            const lambdaApiProps = {
                handler: props.lambdaFunction,
                restApiName: props.apiName,
                proxy: true,
                ...(props.binaryMediaTypes && {
                    binaryMediaTypes: props.binaryMediaTypes,
                }),
                ...props.lambdaApiProps,
            };
            this.apiGateway = new api.LambdaRestApi(this, "ApiGateway", lambdaApiProps);
        }
        else {
            // Prepare RestApi props with binary media types if specified
            const restApiProps = {
                restApiName: props.apiName,
                ...(props.binaryMediaTypes && {
                    binaryMediaTypes: props.binaryMediaTypes,
                }),
                ...props.restApiProps,
            };
            this.apiGateway = new api.RestApi(this, "ApiGateway", restApiProps);
            // Add custom routes if provided
            if (props.customRoutes) {
                this.addCustomRoutes(props.customRoutes);
            }
            // If no custom routes but proxy is disabled, add default integration
            if (!props.customRoutes && props.proxy === false) {
                this.apiGateway.root.addMethod("ANY", new api.LambdaIntegration(props.lambdaFunction));
            }
        }
        // Setup CloudWatch logging
        if (props.enableLogging) {
            const loggingResources = (0, utils_1.setupLogging)(this, props.apiName, props.logGroupProps);
            this.apiGatewayLogGroup = loggingResources.logGroup;
        }
        // Setup custom domain
        if (props.customDomainName) {
            const domainResources = this.setupCustomDomain(props);
            this.certificate = domainResources.certificate;
            this.domain = domainResources.domain;
            this.aRecord = domainResources.aRecord;
        }
        // Setup usage plan
        if (props.createUsagePlan) {
            this.usagePlan = this.setupUsagePlan();
        }
    }
    /**
     * Add custom routes to the API
     */
    addCustomRoutes(routes) {
        const resourceMap = new Map();
        routes.forEach((route) => {
            const pathParts = route.path
                .split("/")
                .filter((part) => part !== "");
            let currentResource = this.apiGateway.root;
            // Build nested resources
            let currentPath = "";
            for (const part of pathParts) {
                currentPath += `/${part}`;
                if (!resourceMap.has(currentPath)) {
                    currentResource = currentResource.addResource(part);
                    resourceMap.set(currentPath, currentResource);
                }
                else {
                    currentResource = resourceMap.get(currentPath);
                }
            }
            // Add method to the final resource
            currentResource.addMethod(route.method, new api.LambdaIntegration(route.handler), route.methodOptions);
        });
    }
    /**
     * Setup custom domain with certificate and Route53 record
     */
    setupCustomDomain(props) {
        let certificate;
        // Use existing certificate or create new one
        if (props.existingCertificate) {
            certificate = props.existingCertificate;
        }
        else if (props.hostedZone) {
            certificate = new acm.Certificate(this, "Certificate", {
                domainName: props.customDomainName,
                validation: acm.CertificateValidation.fromDns(props.hostedZone),
            });
        }
        else {
            throw new Error("Either certificateArn or hostedZone must be provided for custom domain");
        }
        // Create custom domain
        const domain = new api.DomainName(this, "CustomDomain", {
            domainName: props.customDomainName,
            certificate: certificate,
        });
        // Create base path mapping
        new api.BasePathMapping(this, "BasePathMapping", {
            domainName: domain,
            restApi: this.apiGateway,
        });
        let aRecord;
        // Create Route53 alias record if hosted zone provided
        if (props.hostedZone) {
            aRecord = new route53.ARecord(this, "CustomDomainAliasRecord", {
                zone: props.hostedZone,
                recordName: props.customDomainName,
                target: route53.RecordTarget.fromAlias(new targets.ApiGatewayDomain(domain)),
            });
        }
        return { certificate, domain, aRecord };
    }
    /**
     * Setup usage plan
     */
    setupUsagePlan() {
        return new api.UsagePlan(this, "UsagePlan", {
            name: `${this.apiGateway.restApiName}-usage-plan`,
            apiStages: [
                {
                    api: this.apiGateway,
                    stage: this.apiGateway.deploymentStage,
                },
            ],
        });
    }
    /**
     * Add a custom route after construction (for dynamic route addition)
     */
    addRoute(route) {
        const pathParts = route.path
            .split("/")
            .filter((part) => part !== "");
        let currentResource = this.apiGateway.root;
        // Navigate/create resource path
        for (const part of pathParts) {
            const existingResource = currentResource.getResource(part);
            if (existingResource) {
                currentResource = existingResource;
            }
            else {
                currentResource = currentResource.addResource(part);
            }
        }
        return currentResource.addMethod(route.method, new api.LambdaIntegration(route.handler), route.methodOptions);
    }
    get apiUrl() {
        return this.domain?.domainNameAliasDomainName ?? this.apiGateway.url;
    }
}
exports.ApiGatewayToLambda = ApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToLambda[_a] = { fqn: "must-cdk.ApiGatewayToLambda", version: "0.0.111" };
//# sourceMappingURL=data:application/json;base64,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