"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToOrigins = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
const tags_1 = require("../common/tags");
class CloudFrontToOrigins extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.aRecords = [];
        this.s3BucketsMap = new Map();
        this.httpOriginsMap = new Map();
        this.s3OriginsMap = new Map();
        // Validate that at least one origin is provided
        if ((!props.s3Origins || props.s3Origins.length === 0) &&
            (!props.httpOrigins || props.httpOrigins.length === 0)) {
            throw new Error("At least one origin (an S3 Bucket or a HTTP domain) must be provided");
        }
        // Apply intelligent defaults
        const enhancedProps = this.applyDefaults(props);
        // Apply tags to both construct and CloudFormation stack
        (0, tags_1.applyTags)(this, props.tags);
        let certificate;
        let logBucket;
        // Setup all S3 origins
        if (enhancedProps.s3Origins) {
            for (const s3Config of enhancedProps.s3Origins) {
                this.s3BucketsMap.set(s3Config.id, s3Config.bucket);
                const s3Origin = this.setupS3Origin(s3Config);
                this.s3OriginsMap.set(s3Config.id, s3Origin);
            }
        }
        // Setup all HTTP origins
        if (enhancedProps.httpOrigins) {
            for (const httpConfig of enhancedProps.httpOrigins) {
                const httpOrigin = this.setupHttpOrigin(httpConfig);
                this.httpOriginsMap.set(httpConfig.id, httpOrigin);
            }
        }
        // Setup custom domain and certificate
        if (enhancedProps.customDomainName) {
            certificate = this.setupCustomDomain(enhancedProps);
        }
        // Setup logging (enabled by default unless explicitly disabled)
        if (enhancedProps.enableLogging !== false) {
            logBucket = this.setupLogging(enhancedProps);
        }
        // Create CloudFront distribution
        this.distribution = this.createDistribution(enhancedProps, {
            certificate,
            logBucket,
        });
        // Setup Route53 records for all domains
        if (enhancedProps.createRoute53Records !== false &&
            enhancedProps.hostedZone) {
            this.setupRoute53Records(enhancedProps);
        }
        // Assign readonly properties
        this.logBucket = logBucket;
        this.certificate = certificate;
        this.domainNames = this.getAllDomainNames(enhancedProps);
    }
    /**
     * Get all domain names (primary + additional)
     */
    getAllDomainNames(props) {
        const domains = [];
        if (props.customDomainName) {
            domains.push(props.customDomainName);
        }
        if (props.additionalDomainNames) {
            domains.push(...props.additionalDomainNames);
        }
        return domains.length > 0 ? domains : undefined;
    }
    /**
     * Apply intelligent defaults based on origin configuration
     */
    applyDefaults(props) {
        return {
            // Apply base defaults
            enableLogging: true,
            createRoute53Records: true,
            priceClass: cloudfront.PriceClass.PRICE_CLASS_100,
            enabled: true,
            // Override with user props
            ...props,
            // Apply HTTP origin protocol defaults
            httpOrigins: props.httpOrigins?.map((httpOrigin) => ({
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
                ...httpOrigin,
            })),
            // Merge cache behaviors
            cacheBehaviors: [...(props.cacheBehaviors || [])],
        };
    }
    /**
     * Setup S3 origin using Origin Access Identity (OAI)
     */
    setupS3Origin(s3Config) {
        // Use Origin Access Identity
        const originAccessIdentity = s3Config.originAccessIdentity ||
            new cloudfront.OriginAccessIdentity(this, `OAI-${s3Config.id}`, {
                comment: `OAI for ${s3Config.bucket.bucketName}`,
            });
        // Grant CloudFront access to S3 bucket
        s3Config.bucket.grantRead(originAccessIdentity);
        return origins.S3BucketOrigin.withOriginAccessIdentity(s3Config.bucket, {
            originAccessIdentity,
            originPath: s3Config.originPath,
            originId: s3Config.id,
            ...s3Config.s3OriginProps,
        });
    }
    /**
     * Setup HTTP origin
     */
    setupHttpOrigin(httpConfig) {
        return new origins.HttpOrigin(httpConfig.domainName, {
            originPath: httpConfig.originPath,
            protocolPolicy: httpConfig.protocolPolicy || cloudfront.OriginProtocolPolicy.HTTPS_ONLY,
            httpPort: httpConfig.httpPort,
            httpsPort: httpConfig.httpsPort,
            originId: httpConfig.id,
            ...httpConfig.httpOriginProps,
        });
    }
    /**
     * Setup custom domain and certificate
     */
    setupCustomDomain(props) {
        // If certificate is provided directly, use it
        if (props.certificate) {
            return props.certificate;
        }
        // Auto-create certificate in us-east-1 if hostedZone is provided
        if (props.hostedZone && props.customDomainName) {
            return this.createCrossRegionCertificate(props);
        }
        throw new Error("Either certificate or hostedZone must be provided for custom domain. " +
            "If hostedZone is provided, certificate will be automatically created in us-east-1.");
    }
    /**
     * Create certificate in us-east-1 for CloudFront
     */
    createCrossRegionCertificate(props) {
        const allDomains = this.getAllDomainNames(props) || [
            props.customDomainName,
        ];
        // Use DnsValidatedCertificate to create certificate in us-east-1
        return new acm.DnsValidatedCertificate(this, "Certificate", {
            domainName: props.customDomainName,
            subjectAlternativeNames: allDomains.length > 1 ? allDomains.slice(1) : undefined,
            hostedZone: props.hostedZone,
            region: "us-east-1",
        });
    }
    /**
     * Setup CloudFront logging
     */
    setupLogging(props) {
        return (props.logBucket ||
            new s3.Bucket(this, "LogBucket", {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                versioned: false,
                lifecycleRules: [
                    {
                        expiration: aws_cdk_lib_1.Duration.days(90),
                        transitions: [
                            {
                                storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                                transitionAfter: aws_cdk_lib_1.Duration.days(30),
                            },
                        ],
                    },
                ],
            }));
    }
    /**
     * Get origin by ID
     */
    getOriginById(originId) {
        // Check S3 origins first
        const s3Origin = this.s3OriginsMap.get(originId);
        if (s3Origin) {
            return s3Origin;
        }
        // Check HTTP origins
        const httpOrigin = this.httpOriginsMap.get(originId);
        if (httpOrigin) {
            return httpOrigin;
        }
        throw new Error(`Origin with ID '${originId}' not found`);
    }
    /**
     * Determine default origin ID
     */
    getDefaultOriginId(props) {
        if (props.defaultOriginId) {
            return props.defaultOriginId;
        }
        // Default to first HTTP origin, then first S3 origin
        if (props.httpOrigins && props.httpOrigins.length > 0) {
            return props.httpOrigins[0].id;
        }
        if (props.s3Origins && props.s3Origins.length > 0) {
            return props.s3Origins[0].id;
        }
        throw new Error("No origins configured");
    }
    /**
     * Create CloudFront distribution
     */
    createDistribution(props, resources) {
        // Determine default origin
        const defaultOriginId = this.getDefaultOriginId(props);
        const defaultOrigin = this.getOriginById(defaultOriginId);
        const defaultBehavior = {
            origin: defaultOrigin,
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            compress: true,
        };
        // Build additional behaviors
        const additionalBehaviors = {};
        if (props.cacheBehaviors) {
            for (const behavior of props.cacheBehaviors) {
                const origin = this.getOriginById(behavior.originId);
                // Build behavior options object properly
                const behaviorConfig = {
                    origin,
                    viewerProtocolPolicy: behavior.viewerProtocolPolicy ||
                        cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    allowedMethods: behavior.allowedMethods,
                    cachedMethods: behavior.cachedMethods,
                    compress: behavior.compress ?? true,
                };
                // Add cache policy (prioritize direct policy over ID)
                if (behavior.cachePolicy) {
                    behaviorConfig.cachePolicy = behavior.cachePolicy;
                }
                else if (behavior.cachePolicyId) {
                    behaviorConfig.cachePolicy = cloudfront.CachePolicy.fromCachePolicyId(this, `CachePolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.cachePolicyId);
                }
                // Add origin request policy
                if (behavior.originRequestPolicy) {
                    behaviorConfig.originRequestPolicy = behavior.originRequestPolicy;
                }
                else if (behavior.originRequestPolicyId) {
                    behaviorConfig.originRequestPolicy =
                        cloudfront.OriginRequestPolicy.fromOriginRequestPolicyId(this, `OriginRequestPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.originRequestPolicyId);
                }
                // Add response headers policy
                if (behavior.responseHeadersPolicy) {
                    behaviorConfig.responseHeadersPolicy = behavior.responseHeadersPolicy;
                }
                else if (behavior.responseHeadersPolicyId) {
                    behaviorConfig.responseHeadersPolicy =
                        cloudfront.ResponseHeadersPolicy.fromResponseHeadersPolicyId(this, `ResponseHeadersPolicy-${behavior.pathPattern.replace(/[^a-zA-Z0-9]/g, "")}`, behavior.responseHeadersPolicyId);
                }
                additionalBehaviors[behavior.pathPattern] =
                    behaviorConfig;
            }
        }
        // Ensure all origins are referenced by creating behaviors for unused origins
        const usedOriginIds = new Set([defaultOriginId]);
        // Track origins used in cache behaviors
        if (props.cacheBehaviors) {
            props.cacheBehaviors.forEach((behavior) => {
                usedOriginIds.add(behavior.originId);
            });
        }
        // Create default behaviors for unused origins
        const allOriginIds = [
            ...this.s3OriginsMap.keys(),
            ...this.httpOriginsMap.keys(),
        ];
        allOriginIds.forEach((originId) => {
            if (!usedOriginIds.has(originId)) {
                const origin = this.getOriginById(originId);
                // Create a unique path pattern for this origin (lowercase)
                const pathPattern = `/${originId.toLowerCase()}/*`;
                additionalBehaviors[pathPattern] = {
                    origin,
                    viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                    compress: true,
                };
            }
        });
        // Prepare distribution props
        const distributionProps = {
            comment: props.comment || `CloudFront distribution for ${this.node.id}`,
            defaultBehavior,
            additionalBehaviors,
            domainNames: this.getAllDomainNames(props),
            certificate: resources.certificate,
            defaultRootObject: props.defaultRootObject,
            enabled: props.enabled,
            priceClass: props.priceClass,
            geoRestriction: props.geoRestriction,
            enableLogging: props.enableLogging !== false,
            logBucket: resources.logBucket,
            logFilePrefix: props.logPrefix,
            logIncludesCookies: props.logIncludeCookies,
            errorResponses: props.errorPages,
            webAclId: props.webAclId,
            enableIpv6: props.enableIpv6,
            httpVersion: props.httpVersion,
        };
        return new cloudfront.Distribution(this, "Distribution", distributionProps);
    }
    /**
     * Setup Route53 alias records for all domains
     */
    setupRoute53Records(props) {
        if (!props.hostedZone) {
            return;
        }
        const domains = this.getAllDomainNames(props);
        if (!domains) {
            return;
        }
        // Create A record for each domain
        domains.forEach((domain, index) => {
            const recordId = index === 0 ? "AliasRecord" : `AliasRecord${index + 1}`;
            const aRecord = new route53.ARecord(this, recordId, {
                zone: props.hostedZone,
                recordName: domain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
            });
            this.aRecords.push(aRecord);
        });
    }
    /**
     * Get S3 bucket by origin ID
     */
    getS3Bucket(originId) {
        return this.s3BucketsMap.get(originId);
    }
    /**
     * Get HTTP origin by origin ID
     */
    getHttpOrigin(originId) {
        return this.httpOriginsMap.get(originId);
    }
    /**
     * Get all S3 bucket origin IDs
     */
    get s3OriginIds() {
        return Array.from(this.s3BucketsMap.keys());
    }
    /**
     * Get all HTTP origin IDs
     */
    get httpOriginIds() {
        return Array.from(this.httpOriginsMap.keys());
    }
    /**
     * Get all S3 buckets as an array of objects with ID and bucket
     */
    get s3Origins() {
        return Array.from(this.s3BucketsMap.entries()).map(([id, bucket]) => ({
            id,
            bucket,
        }));
    }
    /**
     * Get all HTTP origins as an array of objects with ID and origin
     */
    get httpOrigins() {
        return Array.from(this.httpOriginsMap.entries()).map(([id, origin]) => ({
            id,
            origin,
        }));
    }
    /**
     * Get the CloudFront distribution domain name
     */
    get distributionDomainName() {
        return this.distribution.distributionDomainName;
    }
    /**
     * Get the CloudFront distribution URL with protocol
     */
    get distributionUrl() {
        return `https://${this.distribution.distributionDomainName}`;
    }
    /**
     * Get the custom domain URL (if configured)
     */
    get customDomainUrl() {
        return this.domainNames?.[0] ? `https://${this.domainNames[0]}` : undefined;
    }
}
exports.CloudFrontToOrigins = CloudFrontToOrigins;
_a = JSII_RTTI_SYMBOL_1;
CloudFrontToOrigins[_a] = { fqn: "must-cdk.CloudFrontToOrigins", version: "0.0.111" };
//# sourceMappingURL=data:application/json;base64,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