import { Construct } from "constructs";
/**
 * Reads tags from TAGS environment variable and applies them to all resources.
 */
/**
 * Parses the TAGS environment variable and returns tags as an object.
 * Supports multiple formats:
 * 1. Key-value pairs: key1=value1,key2=value2
 * 2. JSON string: {"key1":"value1","key2":"value2"}
 * 3. JSON object (dict): same as JSON string
 *
 * @returns Parsed tags from environment variable
 */
export declare function getTagsFromEnvironment(): Record<string, string>;
/**
 * Merges user-provided tags with environment tags.
 * Environment tags take precedence over user tags.
 *
 * @param userTags - Optional user-provided tags
 * @returns Merged tags with environment tags taking precedence
 */
export declare function mergeTags(userTags?: Record<string, string>): Record<string, string>;
/**
 * Applies tags to both the construct and the CloudFormation stack.
 * This ensures tags are visible at both the resource and stack level.
 *
 * @param construct - The construct to apply tags to
 * @param userTags - Optional user-provided tags
 * @returns Final merged tags that were applied
 */
export declare function applyTags(construct: Construct, userTags?: Record<string, string>): Record<string, string>;
/**
 * Converts tags object to AWS CDK tag format for CfnResource.
 *
 * @param tags - Tags object
 * @returns Array of tag objects in CDK format
 */
export declare function formatTagsForCfn(tags: Record<string, string>): Array<{
    key: string;
    value: string;
}>;
/**
 * Type for tag-aware construct props.
 */
export interface TaggableProps {
    /**
     * Additional tags to apply to resources.
     * Note: Tags from TAGS environment variable will take precedence over these tags.
     *
     * Environment variable supports multiple formats:
     * - Key-value pairs: TAGS=key1=value1,key2=value2
     * - JSON string: TAGS='{"key1":"value1","key2":"value2"}'
     * - JSON object (dict): TAGS={"Product":"Mufin","Owner":"Platform"}
     */
    readonly tags?: Record<string, string>;
}
