"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTagsFromEnvironment = getTagsFromEnvironment;
exports.mergeTags = mergeTags;
exports.applyTags = applyTags;
exports.formatTagsForCfn = formatTagsForCfn;
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Reads tags from TAGS environment variable and applies them to all resources.
 */
/**
 * Parses the TAGS environment variable and returns tags as an object.
 * Supports multiple formats:
 * 1. Key-value pairs: key1=value1,key2=value2
 * 2. JSON string: {"key1":"value1","key2":"value2"}
 * 3. JSON object (dict): same as JSON string
 *
 * @returns Parsed tags from environment variable
 */
function getTagsFromEnvironment() {
    const tagsEnv = process.env.TAGS;
    if (!tagsEnv || tagsEnv.trim() === "") {
        return {};
    }
    const trimmedTags = tagsEnv.trim();
    try {
        // Try to parse as JSON first (handles both JSON string and dict formats)
        if (trimmedTags.startsWith("{") && trimmedTags.endsWith("}")) {
            const parsedTags = JSON.parse(trimmedTags);
            // Ensure all values are strings
            const stringTags = {};
            for (const [key, value] of Object.entries(parsedTags)) {
                if (typeof value === "string") {
                    stringTags[key] = value;
                }
                else if (value === null || value === undefined) {
                    stringTags[key] = String(value);
                }
                else if (typeof value === "object") {
                    // For objects and arrays, JSON stringify them
                    stringTags[key] = JSON.stringify(value);
                }
                else {
                    // For numbers, booleans, etc.
                    stringTags[key] = String(value);
                }
            }
            return stringTags;
        }
        // Fall back to key=value,key2=value2 format
        const tags = {};
        const tagPairs = trimmedTags.split(",");
        for (const pair of tagPairs) {
            const trimmedPair = pair.trim();
            if (trimmedPair === "")
                continue;
            const [key, ...valueParts] = trimmedPair.split("=");
            if (key && valueParts.length > 0) {
                // Join back in case value contains '=' characters
                const value = valueParts.join("=");
                tags[key.trim()] = value.trim();
            }
        }
        return tags;
    }
    catch (error) {
        console.warn(`Failed to parse TAGS environment variable: ${error}`);
        return {};
    }
}
/**
 * Merges user-provided tags with environment tags.
 * Environment tags take precedence over user tags.
 *
 * @param userTags - Optional user-provided tags
 * @returns Merged tags with environment tags taking precedence
 */
function mergeTags(userTags) {
    const envTags = getTagsFromEnvironment();
    return {
        ...userTags,
        ...envTags,
    };
}
/**
 * Applies tags to both the construct and the CloudFormation stack.
 * This ensures tags are visible at both the resource and stack level.
 *
 * @param construct - The construct to apply tags to
 * @param userTags - Optional user-provided tags
 * @returns Final merged tags that were applied
 */
function applyTags(construct, userTags) {
    const finalTags = mergeTags(userTags);
    // Apply tags to the construct (inherits to child resources)
    Object.entries(finalTags).forEach(([key, value]) => {
        aws_cdk_lib_1.Tags.of(construct).add(key, value);
    });
    // Apply tags to the CloudFormation stack
    const stack = aws_cdk_lib_1.Stack.of(construct);
    Object.entries(finalTags).forEach(([key, value]) => {
        aws_cdk_lib_1.Tags.of(stack).add(key, value);
    });
    return finalTags;
}
/**
 * Converts tags object to AWS CDK tag format for CfnResource.
 *
 * @param tags - Tags object
 * @returns Array of tag objects in CDK format
 */
function formatTagsForCfn(tags) {
    return Object.entries(tags).map(([key, value]) => ({ key, value }));
}
//# sourceMappingURL=data:application/json;base64,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