"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeDeployComponent = exports.LoadBalancerComponent = exports.EcsServiceComponent = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const tags_1 = require("../common/tags");
/**
 * Creates and configures an ECS Fargate service with task definition and containers
 */
class EcsServiceComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.containers = [];
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, "LogGroup", {
            logGroupName: `/aws/ecs/${props.serviceName}`,
            retention: aws_cdk_lib_1.aws_logs.RetentionDays.INFINITE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const taskRole = props.taskDefinitionConfig?.taskRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskRole", {
                roleName: `${props.serviceName}-task-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task role for ${props.serviceName}`,
            });
        const taskExecRole = props.taskDefinitionConfig?.executionRole ??
            new aws_cdk_lib_1.aws_iam.Role(this, "TaskExecRole", {
                roleName: `${props.serviceName}-task-exec-role`,
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal("ecs-tasks.amazonaws.com"),
                description: `Task execution role for ${props.serviceName}`,
                managedPolicies: [
                    aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonECSTaskExecutionRolePolicy"),
                ],
            });
        this.taskRole = taskRole;
        this.taskExecutionRole = taskExecRole;
        this.taskDef = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, "TaskDefinition", {
            memoryLimitMiB: props.taskDefinitionConfig?.memoryLimitMiB ?? 2048,
            cpu: props.taskDefinitionConfig?.cpu ?? 1024,
            executionRole: this.taskExecutionRole,
            taskRole: this.taskRole,
        });
        this.addContainers(props.containers);
        this.service = new aws_cdk_lib_1.aws_ecs.FargateService(this, "Service", {
            cluster: props.cluster,
            serviceName: props.serviceName,
            taskDefinition: this.taskDef,
            assignPublicIp: props.serviceConfig?.assignPublicIp ?? false,
            desiredCount: props.serviceConfig?.desiredCount ?? 1,
            healthCheckGracePeriod: props.serviceConfig?.healthCheckGracePeriod ?? aws_cdk_lib_1.Duration.seconds(300),
            deploymentController: {
                type: aws_cdk_lib_1.aws_ecs.DeploymentControllerType.CODE_DEPLOY,
            },
            vpcSubnets: props.subnets,
            securityGroups: props.securityGroups,
        });
    }
    /**
     * Add containers to the task definition using native CDK interfaces
     */
    addContainers(containers) {
        containers.forEach((container) => {
            const containerOptions = {
                logging: aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                    streamPrefix: "ecs",
                    logGroup: this.logGroup,
                }),
                essential: true,
                ...container.options,
            };
            const containerDef = this.taskDef.addContainer(container.name, containerOptions);
            this.containers.push(containerDef);
        });
    }
}
exports.EcsServiceComponent = EcsServiceComponent;
/**
 * Creates and configures an Application Load Balancer with listeners and target groups
 */
class LoadBalancerComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        this.loadBalancer = new aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationLoadBalancer(this, "LoadBalancer", {
            vpc: props.vpc,
            loadBalancerName: props.loadBalancerConfig?.loadBalancerName ??
                `${props.serviceName}-alb`,
            internetFacing: props.loadBalancerConfig?.internetFacing ?? false,
            vpcSubnets: props.loadBalancerConfig?.vpcSubnets ?? props.subnets,
            securityGroup: props.loadBalancerConfig?.securityGroups?.[0],
        });
        if (props.loadBalancerConfig?.idleTimeoutSeconds !== undefined) {
            this.loadBalancer.setAttribute("idle_timeout.timeout_seconds", props.loadBalancerConfig.idleTimeoutSeconds.toString());
        }
        this.productionListener = this.loadBalancer.addListener("ProductionListener", {
            port: props.listenerConfigs?.production?.port ?? 80,
            protocol: props.listenerConfigs?.production?.protocol ??
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            certificates: props.listenerConfigs?.production?.certificates,
            defaultAction: props.listenerConfigs?.production?.defaultAction,
        });
        // Always create test listener for blue-green deployment (use defaults if not configured)
        this.testListener = this.loadBalancer.addListener("TestListener", {
            port: props.listenerConfigs?.test?.port ?? 8080,
            protocol: props.listenerConfigs?.test?.protocol ?? aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            certificates: props.listenerConfigs?.test?.certificates,
            defaultAction: props.listenerConfigs?.test?.defaultAction,
        });
        this.productionTargetGroup = this.productionListener.addTargets("ProductionTarget", {
            port: props.targetGroupConfigs?.production?.port ?? props.targetPort,
            protocol: props.targetGroupConfigs?.production?.protocol ??
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [
                props.service,
            ],
            healthCheck: props.targetGroupConfigs?.production?.healthCheck ?? {
                path: "/",
                interval: aws_cdk_lib_1.Duration.seconds(30),
                timeout: aws_cdk_lib_1.Duration.seconds(10),
                healthyThresholdCount: 2,
                unhealthyThresholdCount: 5,
            },
            stickinessCookieDuration: props.targetGroupConfigs?.production?.stickinessCookieDuration,
            slowStart: props.targetGroupConfigs?.production?.slowStart,
            loadBalancingAlgorithmType: props.targetGroupConfigs?.production?.loadBalancingAlgorithmType,
            targetGroupName: props.targetGroupConfigs?.production?.targetGroupName,
        });
        // Always create test target group for blue-green deployment (use defaults if not configured)
        this.testTargetGroup = this.testListener.addTargets("TestTarget", {
            port: props.targetGroupConfigs?.test?.port ?? props.targetPort,
            protocol: props.targetGroupConfigs?.test?.protocol ??
                aws_cdk_lib_1.aws_elasticloadbalancingv2.ApplicationProtocol.HTTP,
            targets: [],
            healthCheck: props.targetGroupConfigs?.test?.healthCheck ?? {
                path: "/",
                interval: aws_cdk_lib_1.Duration.seconds(30),
                timeout: aws_cdk_lib_1.Duration.seconds(10),
                healthyThresholdCount: 2,
                unhealthyThresholdCount: 5,
            },
            stickinessCookieDuration: props.targetGroupConfigs?.test?.stickinessCookieDuration,
            slowStart: props.targetGroupConfigs?.test?.slowStart,
            loadBalancingAlgorithmType: props.targetGroupConfigs?.test?.loadBalancingAlgorithmType,
            targetGroupName: props.targetGroupConfigs?.test?.targetGroupName,
        });
    }
}
exports.LoadBalancerComponent = LoadBalancerComponent;
/**
 * Creates and configures a CodeDeploy application and deployment group
 */
class CodeDeployComponent extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.tags) {
            (0, tags_1.applyTags)(this, props.tags);
        }
        this.application = new aws_cdk_lib_1.aws_codedeploy.EcsApplication(this, "Application", {
            applicationName: props.applicationConfig?.applicationName ?? props.appName,
        });
        // CodeDeploy requires two distinct target groups for blue-green deployments
        // We must always have both testListener and testTargetGroup for proper blue-green deployment
        if (!props.testListener || !props.testTargetGroup) {
            throw new Error("ECS CodeDeploy construct requires both testListener and testTargetGroup for blue-green deployments. Both blue and green target groups must be distinct AWS resources.");
        }
        this.deploymentGroup = new aws_cdk_lib_1.aws_codedeploy.EcsDeploymentGroup(this, "DeploymentGroup", {
            application: this.application,
            service: props.service,
            deploymentGroupName: props.deploymentGroupConfig?.deploymentGroupName ?? props.appName,
            deploymentConfig: props.deploymentGroupConfig?.deploymentConfig ??
                aws_cdk_lib_1.aws_codedeploy.EcsDeploymentConfig.ALL_AT_ONCE,
            blueGreenDeploymentConfig: props.deploymentGroupConfig
                ?.blueGreenDeploymentConfig ?? {
                blueTargetGroup: props.productionTargetGroup,
                greenTargetGroup: props.testTargetGroup,
                listener: props.productionListener,
                testListener: props.testListener,
            },
        });
    }
}
exports.CodeDeployComponent = CodeDeployComponent;
//# sourceMappingURL=data:application/json;base64,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