import { aws_ec2 as ec2, aws_ecs as ecs, aws_iam as iam, aws_elasticloadbalancingv2 as elbv2, aws_codedeploy as codedeploy, Duration } from "aws-cdk-lib";
import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import { TaggableProps } from "../common/tags";
/**
 * Container configuration using native CDK interfaces
 */
export interface ContainerConfig {
    /** Container name */
    readonly name: string;
    /** Native CDK container definition options */
    readonly options: ecs.ContainerDefinitionOptions;
}
/**
 * Task definition configuration options
 */
export interface TaskDefinitionConfig {
    /** The number of cpu units used by the task */
    readonly cpu?: number;
    /** The amount (in MiB) of memory used by the task */
    readonly memoryLimitMiB?: number;
    /** Task role for the task */
    readonly taskRole?: iam.IRole;
    /** Execution role for the task */
    readonly executionRole?: iam.IRole;
}
/**
 * Service configuration options
 */
export interface ServiceConfig {
    /** The desired number of instantiations of the task definition to keep running */
    readonly desiredCount?: number;
    /** Whether to assign public IP addresses to the task's ENI */
    readonly assignPublicIp?: boolean;
    /** Health check grace period */
    readonly healthCheckGracePeriod?: Duration;
}
/**
 * Load balancer configuration options
 */
export interface LoadBalancerConfig {
    /** Whether the load balancer is internet facing */
    readonly internetFacing?: boolean;
    /** The VPC subnets to use */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /** Security groups for the load balancer */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /** Load balancer name */
    readonly loadBalancerName?: string;
    /** Idle timeout in seconds for the ALB */
    readonly idleTimeoutSeconds?: number;
}
/**
 * Listener configuration options
 */
export interface ListenerConfig {
    /** The port on which the listener listens for requests */
    readonly port?: number;
    /** The protocol for connections from clients to the load balancer */
    readonly protocol?: elbv2.ApplicationProtocol;
    /** Certificate list of ACM cert ARNs */
    readonly certificates?: ICertificate[];
    /** The default actions for the listener */
    readonly defaultAction?: elbv2.ListenerAction;
}
/**
 * Target group configuration options
 */
export interface TargetGroupConfig {
    /** The port on which the targets receive traffic */
    readonly port?: number;
    /** The protocol to use for routing traffic to the targets */
    readonly protocol?: elbv2.ApplicationProtocol;
    /** Health check configuration */
    readonly healthCheck?: elbv2.HealthCheck;
    /** The time period during which the load balancer sends a newly registered target a linearly increasing share of the traffic */
    readonly slowStart?: Duration;
    /** The load balancing algorithm to use */
    readonly loadBalancingAlgorithmType?: elbv2.TargetGroupLoadBalancingAlgorithmType;
    /** The stickiness cookie duration */
    readonly stickinessCookieDuration?: Duration;
    /** Target group name */
    readonly targetGroupName?: string;
}
/**
 * CodeDeploy application configuration options
 */
export interface CodeDeployApplicationConfig {
    /** The name of the application */
    readonly applicationName?: string;
}
/**
 * CodeDeploy deployment group configuration options
 */
export interface DeploymentGroupConfig {
    /** The name of the deployment group */
    readonly deploymentGroupName?: string;
    /** The deployment configuration to use */
    readonly deploymentConfig?: codedeploy.IEcsDeploymentConfig;
    /** Configuration for blue-green ECS deployments */
    readonly blueGreenDeploymentConfig?: codedeploy.EcsBlueGreenDeploymentConfig;
}
/**
 * Properties for the EcsCodeDeploy construct using native AWS CDK interfaces.
 */
export interface EcsCodeDeployProps extends TaggableProps {
    /** VPC in which to deploy ECS and ALB resources. */
    readonly vpc: ec2.IVpc;
    /** ECS Cluster where the service will run. */
    readonly cluster: ecs.ICluster;
    /** Base name used for resources like log groups, roles, services, etc. */
    readonly serviceName: string;
    /** Container configurations using native CDK interfaces */
    readonly containers: ContainerConfig[];
    /** Task definition configuration options */
    readonly taskDefinition?: TaskDefinitionConfig;
    /** Service configuration options */
    readonly service?: ServiceConfig;
    /** Load balancer configuration options */
    readonly loadBalancer?: LoadBalancerConfig;
    /** Production listener configuration */
    readonly productionListener?: ListenerConfig;
    /** Test listener configuration (optional - defaults will be used if not provided) */
    readonly testListener?: ListenerConfig;
    /** Production target group configuration */
    readonly productionTargetGroup?: TargetGroupConfig;
    /** Test target group configuration (optional - defaults will be used if not provided) */
    readonly testTargetGroup?: TargetGroupConfig;
    /** CodeDeploy application configuration */
    readonly codeDeployApplication?: CodeDeployApplicationConfig;
    /** CodeDeploy deployment group configuration */
    readonly deploymentGroup?: DeploymentGroupConfig;
    /** Security groups for ECS service (required) */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * Subnets for ECS tasks (required)
     *
     * IMPORTANT: This controls where your ECS containers run. Use private subnets for security.
     * To limit to specific AZs: { availabilityZones: ['us-east-1a'] }
     * For all private subnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS }
     */
    readonly taskSubnets: ec2.SubnetSelection;
    /**
     * Subnets for ALB (optional, defaults to taskSubnets)
     *
     * IMPORTANT: This controls where your Load Balancer runs, NOT your containers!
     * - For internet-facing ALBs: Use PUBLIC subnets across multiple AZs
     * - For internal ALBs: Use PRIVATE subnets across multiple AZs
     * - Common mistake: Restricting ALB to single AZ reduces availability
     *
     * Examples:
     * - All public subnets: { subnetType: ec2.SubnetType.PUBLIC }
     * - Specific AZs: { availabilityZones: ['us-east-1a', 'us-east-1b'] }
     * - Default (all subnets): {} or undefined
     */
    readonly loadBalancerSubnets?: ec2.SubnetSelection;
    /** The port to expose on the target group (defaults to first container's port) */
    readonly targetPort?: number;
}
