"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsSolutionsChecks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const nag_pack_1 = require("../nag-pack");
const nag_rules_1 = require("../nag-rules");
const apigw_1 = require("../rules/apigw");
const appsync_1 = require("../rules/appsync");
const autoscaling_1 = require("../rules/autoscaling");
const cloud9_1 = require("../rules/cloud9");
const cloudfront_1 = require("../rules/cloudfront");
const codebuild_1 = require("../rules/codebuild");
const cognito_1 = require("../rules/cognito");
const documentdb_1 = require("../rules/documentdb");
const dynamodb_1 = require("../rules/dynamodb");
const ec2_1 = require("../rules/ec2");
const ecr_1 = require("../rules/ecr");
const ecs_1 = require("../rules/ecs");
const efs_1 = require("../rules/efs");
const eks_1 = require("../rules/eks");
const elasticache_1 = require("../rules/elasticache");
const elasticbeanstalk_1 = require("../rules/elasticbeanstalk");
const elb_1 = require("../rules/elb");
const emr_1 = require("../rules/emr");
const eventbridge_1 = require("../rules/eventbridge");
const glue_1 = require("../rules/glue");
const iam_1 = require("../rules/iam");
const kinesis_1 = require("../rules/kinesis");
const kms_1 = require("../rules/kms");
const lambda_1 = require("../rules/lambda");
const lex_1 = require("../rules/lex");
const mediastore_1 = require("../rules/mediastore");
const msk_1 = require("../rules/msk");
const neptune_1 = require("../rules/neptune");
const opensearch_1 = require("../rules/opensearch");
const quicksight_1 = require("../rules/quicksight");
const rds_1 = require("../rules/rds");
const redshift_1 = require("../rules/redshift");
const s3_1 = require("../rules/s3");
const sagemaker_1 = require("../rules/sagemaker");
const secretsmanager_1 = require("../rules/secretsmanager");
const sns_1 = require("../rules/sns");
const sqs_1 = require("../rules/sqs");
const stepfunctions_1 = require("../rules/stepfunctions");
const timestream_1 = require("../rules/timestream");
const vpc_1 = require("../rules/vpc");
/**
 * Check Best practices based on AWS Solutions Security Matrix
 *
 */
class AwsSolutionsChecks extends nag_pack_1.NagPack {
    constructor(props) {
        super(props);
        this.packName = 'AwsSolutions';
    }
    visit(node) {
        if (node instanceof aws_cdk_lib_1.CfnResource) {
            this.checkCompute(node);
            this.checkStorage(node);
            this.checkDatabases(node);
            this.checkNetworkDelivery(node);
            this.checkManagementGovernance(node);
            this.checkMachineLearning(node);
            this.checkAnalytics(node);
            this.checkSecurityCompliance(node);
            this.checkServerless(node);
            this.checkApplicationIntegration(node);
            this.checkMediaServices(node);
            this.checkDeveloperTools(node);
            this.checkLambda(node);
        }
    }
    /**
     * Check Compute Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkCompute(node) {
        this.applyRule({
            ruleSuffixOverride: 'EB1',
            info: 'The Elastic Beanstalk environment is not configured to use a specific VPC.',
            explanation: 'Use a non-default VPC in order to separate your environment from default resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkVPCSpecified,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB3',
            info: 'The Elastic Beanstalk environment does not have managed updates enabled.',
            explanation: 'Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticbeanstalk_1.ElasticBeanstalkManagedUpdatesEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EB4',
            info: 'The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.',
            explanation: 'Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: elasticbeanstalk_1.ElasticBeanstalkEC2InstanceLogsToS3,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC23',
            info: 'The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.',
            explanation: 'Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when  all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2RestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC26',
            info: 'The resource creates one or more EBS volumes that have encryption disabled.',
            explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2EBSVolumeEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC27',
            info: 'The Security Group does not have a description.',
            explanation: 'Descriptions help simplify operations and remove any opportunities for operator errors.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2SecurityGroupDescription,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC28',
            info: 'The EC2 instance/AutoScaling launch configuration does not have detailed monitoring enabled.',
            explanation: 'Monitoring data helps make better decisions on architecting and managing compute resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceDetailedMonitoringEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EC29',
            info: 'The EC2 instance is not part of an ASG and has Termination Protection disabled.',
            explanation: 'Termination Protection safety feature enabled in order to protect the instances from being accidentally terminated.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ec2_1.EC2InstanceTerminationProtection,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECR1',
            info: 'The ECR Repository allows open access.',
            explanation: 'Removing * principals in an ECR Repository helps protect against unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecr_1.ECROpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS2',
            info: 'The ECS Task Definition includes a container definition that directly specifies environment variables.',
            explanation: 'Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionNoEnvironmentVariables,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS4',
            info: 'The ECS Cluster has CloudWatch Container Insights disabled.',
            explanation: 'CloudWatch Container Insights allow operators to gain a better perspective on how the cluster’s applications and microservices are performing.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSClusterCloudWatchContainerInsights,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ECS7',
            info: 'One or more containers in the ECS Task Definition do not have container logging enabled.',
            explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: ecs_1.ECSTaskDefinitionContainerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS1',
            info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
            explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterNoEndpointPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EKS2',
            info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
            explanation: 'EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::api'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: eks_1.EKSClusterControlPlaneLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB1',
            info: 'The CLB is used for incoming HTTP/HTTPS traffic. Use ALBs instead.',
            explanation: 'HTTP/HTTPS applications (monolithic or containerized) should use the ALB instead of the CLB for enhanced incoming traffic distribution, better performance and lower costs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBNoInboundHttpHttps,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB2',
            info: 'The ELB does not have access logs enabled.',
            explanation: 'Access logs allow operators to analyze traffic patterns and identify and troubleshoot security issues.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB3',
            info: 'The CLB does not have connection draining enabled.',
            explanation: 'With Connection Draining feature enabled, if an EC2 backend instance fails health checks The CLB will not send any new requests to the unhealthy instance. However, it will still allow existing (in-flight) requests to complete for the duration of the configured timeout.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.CLBConnectionDraining,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB4',
            info: 'The CLB does not use at least two AZs with the Cross-Zone Load Balancing feature enabled.',
            explanation: 'CLBs can distribute the traffic evenly across all backend instances. To use Cross-Zone Load Balancing at optimal level, the system should maintain an equal EC2 capacity distribution in each of the AZs registered with the load balancer.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBCrossZoneLoadBalancingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'ELB5',
            info: 'The CLB listener is not configured for secure (HTTPs or SSL) protocols for client communication.',
            explanation: 'The HTTPs or SSL protocols enable secure communication by encrypting the communication between the client and the load balancer.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elb_1.ELBTlsHttpsListenersOnly,
            node: node,
        });
    }
    /**
     * Check Storage Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkStorage(node) {
        this.applyRule({
            ruleSuffixOverride: 'S1',
            info: 'The S3 Bucket has server access logs disabled.',
            explanation: 'The bucket should have server access logging enabled to provide detailed records for the requests that are made to the bucket.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLoggingEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S2',
            info: 'The S3 Bucket does not have public access restricted and blocked.',
            explanation: 'The bucket should have public access restricted and blocked to prevent unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketLevelPublicAccessProhibited,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S5',
            info: 'The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.',
            explanation: 'An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3WebBucketOAIAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'S10',
            info: 'The S3 Bucket or bucket policy does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: s3_1.S3BucketSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EFS1',
            info: 'The EFS is not configured for encryption at rest.',
            explanation: 'By using an encrypted file system, data and metadata are automatically encrypted before being written to the file system. Similarly, as data and metadata are read, they are automatically decrypted before being presented to the application. These processes are handled transparently by EFS without requiring modification of applications.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: efs_1.EFSEncrypted,
            node: node,
        });
    }
    /**
     * Check Database Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDatabases(node) {
        this.applyRule({
            ruleSuffixOverride: 'RDS2',
            info: 'The RDS instance or Aurora DB cluster does not have storage encryption enabled.',
            explanation: 'Storage encryption helps protect data-at-rest by encrypting the underlying storage, automated backups, read replicas, and snapshots for the database.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSStorageEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS3',
            info: 'The non-Aurora RDS DB instance does not have multi-AZ support enabled.',
            explanation: 'Use multi-AZ deployment configurations for high availability and automatic failover support fully managed by AWS.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSMultiAZSupport,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS6',
            info: 'The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLPostgresIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS8',
            info: 'The RDS DB Security Group allows for 0.0.0.0/0 inbound access.',
            explanation: 'RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSRestrictedInbound,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS10',
            info: 'The RDS instance or Aurora DB cluster does not have deletion protection enabled.',
            explanation: 'Enabling Deletion Protection at the cluster level for Amazon Aurora databases or instance level for non Aurora instances helps protect from accidental deletion.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceDeletionProtectionEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS11',
            info: 'The RDS instance or Aurora DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS13',
            info: 'The RDS instance is not configured for automated backups.',
            explanation: 'Automated backups allow for point-in-time recovery.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.RDSInstanceBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS14',
            info: 'The RDS Aurora MySQL cluster does not have Backtrack enabled.',
            explanation: 'Backtrack helps order to rewind cluster tables to a specific time, without using backups.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLBacktrack,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RDS16',
            info: 'The RDS Aurora MySQL serverless cluster does not have audit, error, general, and slowquery Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::audit'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: rds_1.AuroraMySQLLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB3',
            info: 'The DynamoDB table does not have Point-in-time Recovery enabled.',
            explanation: 'DynamoDB continuous backups represent an additional layer of insurance against accidental loss of data on top of on-demand backups. The DynamoDB service can back up the data with per-second granularity and restore it to any single second from the time PITR was enabled up to the prior 35 days.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: dynamodb_1.DynamoDBPITREnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DDB4',
            info: 'The DAX cluster does not have server-side encryption enabled.',
            explanation: 'Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: dynamodb_1.DAXEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC1',
            info: 'The ElastiCache cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC3',
            info: 'The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.',
            explanation: 'Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC4',
            info: 'The ElastiCache Redis cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC5',
            info: 'The ElastiCache cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AEC6',
            info: 'The ElastiCache Redis cluster does not use Redis AUTH for user authentication.',
            explanation: 'Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: elasticache_1.ElastiCacheRedisClusterRedisAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N1',
            info: 'The Neptune DB cluster is not deployed in a Multi-AZ configuration.',
            explanation: 'The cluster should use a Multi-AZ deployment configuration for high availability.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterMultiAZ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N2',
            info: 'The Neptune DB instance does not have Auto Minor Version Upgrade enabled.',
            explanation: 'The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterAutomaticMinorVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N3',
            info: 'The Neptune DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N4',
            info: 'The Neptune DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'N5',
            info: 'The Neptune DB cluster does not have IAM Database Authentication enabled.',
            explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: neptune_1.NeptuneClusterIAMAuth,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS1',
            info: 'The Redshift cluster does not require TLS/SSL encryption.',
            explanation: 'Enabling the "require_ssl" parameter secures data-in-transit by encrypting the connection between the clients and the Redshift clusters.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftRequireTlsSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS2',
            info: 'The Redshift cluster is not provisioned in a VPC.',
            explanation: 'Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS3',
            info: 'The Redshift cluster uses the default "awsuser" username.',
            explanation: 'Using a custom master user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultUsername,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS4',
            info: 'The Redshift cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS5',
            info: 'The Redshift cluster does not have audit logging enabled.',
            explanation: 'Audit logging helps operators troubleshoot issues and ensure security.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterAuditLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS6',
            info: 'The Redshift cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS8',
            info: 'The Redshift cluster is publicly accessible.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterPublicAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS9',
            info: 'The Redshift cluster does not have version upgrade enabled.',
            explanation: 'Version Upgrade must enabled to enable the cluster to automatically receive upgrades during the maintenance window.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterVersionUpgrade,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS10',
            info: 'The Redshift cluster does not have a retention period for automated snapshots configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A positive retention period should be set to configure this feature.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftBackupEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'RS11',
            info: 'The Redshift cluster does not have user activity logging enabled.',
            explanation: 'User activity logging logs each query before it is performed on the clusters database. To enable this feature associate a Redshift Cluster Parameter Group with the "enable_user_activity_logging" parameter set to "true".',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: redshift_1.RedshiftClusterUserActivityLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC1',
            info: 'The Document DB cluster does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterEncryptionAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC2',
            info: 'The Document DB cluster uses the default endpoint port.',
            explanation: 'Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterNonDefaultPort,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC3',
            info: 'The Document DB cluster does not have the username and password stored in Secrets Manager.',
            explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBCredentialsInSecretsManager,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC4',
            info: 'The Document DB cluster does not have a reasonable minimum backup retention period configured.',
            explanation: 'The retention period represents the number of days to retain automated snapshots. A minimum retention period of 7 days is recommended but can be adjust to meet system requirements.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterBackupRetentionPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'DOC5',
            info: 'The Document DB cluster does not have authenticate, createIndex, and dropCollection Log Exports enabled.',
            explanation: 'This allows operators to use CloudWatch to view logs to help diagnose problems in the database. The events recorded by the AWS DocumentDB audit logs include successful and failed authentication attempts, creating indexes or dropping a collection in a database within the DocumentDB cluster.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: documentdb_1.DocumentDBClusterLogExports,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'TS3',
            info: 'The Timestream database does not use a Customer Managed KMS Key for at rest encryption.',
            explanation: 'All Timestream tables in a database are encrypted at rest by default using AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an annual basis.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: timestream_1.TimestreamDatabaseCustomerManagedKey,
            node: node,
        });
    }
    /**
     * Check Network and Delivery Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkNetworkDelivery(node) {
        this.applyRule({
            ruleSuffixOverride: 'VPC3',
            info: 'A Network ACL or Network ACL entry has been implemented.',
            explanation: 'Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: vpc_1.VPCNoNACLs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'VPC7',
            info: 'The VPC does not have an associated Flow Log.',
            explanation: 'VPC Flow Logs capture network flow information for a VPC, subnet, or network interface and stores it in Amazon CloudWatch Logs. Flow log data can help customers troubleshoot network issues; for example, to diagnose why specific traffic is not reaching an instance, which might be a result of overly restrictive security group rules.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: vpc_1.VPCFlowLogsEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR1',
            info: 'The CloudFront distribution may require Geo restrictions.',
            explanation: 'Geo restriction may need to be enabled for the distribution in order to allow or deny a country in order to allow or restrict users in specific locations from accessing content.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionGeoRestrictions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR2',
            info: 'The CloudFront distribution may require integration with AWS WAF.',
            explanation: 'The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: cloudfront_1.CloudFrontDistributionWAFIntegration,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR3',
            info: 'The CloudFront distribution does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators track all viewer requests for the content delivered through the Content Delivery Network.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR4',
            info: 'The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.',
            explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use the default CloudFront viewer certificate or use 'vip' for the 'SslSupportMethod' are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified 'MinimumProtocolVersion'.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR5',
            info: 'The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.',
            explanation: 'Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionNoOutdatedSSL,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR6',
            info: 'The CloudFront Streaming distribution does not use an origin access identity with an S3 origin.',
            explanation: 'Origin access identities help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessIdentity,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CFR7',
            info: 'The CloudFront distribution does not use an origin access control with an S3 origin.',
            explanation: 'Origin access controls help with security by restricting any direct access to objects through S3 URLs.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloudfront_1.CloudFrontDistributionS3OriginAccessControl,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG1',
            info: 'The API does not have access logging enabled.',
            explanation: 'Enabling access logs helps operators view who accessed an API and how the caller accessed the API.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG2',
            info: 'The REST API does not have request validation enabled.',
            explanation: 'The API should have basic request validation enabled. If the API is integrated with custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWRequestValidation,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG3',
            info: 'The REST API stage is not associated with AWS WAFv2 web ACL.',
            explanation: 'AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: apigw_1.APIGWAssociatedWithWAF,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG4',
            info: 'The API does not implement authorization.',
            explanation: 'In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWAuthorization,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'APIG6',
            info: 'The REST API Stage does not have CloudWatch logging enabled for all methods.',
            explanation: 'Enabling CloudWatch logs at the stage level helps operators to track and analyze execution behavior at the API stage level.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: apigw_1.APIGWExecutionLoggingEnabled,
            node: node,
        });
    }
    /**
     * Check Management and Governance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkManagementGovernance(node) {
        this.applyRule({
            ruleSuffixOverride: 'AS1',
            info: 'The Auto Scaling Group does not have a cooldown period.',
            explanation: 'A cooldown period temporarily suspends any scaling activities in order to allow the newly launched EC2 instance(s) some time to start handling the application traffic.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupCooldownPeriod,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS2',
            info: 'The Auto Scaling Group does not have properly configured health checks.',
            explanation: 'The health check feature enables the service to detect whether its registered EC2 instances are healthy or not.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupHealthCheck,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'AS3',
            info: 'The Auto Scaling Group does not have notifications configured for all scaling events.',
            explanation: 'Notifications on EC2 instance launch, launch error, termination, and termination errors allow operators to gain better insights into systems attributes such as activity and health.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: autoscaling_1.AutoScalingGroupScalingNotifications,
            node: node,
        });
    }
    /**
     * Check Machine Learning Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMachineLearning(node) {
        this.applyRule({
            ruleSuffixOverride: 'LEX4',
            info: 'Lex conversation logs are enabled and not encrypted with a KMS key.',
            explanation: 'You can use encryption to help protect the contents of your conversation logs. For text and audio logs, you can use AWS KMS customer managed CMKs to encrypt data in your CloudWatch Logs log group and S3 bucket.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lex_1.LexBotAliasEncryptedConversationLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM1',
            info: 'The SageMaker notebook instance is not provisioned inside a VPC.',
            explanation: 'Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInVPC,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM2',
            info: 'The SageMaker notebook instance does not have an encrypted storage volume.',
            explanation: 'Encrypting storage volumes helps protect SageMaker data-at-rest.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookInstanceKMSKeyConfigured,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SM3',
            info: 'The SageMaker notebook instance has direct internet access enabled.',
            explanation: 'Disabling public accessibility helps minimize security risks.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sagemaker_1.SageMakerNotebookNoDirectInternetAccess,
            node: node,
        });
    }
    /**
     * Check Analytics Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkAnalytics(node) {
        this.applyRule({
            ruleSuffixOverride: 'EMR2',
            info: 'The EMR cluster does not have S3 logging enabled.',
            explanation: 'Uploading logs to S3 enables the system to keep the logging data for historical purposes or to track and analyze the clusters behavior.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRS3AccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR4',
            info: 'The EMR cluster does not use a security configuration with local disk encryption enabled.',
            explanation: 'Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRLocalDiskEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR5',
            info: 'The EMR cluster does not use a security configuration with encryption in transit enabled and configured.',
            explanation: 'EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: emr_1.EMREncryptionInTransit,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'EMR6',
            info: 'The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.',
            explanation: 'SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: emr_1.EMRAuthEC2KeyPairOrKerberos,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL1',
            info: 'The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.',
            explanation: 'Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: glue_1.GlueEncryptedCloudWatchLogs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'GL3',
            info: 'The Glue job does not use a security configuration with job bookmark encryption enabled.',
            explanation: 'Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: glue_1.GlueJobBookmarkEncrypted,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDA3',
            info: 'The Kinesis Data Analytics Flink Application does not have checkpointing enabled.',
            explanation: 'Checkpoints are backups of application state that KDA automatically creates periodically and uses to restore from faults.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataAnalyticsFlinkCheckpointing,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDF1',
            info: 'The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.',
            explanation: 'This allows the system to meet strict regulatory requirements and enhance the security of system data.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataFirehoseSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS1',
            info: 'The Kinesis Data Stream does not has server-side encryption enabled.',
            explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kinesis_1.KinesisDataStreamSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KDS3',
            info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
            explanation: 'Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: kinesis_1.KinesisDataStreamDefaultKeyWhenSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK2',
            info: 'The MSK cluster uses plaintext communication between clients and brokers.',
            explanation: 'TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKClientToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK3',
            info: 'The MSK cluster uses plaintext communication between brokers.',
            explanation: 'TLS communication secures data-in-transit by encrypting the connection between brokers.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerToBrokerTLS,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MSK6',
            info: 'The MSK cluster does not send broker logs to a supported destination.',
            explanation: 'Broker logs enable operators to troubleshoot Apache Kafka applications and to analyze their communications with the MSK cluster. The cluster can deliver logs to the following resources: a CloudWatch log group, an S3 bucket, a Kinesis Data Firehose delivery stream.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: msk_1.MSKBrokerLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS1',
            info: 'The OpenSearch Service domain is not provisioned inside a VPC.',
            explanation: 'Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchInVPCOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS2',
            info: 'The OpenSearch Service domain does not have node-to-node encryption enabled.',
            explanation: 'Enabling the node-to-node encryption feature adds an extra layer of data protection on top of the existing ES security features such as HTTPS client to cluster encryption and data-at-rest encryption.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNodeToNodeEncryption,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS3',
            info: 'The OpenSearch Service domain does not only grant access via allowlisted IP addresses.',
            explanation: 'Using allowlisted IP addresses helps protect the domain against unauthorized access.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchAllowlistedIPs,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS4',
            info: 'The OpenSearch Service domain does not use dedicated master nodes.',
            explanation: 'Using dedicated master nodes helps improve environmental stability by offloading all the management tasks from the data nodes.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchDedicatedMasterNode,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS5',
            info: 'The OpenSearch Service domain allows for unsigned requests or anonymous access.',
            explanation: 'Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchNoUnsignedOrAnonymousAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS7',
            info: 'The OpenSearch Service domain does not have Zone Awareness enabled.',
            explanation: 'Enabling cross-zone replication (Zone Awareness) increases the availability of the OpenSearch Service domain by allocating the nodes and replicate the data across two AZs in the same region in order to prevent data loss and minimize downtime in the event of node or AZ failure.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchZoneAwareness,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS8',
            info: 'The OpenSearch Service domain does not have encryption at rest enabled.',
            explanation: 'Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchEncryptedAtRest,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'OS9',
            info: 'The OpenSearch Service domain does not minimally publish SEARCH_SLOW_LOGS and INDEX_SLOW_LOGS to CloudWatch Logs.',
            explanation: 'These logs enable operators to gain full insight into the performance of these operations.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'LogExport::<log>' for exported logs. Example: appliesTo: ['LogExport::SEARCH_SLOW_LOGS'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: opensearch_1.OpenSearchSlowLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'QS1',
            info: 'The Quicksight data sources connection is not configured to use SSL.',
            explanation: 'SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: quicksight_1.QuicksightSSLConnections,
            node: node,
        });
    }
    /**
     * Check Security and Compliance Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkSecurityCompliance(node) {
        this.applyRule({
            ruleSuffixOverride: 'IAM4',
            info: 'The IAM user, role, or group uses AWS managed policies.',
            explanation: 'An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies.' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Policy::<policy>' for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoManagedPolicies,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'IAM5',
            info: 'The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.',
            explanation: 'Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. ' +
                "This is a granular rule that returns individual findings that can be suppressed with 'appliesTo'. The findings are in the format 'Action::<action>' for policy actions and 'Resource::<resource>' for resources. Example: appliesTo: ['Action::s3:*'].",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: iam_1.IAMNoWildcardPermissions,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG1',
            info: 'The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.',
            explanation: 'Strong password policies increase system security by encouraging users to create reliable and secure passwords.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolStrongPasswordPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG2',
            info: 'The Cognito user pool does not require MFA.',
            explanation: 'Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: cognito_1.CognitoUserPoolMFA,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG3',
            info: 'The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.',
            explanation: 'Advanced security features enable the system to detect and act upon malicious sign-in attempts.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAdvancedSecurityModeEnforced,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG4',
            info: 'The API GW method does not use a Cognito user pool authorizer.',
            explanation: 'API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolAPIGWAuthorizer,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'COG7',
            info: 'The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.',
            explanation: 'In many cases applications do not warrant unauthenticated guest access applications. Metadata explaining the use case allows for transparency to operators.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cognito_1.CognitoUserPoolNoUnauthenticatedLogins,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'KMS5',
            info: 'The KMS Symmetric key does not have automatic key rotation enabled.',
            explanation: 'KMS key rotation allows a system to set an yearly rotation schedule for a KMS key so when an AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: kms_1.KMSBackingKeyRotationEnabled,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SMG4',
            info: 'The secret does not have automatic rotation scheduled.',
            explanation: 'AWS Secrets Manager can be configured to automatically rotate the secret for a secured service or database.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: secretsmanager_1.SecretsManagerRotationEnabled,
            node: node,
        });
    }
    /**
     * Check Serverless Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkServerless(node) {
        this.applyRule({
            ruleSuffixOverride: 'ASC3',
            info: 'The GraphQL API does not have request level logging enabled.',
            explanation: 'It is important to use CloudWatch Logs to log metrics such as who has accessed the GraphQL API, how the caller accessed the API, and invalid requests.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: appsync_1.AppSyncGraphQLRequestLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF1',
            info: 'The Step Function does not log "ALL" events to CloudWatch Logs.',
            explanation: 'Logging "ALL" events to CloudWatch logs help operators troubleshoot and audit systems.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineAllLogsToCloudWatch,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SF2',
            info: 'The Step Function does not have X-Ray tracing enabled.',
            explanation: 'X-Ray provides an end-to-end view of how an application is performing. This helps operators to discover performance issues, detect permission problems, and track requests made to and from other AWS services.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: stepfunctions_1.StepFunctionStateMachineXray,
            node: node,
        });
    }
    /**
     * Check Application Integration Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkApplicationIntegration(node) {
        this.applyRule({
            ruleSuffixOverride: 'EVB1',
            info: 'The event bus policy allows for open access.',
            explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: eventbridge_1.EventBusOpenAccess,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SNS3',
            info: 'The SNS Topic does not require publishers to use SSL.',
            explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns: Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sns_1.SNSTopicSSLPublishOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS2',
            info: 'The SQS Queue does not have server-side encryption enabled.',
            explanation: 'Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSE,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS3',
            info: 'The SQS queue is not used as a dead-letter queue (DLQ) and does not have a DLQ enabled.',
            explanation: 'Using a DLQ helps maintain the queue flow and avoid losing data by detecting and mitigating failures and service disruptions on time.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueDLQ,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'SQS4',
            info: 'The SQS queue does not require requests to use SSL.',
            explanation: 'Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: sqs_1.SQSQueueSSLRequestsOnly,
            node: node,
        });
    }
    /**
     * Check Media Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkMediaServices(node) {
        this.applyRule({
            ruleSuffixOverride: 'MS1',
            info: 'The MediaStore container does not have container access logging enabled.',
            explanation: 'The container should have access logging enabled to provide detailed records for the requests that are made to the container.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerAccessLogging,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS3',
            info: 'The MediaStore container does not require requests to use SSL.',
            explanation: 'You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: mediastore_1.MediaStoreContainerSSLRequestsOnly,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS4',
            info: 'The MediaStore container does not define a metric policy to send metrics to CloudWatch.',
            explanation: 'Using a combination of MediaStore metrics and CloudWatch alarms helps operators gain better insights into container operations.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreCloudWatchMetricPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS7',
            info: 'The MediaStore container does not define a container policy.',
            explanation: 'Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerHasContainerPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS8',
            info: 'The MediaStore container does not define a CORS policy.',
            explanation: 'Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerCORSPolicy,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'MS10',
            info: 'The MediaStore container does not define a lifecycle policy.',
            explanation: 'Many use cases warrant the usage of lifecycle configurations to manage container objects during their lifetime.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: mediastore_1.MediaStoreContainerLifecyclePolicy,
            node: node,
        });
    }
    /**
     * Check Developer Tools Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkDeveloperTools(node) {
        this.applyRule({
            ruleSuffixOverride: 'CB4',
            info: 'The CodeBuild project does not use an AWS KMS key for encryption.',
            explanation: 'Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: codebuild_1.CodeBuildProjectKMSEncryptedArtifacts,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'CB5',
            info: 'The Codebuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.',
            explanation: 'Explaining differences/edits to Docker images helps operators better understand system dependencies.',
            level: nag_rules_1.NagMessageLevel.WARN,
            rule: codebuild_1.CodeBuildProjectManagedImages,
            node: node,
        });
        this.applyRule({
            ruleSuffixOverride: 'C91',
            info: 'The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.',
            explanation: 'SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: cloud9_1.Cloud9InstanceNoIngressSystemsManager,
            node: node,
        });
    }
    /**
     * Check Lambda Services
     * @param node the CfnResource to check
     * @param ignores list of ignores for the resource
     */
    checkLambda(node) {
        this.applyRule({
            ruleSuffixOverride: 'L1',
            info: 'The non-container Lambda function is not configured to use the latest runtime version.',
            explanation: 'Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.',
            level: nag_rules_1.NagMessageLevel.ERROR,
            rule: lambda_1.LambdaLatestVersion,
            node: node,
        });
    }
}
exports.AwsSolutionsChecks = AwsSolutionsChecks;
_a = JSII_RTTI_SYMBOL_1;
AwsSolutionsChecks[_a] = { fqn: "cdk-nag.AwsSolutionsChecks", version: "2.37.15" };
//# sourceMappingURL=data:application/json;base64,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