"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const nag_rules_1 = require("../../nag-rules");
/**
 * The EC2 Instance requires IMDsv2
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_ec2_1.CfnInstance) {
        if (node.launchTemplate === undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        const instanceLaunchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
        for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
            if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                if (isMatchingLaunchTemplate(child, nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, instanceLaunchTemplate.launchTemplateName), nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, instanceLaunchTemplate.launchTemplateId))) {
                    return hasHttpTokens(child)
                        ? nag_rules_1.NagRuleCompliance.COMPLIANT
                        : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else if (node instanceof aws_autoscaling_1.CfnAutoScalingGroup) {
        if (node.launchTemplate) {
            const nodeLaunchTemplate = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplate);
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_ec2_1.CfnLaunchTemplate) {
                    if (isMatchingLaunchTemplate(child, nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, nodeLaunchTemplate.launchTemplateName), nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, nodeLaunchTemplate.launchTemplateId))) {
                        return hasHttpTokens(child)
                            ? nag_rules_1.NagRuleCompliance.COMPLIANT
                            : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        else if (node.launchConfigurationName) {
            for (const child of aws_cdk_lib_1.Stack.of(node).node.findAll()) {
                if (child instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
                    if (isMatchingLaunchConfiguration(child, node.launchConfigurationName)) {
                        return hasHttpTokens(child)
                            ? nag_rules_1.NagRuleCompliance.COMPLIANT
                            : nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: (0, path_1.parse)(__filename).name });
function isMatchingLaunchTemplate(node, launchTemplateName, launchTemplateId) {
    return (launchTemplateName === node.launchTemplateName ||
        launchTemplateId === nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.ref));
}
function isMatchingLaunchConfiguration(node, launchConfigurationName) {
    return (launchConfigurationName === node.launchConfigurationName ||
        nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, launchConfigurationName) ===
            nag_rules_1.NagRules.resolveResourceFromIntrinsic(node, node.ref));
}
function hasHttpTokens(node) {
    let meta;
    if (node instanceof aws_ec2_1.CfnLaunchTemplate) {
        const launchTemplateData = aws_cdk_lib_1.Stack.of(node).resolve(node.launchTemplateData);
        meta = aws_cdk_lib_1.Stack.of(node).resolve(launchTemplateData.metadataOptions);
    }
    else if (node instanceof aws_autoscaling_1.CfnLaunchConfiguration) {
        meta = aws_cdk_lib_1.Stack.of(node).resolve(node.metadataOptions);
    }
    return meta?.httpTokens === 'required';
}
//# sourceMappingURL=data:application/json;base64,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