"""
Pydantic models for Cosmos DB prompt data validation and serialization.
"""

from typing import Optional, Dict, Any
from pydantic import BaseModel, Field

class PromptModel(BaseModel):
    """
    Pydantic model representing a Cosmos DB prompt document.

    Fields:
        id: Unique identifier for the prompt (same as prompt_name)
        name: Name of the prompt
        prompt_template: The actual prompt template content
        description: Optional description of the prompt
        version: Optional version string
        timestamp: Optional UTC timestamp
    """
    id: str = Field(..., alias="id")
    name: str = Field(..., alias="name")
    prompt_template: str
    description: Optional[str] = "Autogenerated prompt"
    version: Optional[str] = "1.0.0"
    timestamp: Optional[str]

    class Config:
        # Allow extra fields that might be present in Cosmos DB documents
        extra = 'allow'
        # Use the field names as they appear in Cosmos DB (with aliases)
        allow_population_by_field_name = True

    def to_dict(self) -> Dict[str, Any]:
        """Convert the model to a dictionary suitable for Cosmos DB storage."""
        return self.dict(by_alias=True)

    @classmethod
    def from_cosmos_doc(cls, doc: Dict[str, Any]) -> 'PromptModel':
        """
        Create a PromptModel instance from a raw Cosmos DB document.

        Args:
            doc: Raw dictionary from Cosmos DB

        Returns:
            PromptModel instance
        """
        # Ensure required fields exist
        if not doc.get('id') or not doc.get('prompt_template'):
            raise ValueError("Invalid prompt document: missing required fields")

        return cls(
            id=doc['id'],
            name=doc['id'],  # id and name are the same in our schema
            prompt_template=doc['prompt_template'],
            description=doc.get('description'),
            version=doc.get('version'),
            timestamp=doc.get('timestamp')
        )