import pytest
from tortitle.torsubtitle import TorSubtitle, parse_subtitle


test_cases_for_title_parsing = [
    ("伊甸深宅 | 类别：恐怖", "伊甸深宅"),
    ("夜樱家的大作战 | 全27集 | 4K 高码 | 类型: 剧情/动作/动画 | 导演: 凑未来 | 主演: 川岛零士/本渡枫/小西克幸/鬼头明里/兴津和幸", "夜樱家的大作战"),
    ("坏蛋联盟2 / 大坏蛋2 / 坏家伙2 | 类别：喜剧 动作 动画 悬疑 家庭 犯罪 冒险", "坏蛋联盟2"),
    ("盾之勇者成名录 / The Rising of the Shield Hero Season 4 第四季 | 第07集 | 1080p | 类型: 动画 | 导演: 垪和等 | 主演: 石川界人/濑户麻沙美/日高里菜/天崎滉平/小原好美 [内嵌中字]", "盾之勇者成名录"),
    ("锦月如歌 / 女将星 / 重生之女将星 | 第32-36集 | 4K 杜比视界 | 类型: 剧情/古装 | 导演: 蒋家骏 | 主演: 周也/丞磊/张康乐/张淼怡/李卿", "锦月如歌"),
    ("官方国语中字 明月入卿怀/师叔你这样很容易失去我 第7-8集 | 4K (IQ)杜比视界 | 国语/简繁英等多国软字幕 | 类型:古装 ..", "明月入卿怀"),
    ("官方中字 生命清单 The Life List | 4K SDR | 类型:剧情/喜剧/爱情 | 内封网飞中字", "生命清单"),
    ("官方中字 美剧：联邦调查局：通缉要犯 第六季 第12集 | 类型:剧情/犯罪 | 内嵌中字", "联邦调查局：通缉要犯"),
    ("暗蚀 Andhera 全8集 | 类型: 剧情 主演: 苏莉温.查瓦拉 / Pranay Pachauri / Priya Bapat / 普拉加克塔·科利", "暗蚀"),
    ("夏目友人帐 第七季 / 夏目友人帐 柒 [中日双语字幕] [01-12Fin+OVA]", "夏目友人帐"),
    ("2025年7月新番 这届侦探真不行 / 最近的侦探真没用 / まったく最近の探偵ときたら 第08集", "这届侦探真不行"),
    ("蝴蝶风暴 Butterfly 全6集 | 类型: 动作 / 惊悚 主演: 派珀·佩拉博 / 金大贤 / 蕾娜·哈迪斯蒂 / 查尔斯·帕内尔 / 金志勋", "蝴蝶风暴"),
    ("生命之树（加长版）[6.8][tt0478304] 第64届戛纳电影节·金棕榈奖 CC#942", "生命之树"),
    ("[法国] 百万法郎【7.3】【tt0022150】【Edward】导演：雷内·克莱尔 | 中字", "百万法郎"),
    ("【致命感应/Malignant Man/恶性/恶毒/恶煞(港)/疾厄(台)/肿瘤/肿瘤侠】转载", "致命感应"),
    ("诱饵新娘 [6.1][tt1657299]", "诱饵新娘"),
    ("【教父/Mario Puzo's The Godfather】mUHD作品 4k 杜比视界版本 国英双语 东影上译/上音内参/央视长译/VCD国配 评论音轨 简英双语特效/国配特效字幕 带黑边", "教父"),
    ("异形：地球 / Alien: Earth 第1季 | 类型：剧情 Sci-Fi & Fantasy | 语言：Afrikaans,English | *内封简繁英多国软字幕* 第1季 第03集", "异形：地球"),
    ("2025年7月新番：克雷瓦提斯-魔兽之王与婴儿与尸之勇者- / Clevatess －魔獸之王與嬰兒與屍之勇者－ / クレバテス-魔獣の王と赤子と屍の勇者 | 类型：动画 动作冒险 Sci-Fi & Fantasy 剧情 | 语言：日本語 | *内嵌繁中* 第1季 第08集", "克雷瓦提斯-魔兽之王与婴儿与尸之勇者-"),
    ("保龄球少女！ / Turkey! 第1季 | 类型：动画 喜剧 | 语言：日本語 | *内嵌繁中* 第1季 第07集", "保龄球少女！"),
    ("ViuTV | 模范的士 2 / Taxi Driver 2 (2023) | 全16集", "模范的士"),
    ("Jade | TVB | 寻秦记", "寻秦记"),
    ("ViuTV | 蓝色时刻/暮色气象英雄/ブルーモーメント | 第1季 第08-09集 | 导演:田中亮 / 森胁智延 | 主演:山下智久..Blue Moment 2024 S01E08-E09 1080i HDTV H264 DD2.0 2Audios-PTerTV", "蓝色时刻"),
    ("HOYTV | 我的完美秘书/打招呼的关系/나의 완벽한 비서 | 第1季 第01-05集 | 导演:咸俊浩 / 金宰弘 | 主演:韩志..Love Scout 2025 S01E01-E05 1080i HDTV H264 DD2.0 2Audios-PTerTV", "我的完美秘书"),
    ("Jade | 麻雀樂團/麻雀乐团 | 第1季 第16-17集 | 导演:罗永贤 | 主演:谭俊彦 / 李佳芯 / 郭柏妍 | [粤语简繁..Heavenly Hand 2025 S01E16-E17 1080i HDTV H264 DD5.1-PTerTV", "麻雀樂團"),
    ("ViuTV | 翻盘下半场/翻盤下半場 | 第一季 第01集 | 导演:叶镇辉 / 錢颖芝 | 主演:黄浩然 / 周家怡 / 欧镇灏 |..", "翻盘下半场"),
    ("官方 泰剧：锁爱于心 第18集 ใจขังเจ้า/Captive Heart/俘虏之心 | 类型:动作/爱情/古装 | 泰语", "锁爱于心"),
    ("韩剧 | 初次，为了爱/First, For Love/첫, 사랑을 위하여/首先，为了爱 | 第06集 |＊内封简繁英软字幕＊| 喜剧..", "初次，为了爱"),
    ("2025 年7月档新番：租借女友 / 彼女、お借りします 第四季 | 类型：动画 喜剧 | 语言：日本語 第1季 第42集", "租借女友"),
    ("2025年 7 月档新番：公爵千金的家庭教师 / 公女殿下の家庭教師 | 类型：动画 Sci-Fi & Fantasy | 语言：日本語 | *内封简繁英多国软字幕* 第1季 第07集", "公爵千金的家庭教师"),
    ("W的悲剧丨导演: 泽井信一郎 主演: 药师丸博子 三田佳子 世良公则 Yumi Kusaka 草薙幸二郎[日语+双国语][简繁日字+章节]", "W的悲剧"),
    ("酱园弄·悬案/酱园弄第一部/酱园弄(上)/酱园弄杀夫案/She Has No Name | 1080p | 国语/中字 | 类型:剧情/犯罪 | 导演:陈可辛/韩帅 | 主演:章子怡/王传君/易烊千玺", "酱园弄·悬案"),
    ("[晶兵总动员/魔幻小战士]mUHD作品 4k 杜比视界版本 国英双语 简繁英双语字幕", "晶兵总动员"),
    ("[纪录] 第二季 全7集[菁彩影像 | 4K高码率 | 高帧率][导演：程工 | 配音讲解：赵丽颖]QHstudIo小组作品", "菁彩影像"),
    ("舌尖上的中国 第一季 | 全7集 | 导演: 陈晓卿 | 主演: 李立宏 国语/中字 4K高码版 | ARDT", "舌尖上的中国"),
    ("国漫: 斗罗大陆Ⅱ绝世唐门 / 斗罗大陆Ⅱ绝世唐门 年番1 / 斗罗大陆2绝世唐门 / 斗罗大陆2：绝世唐门", "斗罗大陆Ⅱ绝世唐门"),
    ("电影 酱园弄·悬案 HDR Vivid", "酱园弄·悬案"),
    ("韩综|只要有空 第三季/틈만나면 시즌3 |全15集|真人秀|刘在锡 /柳演锡/车胜元/孔明＊内封简繁英软字幕＊", "只要有空"),
    ("2025年7月新番 Silent Witch 沉默魔女的秘密 / サイレント・ウィッチ 沈黙の魔女の隠しごと 第07集 *内封简繁英多国软字幕*", "沉默魔女的秘密"),
    ("国语中字 我是传奇 / 抗日奇侠2我是传奇 | 国语中字 | 全30集 | 类型：剧情 | 演员：朱雨辰 / 吴樾 / 周知 / 斯", "我是传奇"),
    ("DIY国语中字 我是传奇/末日传奇/魔间传奇(港) 加长剧场二合一 国配AC3+DIY简繁字幕", "我是传奇"),
    ("国语中字特效 【我是传奇/末日传奇/魔间传奇】国英双语 MNHD作品 1080p HDR10版本 国配/中英特效字幕", "我是传奇"),
    ("官方国语中字限转特效 晶兵总动员 h3", "晶兵总动员"),
    ("DIY中字", ""),
    ("悠哈璃羽字幕社 简繁外挂 | 裤袜视界 / 赏袜 / 丝袜视界 / 从裤袜下看 / 黑丝袜风情 / みるタイツ", "裤袜视界"),
    ("遗传学歌剧 / Repo, The Genetic Opera! / 莎拉布莱曼之生化歌剧 / 基因歌剧之透支", "遗传学歌剧"),
    ("[美国/法国][机械师2：复活][Mechanic.Resurrection.2016.720p.BluRay.x264.DD5.1-HDChina]", "机械师2：复活"),
    ("主演: 石原里美 / 峯田和伸 / 芳根京子 / 千叶雄大 FIX字幕侠内嵌中文版", ""),
    ("(新)黑寡妇/斯嘉丽·约翰逊 弗洛伦丝·皮尤 蕾切尔·薇兹 *国英双语 *官译简繁英字幕", "黑寡妇"),
    ("全1集 粤语 | 繁体DvbSub字幕", ""),
    ("官方 首发 疾速追杀：芭蕾杀姬/捍卫任务：复仇芭蕾(台)/杀神John Wick外传：芭蕾杀姬(港) h5", "疾速追杀：芭蕾杀姬"),
    ("[国创] 兵主奇魂 全24集 ｜ 类型: 剧情 / 动画 / 奇幻 / 冒险 *含B站水印*", "兵主奇魂"),
    ("官方 Kill Tony: Mayhem at Madison Square Garden | 类型:真人秀 | 英语", "Kill Tony: Mayhem at Madison Square Garden"),
    ("官方独占 港劇:愛.回家之開心速遞(第2626集)[粤语][簡繁DVB字幕][劉丹/單立文/湯盈盈/呂慧儀 主演][CHDHKTV", "港劇:愛.回家之開心速遞"),
    ("官方独占中字特效 超人 [KKYY] [内封简英+繁英SDR特效字幕] h3", "超人"),
    ("国语/中字", ""),
    ("纪录片 我们的浩瀚宇宙 全6集 *4K+HDR10+Dolby Atmos* *内封中字*", "我们的浩瀚宇宙"),
    ("Evening of Roses / Night Rose | 导演：胡玥 / 吴思远 | 演员：林心如 / 钟汉良 / 马天宇 / 倪虹洁 / 贾乃亮 | [汉语普通话]", "Evening of Roses"),
    ("[DIY]官译简繁+简英繁英双语字幕*DTS:X临境音效*", ""),
    ("Lord of Mysteries | 诡秘之主 小丑卷 第10集", "诡秘之主" ),
    ("[Checked by maomao] 在远方 第一季 全54集 | 类型:剧情/爱情 主演:刘烨/马伊琍/梅婷 [国语/中字] | *4K* ", "在远方"),
    ("[Checked by Ticutme] 鹊刀门传奇 全40集【杜比音效】【主演: 赵本山 | 宋小宝 | 宋晓峰 | 文松 | 程野】QHstudIo小组作品", "鹊刀门传奇"),
    ("官方国语中字 [2023年9月国漫] 仙逆 第103集", "仙逆"),
    ("官方国语中字 [短剧] 匿爱 全23集 | 4K 高码 | 国语/中字 | 类型:剧情/爱情/短片 | 主演:施南 | *第15/16集..", "匿爱"),
    ("【台湾(区)】零日攻击 | 国语中字", "零日攻击"),
    ("【韩国/南韩】我的完美秘书 | 国语中字", "我的完美秘书"),
    ("驱魔 / Exorcisms / L‘éventreur de Notre-Dame | 类别：惊悚 恐怖", "驱魔"),
    ("4K 杜比视界 | 类型: 剧情/惊悚/犯罪 | 导演: 泰勒·派瑞 | 主演: 凯莉·罗兰/崔凡特·罗兹/肖恩·萨加尔/RonReacoLee/凯莉·奥马利", ""), 
    ("[泰剧] 明星的重生 第一季 全10集 / 巨星重生 / การเกิดใหม่ของดวงดาว / ResetTheSeries |..", "明星的重生"),
    ("[国漫连载] 我的师兄太强了 / 我的师兄太强了之重见天明篇 / 我的师兄太强了之风起云涌篇 / 我的师兄太强了之拨云见日篇 / 我的师兄太强了之群雄逐鹿篇 / 我的师兄太强了之波云诡谲篇 第48集 4K [bilibili大陆]", "我的师兄太强了"),
    ("日本沉没 / 日本沉没2020 / 日本沉没：希望之人 / Nihon Chinbotsu 2020 / Japan Sinks: 2020 | 全10集 | 4K HDR10+", "日本沉没"),
    ("【果宝特攻 第一季 全52集 *部分720p* | 类型：喜剧 动作 动画 儿童 奇幻 冒险 | 主演：王巍 陆双 邓红 祖晴 赵然 *云视听极光*】", "果宝特攻"),
    ("新·驯龙高手/驯龙记(港) | DiY国粤双语配音 | 官译简繁+简英繁英双语字幕", "新·驯龙高手"),
    ("港剧: 经典台 雪山飞狐 [粤语/简体字幕]", "雪山飞狐"),
]   


@pytest.mark.parametrize("case, expected", test_cases_for_title_parsing)
def test_parse_title_from_raw_name(case, expected):
    assert parse_subtitle(case) == expected


test_cases_for_season_and_episode_parsing = {
    "盾之勇者成名录 第四季 | 第07集": {"season": 4, "episode": "E07", "extitle": "盾之勇者成名录"},
    "锦月如歌 | 第32-36集": {"season": "", "episode": "E32-E36", "extitle": "锦月如歌"},
    "明月入卿怀 第7-8集": {"season": "", "episode": "E07-E08", "extitle": "明月入卿怀"},
    "美剧：联邦调查局：通缉要犯 第六季 第12集": {"season": 6, "episode": "E12", "extitle": "联邦调查局：通缉要犯"},
    "暗蚀 Andhera 全8集": {"season": "", "episode": "", "total_episodes": 8, "extitle": "暗蚀"},
    "夏目友人帐 第七季": {"season": 7, "episode": "", "extitle": "夏目友人帐"},
    "异形：地球 / Alien: Earth 第1季 第03集": {"season": 1, "episode": "E03", "extitle": "异形：地球"},
    "酱园弄·悬案/酱园弄第一部/酱园弄(上)/酱园弄杀夫案/She Has No Name | 1080p | 国语/中字 | 类型:剧情/犯罪 | 导演:陈可辛/韩帅 | 主演:章子怡/王传君/易烊千玺": 
    {"season": "", "episode": "", "extitle": "酱园弄·悬案"}
}


@pytest.mark.parametrize("name, expected", test_cases_for_season_and_episode_parsing.items())
def test_season_and_episode_parsing(name, expected):
    parser = TorSubtitle(name)
    assert parser.season == expected["season"]
    assert parser.episode == expected["episode"]
    assert parser.extitle == expected["extitle"]
    if "total_episodes" in expected:
        assert parser.total_episodes == expected["total_episodes"]
