"""
Copyright 2023-2023 VMware Inc.
SPDX-License-Identifier: Apache-2.0

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

import click
from vhcs.ctxp import profile
from vhcs.service.pki import certificate
from vhcs.util import pki_util
import vhcs.sglib.cli_options as cli


def _write_file(file_path: str, text: str):
    with open(file_path, "w") as file:
        file.write(text)


@click.command()
@click.argument("resource-name", type=str, required=True)
@click.option(
    "--key-length",
    type=int,
    default=2048,
    required=False,
    help="Private key length",
)
@cli.org_id
def sign_resource_cert(resource_name: str, key_length: int, org: str):
    """Get the certificate for a specific resource"""
    org_id = cli.get_org_id(org)
    csr_pem, private_key_pem = pki_util.generate_CSR(resource_name, key_length=key_length)
    ret = certificate.sign_resource_cert(org_id, csr_pem)

    long_name = f"{profile.name()}-{org_id}-{resource_name}"
    key_file = resource_name + ".key"
    print("Private key (generated by CLI): " + key_file)
    _write_file(key_file, private_key_pem)

    client_cert_chain_file = long_name + ".crt"
    print("Signed resource certificate with chain (response from server): " + client_cert_chain_file)
    _write_file(client_cert_chain_file, ret)
