"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlarmFactory = exports.CompositeAlarmOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const metric_1 = require("../metric");
const strings_1 = require("../strings");
const AlarmNamingStrategy_1 = require("./AlarmNamingStrategy");
const IAlarmAnnotationStrategy_1 = require("./IAlarmAnnotationStrategy");
const NoopAlarmActionStrategy_1 = require("./NoopAlarmActionStrategy");
const DefaultDatapointsToAlarm = 3;
var CompositeAlarmOperator;
(function (CompositeAlarmOperator) {
    /**
     * trigger only if all the alarms are triggered
     */
    CompositeAlarmOperator[CompositeAlarmOperator["AND"] = 0] = "AND";
    /**
     * trigger if any of the alarms is triggered
     */
    CompositeAlarmOperator[CompositeAlarmOperator["OR"] = 1] = "OR";
})(CompositeAlarmOperator = exports.CompositeAlarmOperator || (exports.CompositeAlarmOperator = {}));
class AlarmFactory {
    constructor(alarmScope, props) {
        this.alarmScope = alarmScope;
        this.globalAlarmDefaults = props.globalAlarmDefaults;
        this.globalMetricDefaults = props.globalMetricDefaults;
        this.alarmNamingStrategy = new AlarmNamingStrategy_1.AlarmNamingStrategy(props.globalAlarmDefaults.alarmNamePrefix, props.localAlarmNamePrefix, props.globalAlarmDefaults.dedupeStringProcessor);
    }
    addAlarm(metric, props) {
        // prepare the metric
        let adjustedMetric = metric;
        if (props.period) {
            // Adjust metric period for the alarm
            adjustedMetric = adjustedMetric.with({ period: props.period });
        }
        if (adjustedMetric.label) {
            // Annotations do not support dynamic labels, so we have to remove them from metric name
            adjustedMetric = adjustedMetric.with({
                label: strings_1.removeBracketsWithDynamicLabels(adjustedMetric.label),
            });
        }
        // prepare primary alarm properties
        const actionsEnabled = this.determineActionsEnabled(props.actionsEnabled, props.disambiguator);
        const action = this.determineAction(props.disambiguator, props.actionOverride);
        const alarmName = this.alarmNamingStrategy.getName(props);
        const alarmNameSuffix = props.alarmNameSuffix;
        const alarmLabel = this.alarmNamingStrategy.getWidgetLabel(props);
        const alarmDescription = this.generateDescription(props.alarmDescription, props.alarmDescriptionOverride, props.runbookLink, props.documentationLink);
        const dedupeString = this.alarmNamingStrategy.getDedupeString(props);
        const evaluateLowSampleCountPercentile = props.evaluateLowSampleCountPercentile ?? true;
        const datapointsToAlarm = props.datapointsToAlarm ??
            this.globalAlarmDefaults.datapointsToAlarm ??
            DefaultDatapointsToAlarm;
        const evaluationPeriods = props.evaluationPeriods ??
            this.globalAlarmDefaults.evaluationPeriods ??
            datapointsToAlarm;
        if (evaluationPeriods < datapointsToAlarm) {
            throw new Error(`evaluationPeriods must be greater than or equal to datapointsToAlarm for ${alarmName}`);
        }
        // create primary alarm
        const primaryAlarm = adjustedMetric.createAlarm(this.alarmScope, alarmName, {
            alarmName,
            alarmDescription,
            threshold: props.threshold,
            comparisonOperator: props.comparisonOperator,
            treatMissingData: props.treatMissingData,
            // default value (undefined) means "evaluate"
            evaluateLowSampleCountPercentile: evaluateLowSampleCountPercentile
                ? undefined
                : "ignore",
            datapointsToAlarm,
            evaluationPeriods,
            actionsEnabled,
        });
        let alarm = primaryAlarm;
        // create composite alarm for min metric samples (if defined)
        if (props.minMetricSamplesToAlarm) {
            const metricSampleCount = adjustedMetric.with({
                statistic: metric_1.MetricStatistic.N,
            });
            const noSamplesAlarm = metricSampleCount.createAlarm(this.alarmScope, `${alarmName}-NoSamples`, {
                alarmName: `${alarmName}-NoSamples`,
                alarmDescription: `The metric (${adjustedMetric}) does not have enough samples to alarm. Must have at least ${props.minMetricSamplesToAlarm}.`,
                threshold: props.minMetricSamplesToAlarm,
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
                treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
                datapointsToAlarm: 1,
                evaluationPeriods: 1,
                actionsEnabled,
            });
            alarm = new aws_cloudwatch_1.CompositeAlarm(this.alarmScope, `${alarmName}-WithSamples`, {
                actionsEnabled,
                compositeAlarmName: `${alarmName}-WithSamples`,
                alarmDescription: this.joinDescriptionParts(alarmDescription, `Min number of samples to alarm: ${props.minMetricSamplesToAlarm}`),
                alarmRule: aws_cloudwatch_1.AlarmRule.allOf(aws_cloudwatch_1.AlarmRule.fromAlarm(primaryAlarm, aws_cloudwatch_1.AlarmState.ALARM), aws_cloudwatch_1.AlarmRule.not(aws_cloudwatch_1.AlarmRule.fromAlarm(noSamplesAlarm, aws_cloudwatch_1.AlarmState.ALARM))),
            });
        }
        // attach alarm actions
        action.addAlarmActions({
            alarm,
            action,
            dedupeString,
            disambiguator: props.disambiguator,
            customTags: props.customTags ?? [],
            customParams: props.customParams ?? {},
        });
        // create annotation for the primary alarm
        const annotation = this.createAnnotation({
            alarm: primaryAlarm,
            action,
            metric: adjustedMetric,
            evaluationPeriods,
            datapointsToAlarm,
            dedupeString,
            minMetricSamplesToAlarm: props.minMetricSamplesToAlarm,
            fillAlarmRange: props.fillAlarmRange ?? false,
            overrideAnnotationColor: props.overrideAnnotationColor,
            overrideAnnotationLabel: props.overrideAnnotationLabel,
            overrideAnnotationVisibility: props.overrideAnnotationVisibility,
            comparisonOperator: props.comparisonOperator,
            threshold: props.threshold,
            disambiguator: props.disambiguator,
            customTags: props.customTags ?? [],
            customParams: props.customParams ?? {},
        });
        // return the final result
        return {
            alarm,
            action,
            alarmName,
            alarmNameSuffix,
            alarmLabel,
            alarmDescription,
            customTags: props.customTags,
            customParams: props.customParams,
            alarmRuleWhenOk: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.OK),
            alarmRuleWhenAlarming: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.ALARM),
            alarmRuleWhenInsufficientData: aws_cloudwatch_1.AlarmRule.fromAlarm(alarm, aws_cloudwatch_1.AlarmState.INSUFFICIENT_DATA),
            dedupeString,
            annotation,
            disambiguator: props.disambiguator,
        };
    }
    addCompositeAlarm(alarms, props) {
        const actionsEnabled = this.determineActionsEnabled(props?.actionsEnabled, props?.disambiguator);
        const action = props.actionOverride ?? this.globalAlarmDefaults.action ?? NoopAlarmActionStrategy_1.noopAction();
        const namingInput = { alarmNameSuffix: "Composite", ...props };
        const alarmName = this.alarmNamingStrategy.getName(namingInput);
        const alarmDescription = this.generateDescription(props?.alarmDescription ?? "Composite alarm", props?.alarmDescriptionOverride, props?.runbookLink, props?.documentationLink);
        const dedupeString = this.alarmNamingStrategy.getDedupeString(namingInput);
        const alarmRule = this.determineCompositeAlarmRule(alarms, props);
        const alarm = new aws_cloudwatch_1.CompositeAlarm(this.alarmScope, alarmName, {
            compositeAlarmName: alarmName,
            alarmDescription,
            alarmRule,
            actionsEnabled,
        });
        action.addAlarmActions({
            alarm,
            action,
            dedupeString,
            disambiguator: props?.disambiguator,
            customTags: props?.customTags,
            customParams: props?.customParams,
        });
        return alarm;
    }
    determineCompositeAlarmRule(alarms, props) {
        const alarmRules = alarms.map((alarm) => alarm.alarmRuleWhenAlarming);
        const operator = props.compositeOperator ?? CompositeAlarmOperator.OR;
        switch (operator) {
            case CompositeAlarmOperator.AND:
                return aws_cloudwatch_1.AlarmRule.allOf(...alarmRules);
            case CompositeAlarmOperator.OR:
                return aws_cloudwatch_1.AlarmRule.anyOf(...alarmRules);
            default:
                throw new Error(`Unsupported composite alarm operator: ${operator}`);
        }
    }
    determineActionsEnabled(actionsEnabled, disambiguator) {
        if (actionsEnabled !== undefined) {
            // alarm-specific override to true or false
            return actionsEnabled;
        }
        if (typeof this.globalAlarmDefaults.actionsEnabled === "boolean") {
            // global defaults: boolean value
            return this.globalAlarmDefaults.actionsEnabled;
        }
        if (disambiguator) {
            // global defaults: disambiguator based value
            return this.globalAlarmDefaults.actionsEnabled[disambiguator] ?? false;
        }
        return false;
    }
    determineAction(disambiguator, actionOverride) {
        // Explicit override
        if (actionOverride) {
            return actionOverride;
        }
        // Default by disambiugator
        if (disambiguator &&
            this.globalAlarmDefaults.disambiguatorAction?.[disambiguator]) {
            return this.globalAlarmDefaults.disambiguatorAction[disambiguator];
        }
        // Default global action
        return this.globalAlarmDefaults.action ?? NoopAlarmActionStrategy_1.noopAction();
    }
    get shouldUseDefaultDedupeForError() {
        return this.globalAlarmDefaults.useDefaultDedupeForError ?? true;
    }
    get shouldUseDefaultDedupeForLatency() {
        return this.globalAlarmDefaults.useDefaultDedupeForLatency ?? true;
    }
    generateDescription(alarmDescription, alarmDescriptionOverride, runbookLinkOverride, documentationLinkOverride) {
        const parts = [alarmDescriptionOverride ?? alarmDescription];
        const runbookLink = runbookLinkOverride ?? this.globalAlarmDefaults.runbookLink;
        const documentationLink = documentationLinkOverride ?? this.globalAlarmDefaults.documentationLink;
        if (runbookLink) {
            parts.push(`Runbook: ${runbookLink}`);
        }
        if (documentationLink) {
            parts.push(`Documentation: ${documentationLink}`);
        }
        return this.joinDescriptionParts(...parts);
    }
    joinDescriptionParts(...parts) {
        return parts.join(" \r\n");
    }
    createAnnotation(props) {
        const strategy = this.globalAlarmDefaults.annotationStrategy ??
            new IAlarmAnnotationStrategy_1.DefaultAlarmAnnotationStrategy();
        return strategy.createAnnotation(props);
    }
}
exports.AlarmFactory = AlarmFactory;
_a = JSII_RTTI_SYMBOL_1;
AlarmFactory[_a] = { fqn: "cdk-monitoring-constructs.AlarmFactory", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,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