"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsConsoleUrlFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const monitoring_1 = require("../../monitoring");
class AwsConsoleUrlFactory {
    constructor(props) {
        this.awsAccountId = props.awsAccountId;
        this.awsAccountRegion = props.awsAccountRegion;
    }
    getAwsConsoleUrl(destinationUrl) {
        if (this.awsAccountId) {
            return aws_cdk_lib_1.Lazy.uncachedString({
                produce: (context) => {
                    if (destinationUrl) {
                        return this.getResolvedDestinationUrl(context, destinationUrl);
                    }
                    // simply go to the home page of the AWS console
                    return `https://${this.awsAccountRegion}.console.aws.amazon.com`;
                },
            });
        }
        return undefined;
    }
    getCloudWatchLogGroupUrl(logGroupName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/cloudwatch/home?region=${region}#logsV2:log-groups/log-group/${logGroupName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getCodeBuildProjectUrl(projectName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/codebuild/home?region=${region}#/projects/${projectName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getSnsTopicUrl(topicArn) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/sns/v3/home?region=${region}#/topic/${topicArn}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getSqsQueueUrl(queueUrl) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/sqs/v2/home?region=${region}#/queues/${queueUrl}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getLambdaFunctionUrl(functionName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/lambda/home?region=${region}#/functions/${functionName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getDynamoTableUrl(tableName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/dynamodb/home?region=${region}#tables:selected=${tableName}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getStateMachineUrl(stateMachineArn) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/states/home?region=${region}#/statemachines/view/${stateMachineArn}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisDataStreamUrl(streamName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/kinesis/home?region=${region}#/streams/details/${streamName}/monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisFirehoseDeliveryStreamUrl(streamName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/firehose/home?region=${region}#/details/${streamName}/monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getKinesisAnalyticsUrl(application) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/kinesisanalytics/home?region=${region}#/details?applicationName=${application}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getS3BucketUrl(bucketName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://s3.console.aws.amazon.com/s3/buckets/${bucketName}?region=${region}&tab=metrics`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getApiGatewayUrl(restApiId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/apigateway/home?region=${region}#/apis/${restApiId}/dashboard`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getRdsClusterUrl(clusterId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/rds/home?region=${region}#database:id=${clusterId};is-cluster=true;tab=monitoring`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getRedshiftClusterUrl(clusterId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/redshiftv2/home?region=${region}#cluster-details?cluster=${clusterId}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getOpenSearchClusterUrl(domainName) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/es/home?region=${region}#domain:resource=${domainName};action=dashboard;tab=TAB_CLUSTER_HEALTH_ID_V2`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getCloudFrontDistributionUrl(distributionId) {
        const destinationUrl = `https://console.aws.amazon.com/cloudfront/v2/home#/monitoring/${distributionId}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    getElastiCacheClusterUrl(clusterId, clusterType) {
        const region = this.awsAccountRegion;
        switch (clusterType) {
            case monitoring_1.ElastiCacheClusterType.MEMCACHED:
                const destinationUrlMemcached = `https://${region}.console.aws.amazon.com/elasticache/home?region=${region}#memcached-nodes:id=${clusterId};nodes`;
                return this.getAwsConsoleUrl(destinationUrlMemcached);
            case monitoring_1.ElastiCacheClusterType.REDIS:
                const destinationUrlRedis = `https://${region}.console.aws.amazon.com/elasticache/home?region=${region}#redis-shards:redis-id=${clusterId}`;
                return this.getAwsConsoleUrl(destinationUrlRedis);
            default:
                throw new Error(`Invalid cache type: ${clusterType}`);
        }
    }
    getDocumentDbClusterUrl(clusterId) {
        const region = this.awsAccountRegion;
        const destinationUrl = `https://${region}.console.aws.amazon.com/docdb/home?region=${region}#cluster-details/${clusterId}`;
        return this.getAwsConsoleUrl(destinationUrl);
    }
    /**
     * Resolves a destination URL within a resolution context.
     * @param context The resolution context.
     * @param destinationUrl The destination URL to resolve since it may contain CDK tokens.
     * @see https://docs.aws.amazon.com/cdk/latest/guide/tokens.html
     */
    getResolvedDestinationUrl(context, destinationUrl) {
        let resolvedDestination = context.resolve(destinationUrl);
        if (typeof resolvedDestination !== "string") {
            // keep unresolved since we do not know how to deal with objects
            resolvedDestination = destinationUrl;
        }
        return resolvedDestination;
    }
}
exports.AwsConsoleUrlFactory = AwsConsoleUrlFactory;
_a = JSII_RTTI_SYMBOL_1;
AwsConsoleUrlFactory[_a] = { fqn: "cdk-monitoring-constructs.AwsConsoleUrlFactory", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,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