"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayMonitoring = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const common_1 = require("../../common");
const dashboard_1 = require("../../dashboard");
const ApiGatewayMetricFactory_1 = require("./ApiGatewayMetricFactory");
const DefaultLatencyTypesToRender = [
    common_1.LatencyType.P50,
    common_1.LatencyType.P90,
    common_1.LatencyType.P99,
];
class ApiGatewayMonitoring extends common_1.Monitoring {
    constructor(scope, props) {
        super(scope, props);
        // used when humanReadableName is not provided by user
        const fallbackNameArray = [props.api.restApiName];
        fallbackNameArray.push(props.apiStage ?? "prod");
        if (props.apiMethod) {
            fallbackNameArray.push(props.apiMethod);
        }
        if (props.apiResource) {
            fallbackNameArray.push(props.apiResource);
        }
        const namingStrategy = new dashboard_1.MonitoringNamingStrategy({
            ...props,
            namedConstruct: props.api,
            fallbackConstructName: fallbackNameArray
                .join("-")
                .replace(/[^a-zA-Z0-9-_]/g, ""),
            humanReadableName: props.humanReadableName ?? fallbackNameArray.join(" "),
        });
        this.title = namingStrategy.resolveHumanReadableName();
        this.alarmFactory = this.createAlarmFactory(namingStrategy.resolveAlarmFriendlyName());
        this.errorAlarmFactory = new common_1.ErrorAlarmFactory(this.alarmFactory);
        this.tpsAlarmFactory = new common_1.TpsAlarmFactory(this.alarmFactory);
        this.latencyAlarmFactory = new common_1.LatencyAlarmFactory(this.alarmFactory);
        this.tpsAnnotations = [];
        this.latencyAnnotations = [];
        this.errorCountAnnotations = [];
        this.errorRateAnnotations = [];
        const metricFactory = new ApiGatewayMetricFactory_1.ApiGatewayMetricFactory(scope.createMetricFactory(), props);
        this.tpsMetric = metricFactory.metricTps();
        this.latencyMetrics = {};
        this.latencyTypesToRender = [
            ...(props.latencyTypesToRender ?? DefaultLatencyTypesToRender),
        ];
        this.error4XXCountMetric = metricFactory.metric4XXErrorCount();
        this.error4XXRateMetric = metricFactory.metric4XXErrorRate();
        this.fault5XXCountMetric = metricFactory.metric5XXFaultCount();
        this.fault5XXRateMetric = metricFactory.metric5XXFaultRate();
        const latencyAlarmDefinitions = {
            [common_1.LatencyType.P50]: props.addLatencyP50Alarm,
            [common_1.LatencyType.P70]: props.addLatencyP70Alarm,
            [common_1.LatencyType.P90]: props.addLatencyP90Alarm,
            [common_1.LatencyType.P95]: props.addLatencyP95Alarm,
            [common_1.LatencyType.P99]: props.addLatencyP99Alarm,
            [common_1.LatencyType.P999]: props.addLatencyP999Alarm,
            [common_1.LatencyType.P9999]: props.addLatencyP9999Alarm,
            [common_1.LatencyType.P100]: props.addLatencyP100Alarm,
            [common_1.LatencyType.TM50]: props.addLatencyTM50Alarm,
            [common_1.LatencyType.TM70]: props.addLatencyTM70Alarm,
            [common_1.LatencyType.TM90]: props.addLatencyTM90Alarm,
            [common_1.LatencyType.TM95]: props.addLatencyTM95Alarm,
            [common_1.LatencyType.TM99]: props.addLatencyTM99Alarm,
            [common_1.LatencyType.TM999]: props.addLatencyTM999Alarm,
            [common_1.LatencyType.TM9999]: props.addLatencyTM9999Alarm,
            [common_1.LatencyType.AVERAGE]: props.addLatencyAverageAlarm,
        };
        Object.values(common_1.LatencyType).forEach((latencyType) => {
            this.latencyMetrics[latencyType] =
                metricFactory.metricLatencyInMillis(latencyType);
        });
        for (const [latencyType, alarmDefinition] of Object.entries(latencyAlarmDefinitions)) {
            for (const disambiguator in alarmDefinition) {
                const alarmProps = alarmDefinition[disambiguator];
                const latencyTypeEnum = latencyType;
                const metric = this.latencyMetrics[latencyTypeEnum];
                const createdAlarm = this.latencyAlarmFactory.addLatencyAlarm(metric, latencyTypeEnum, alarmProps, disambiguator);
                this.latencyAnnotations.push(createdAlarm.annotation);
                this.latencyTypesToRender.push(latencyTypeEnum);
                this.addAlarm(createdAlarm);
            }
        }
        for (const disambiguator in props.add5XXFaultCountAlarm) {
            const alarmProps = props.add5XXFaultCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.fault5XXCountMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add4XXErrorCountAlarm) {
            const alarmProps = props.add4XXErrorCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.error4XXCountMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add4XXErrorRateAlarm) {
            const alarmProps = props.add4XXErrorRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.error4XXRateMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add5XXFaultRateAlarm) {
            const alarmProps = props.add5XXFaultRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.fault5XXRateMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addLowTpsAlarm) {
            const alarmProps = props.addLowTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMinTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addHighTpsAlarm) {
            const alarmProps = props.addHighTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMaxTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        props.useCreatedAlarms?.consume(this.createdAlarms());
    }
    summaryWidgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createLatencyWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createErrorRateWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
        ];
    }
    widgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createLatencyWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorCountWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorRateWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
        ];
    }
    createTitleWidget() {
        return new dashboard_1.MonitoringHeaderWidget({
            family: "API Gateway Endpoint",
            title: this.title,
        });
    }
    createTpsWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "TPS",
            left: [this.tpsMetric],
            leftYAxis: common_1.CountAxisFromZero,
            leftAnnotations: this.tpsAnnotations,
        });
    }
    createLatencyWidget(width, height) {
        const left = Array.from(new Set(this.latencyTypesToRender))
            .sort()
            .map((type) => this.latencyMetrics[type]);
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Latency",
            left,
            leftYAxis: common_1.TimeAxisMillisFromZero,
            leftAnnotations: this.latencyAnnotations,
        });
    }
    createErrorCountWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors",
            left: [this.error4XXCountMetric, this.fault5XXCountMetric],
            leftYAxis: common_1.CountAxisFromZero,
            leftAnnotations: this.errorCountAnnotations,
        });
    }
    createErrorRateWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors (rate)",
            left: [this.error4XXRateMetric, this.fault5XXRateMetric],
            leftYAxis: common_1.RateAxisFromZero,
            leftAnnotations: this.errorRateAnnotations,
        });
    }
}
exports.ApiGatewayMonitoring = ApiGatewayMonitoring;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayMonitoring[_a] = { fqn: "cdk-monitoring-constructs.ApiGatewayMonitoring", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,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