"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayV2HttpApiMonitoring = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const common_1 = require("../../common");
const dashboard_1 = require("../../dashboard");
const ApiGatewayV2HttpApiMetricFactory_1 = require("./ApiGatewayV2HttpApiMetricFactory");
const DefaultLatencyTypesToRender = [
    common_1.LatencyType.P50,
    common_1.LatencyType.P90,
    common_1.LatencyType.P99,
];
class ApiGatewayV2HttpApiMonitoring extends common_1.Monitoring {
    constructor(scope, props) {
        super(scope, props);
        // used when humanReadableName is not provided by user
        const fallbackNameArray = [props.api.apiId];
        fallbackNameArray.push(props.apiStage ?? "$default");
        if (props.apiMethod) {
            fallbackNameArray.push(props.apiMethod);
        }
        if (props.apiResource) {
            fallbackNameArray.push(props.apiResource);
        }
        const namingStrategy = new dashboard_1.MonitoringNamingStrategy({
            ...props,
            namedConstruct: props.api,
            fallbackConstructName: fallbackNameArray
                .join("-")
                .replace(/[^a-zA-Z0-9-_]/g, ""),
            humanReadableName: props.humanReadableName ?? fallbackNameArray.join(" "),
        });
        this.title = namingStrategy.resolveHumanReadableName();
        this.alarmFactory = this.createAlarmFactory(namingStrategy.resolveAlarmFriendlyName());
        this.errorAlarmFactory = new common_1.ErrorAlarmFactory(this.alarmFactory);
        this.tpsAlarmFactory = new common_1.TpsAlarmFactory(this.alarmFactory);
        this.latencyAlarmFactory = new common_1.LatencyAlarmFactory(this.alarmFactory);
        this.tpsAnnotations = [];
        this.latencyAnnotations = [];
        this.errorCountAnnotations = [];
        this.errorRateAnnotations = [];
        const metricFactory = new ApiGatewayV2HttpApiMetricFactory_1.ApiGatewayV2HttpApiMetricFactory(scope.createMetricFactory(), props);
        this.tpsMetric = metricFactory.metricTps();
        this.latencyMetrics = {};
        this.integrationLatencyMetrics = {};
        this.latencyTypesToRender = [
            ...(props.latencyTypesToRender ?? DefaultLatencyTypesToRender),
        ];
        this.error4xxCountMetric = metricFactory.metric4xxCount();
        this.error4xxRateMetric = metricFactory.metric4xxRate();
        this.error5xxCountMetric = metricFactory.metric5xxCount();
        this.error5xxRateMetric = metricFactory.metric5xxRate();
        const latencyAlarmDefinitions = {
            [common_1.LatencyType.P50]: props.addLatencyP50Alarm,
            [common_1.LatencyType.P70]: props.addLatencyP70Alarm,
            [common_1.LatencyType.P90]: props.addLatencyP90Alarm,
            [common_1.LatencyType.P95]: props.addLatencyP95Alarm,
            [common_1.LatencyType.P99]: props.addLatencyP99Alarm,
            [common_1.LatencyType.P999]: props.addLatencyP999Alarm,
            [common_1.LatencyType.P9999]: props.addLatencyP9999Alarm,
            [common_1.LatencyType.P100]: props.addLatencyP100Alarm,
            [common_1.LatencyType.TM50]: props.addLatencyTM50Alarm,
            [common_1.LatencyType.TM70]: props.addLatencyTM70Alarm,
            [common_1.LatencyType.TM90]: props.addLatencyTM90Alarm,
            [common_1.LatencyType.TM95]: props.addLatencyTM95Alarm,
            [common_1.LatencyType.TM99]: props.addLatencyTM99Alarm,
            [common_1.LatencyType.TM999]: props.addLatencyTM999Alarm,
            [common_1.LatencyType.TM9999]: props.addLatencyTM9999Alarm,
            [common_1.LatencyType.AVERAGE]: props.addLatencyAverageAlarm,
        };
        const integrationLatencyAlarmDefinitions = {
            [common_1.LatencyType.P50]: props.addIntegrationLatencyP50Alarm,
            [common_1.LatencyType.P70]: props.addIntegrationLatencyP70Alarm,
            [common_1.LatencyType.P90]: props.addIntegrationLatencyP90Alarm,
            [common_1.LatencyType.P95]: props.addIntegrationLatencyP95Alarm,
            [common_1.LatencyType.P99]: props.addIntegrationLatencyP99Alarm,
            [common_1.LatencyType.P999]: props.addIntegrationLatencyP999Alarm,
            [common_1.LatencyType.P9999]: props.addIntegrationLatencyP9999Alarm,
            [common_1.LatencyType.P100]: props.addIntegrationLatencyP100Alarm,
            [common_1.LatencyType.TM50]: props.addIntegrationLatencyTM50Alarm,
            [common_1.LatencyType.TM70]: props.addIntegrationLatencyTM70Alarm,
            [common_1.LatencyType.TM90]: props.addIntegrationLatencyTM90Alarm,
            [common_1.LatencyType.TM95]: props.addIntegrationLatencyTM95Alarm,
            [common_1.LatencyType.TM99]: props.addIntegrationLatencyTM99Alarm,
            [common_1.LatencyType.TM999]: props.addIntegrationLatencyTM999Alarm,
            [common_1.LatencyType.TM9999]: props.addIntegrationLatencyTM9999Alarm,
            [common_1.LatencyType.AVERAGE]: props.addIntegrationLatencyAverageAlarm,
        };
        Object.values(common_1.LatencyType).forEach((latencyType) => {
            this.latencyMetrics[latencyType] =
                metricFactory.metricLatencyInMillis(latencyType);
            this.integrationLatencyMetrics[latencyType] =
                metricFactory.metricIntegrationLatencyInMillis(latencyType);
        });
        for (const [latencyType, alarmDefinition] of Object.entries(latencyAlarmDefinitions)) {
            for (const disambiguator in alarmDefinition) {
                const alarmProps = alarmDefinition[disambiguator];
                const latencyTypeEnum = latencyType;
                const metric = this.latencyMetrics[latencyTypeEnum];
                const createdAlarm = this.latencyAlarmFactory.addLatencyAlarm(metric, latencyTypeEnum, alarmProps, disambiguator);
                this.latencyAnnotations.push(createdAlarm.annotation);
                this.latencyTypesToRender.push(latencyTypeEnum);
                this.addAlarm(createdAlarm);
            }
        }
        for (const [latencyType, alarmDefinition] of Object.entries(integrationLatencyAlarmDefinitions)) {
            for (const disambiguator in alarmDefinition) {
                const alarmProps = alarmDefinition[disambiguator];
                const latencyTypeEnum = latencyType;
                const metric = this.integrationLatencyMetrics[latencyTypeEnum];
                const createdAlarm = this.latencyAlarmFactory.addIntegrationLatencyAlarm(metric, latencyTypeEnum, alarmProps, disambiguator);
                this.latencyAnnotations.push(createdAlarm.annotation);
                this.latencyTypesToRender.push(latencyTypeEnum);
                this.addAlarm(createdAlarm);
            }
        }
        for (const disambiguator in props.add4xxCountAlarm) {
            const alarmProps = props.add4xxCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.error4xxCountMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add4xxRateAlarm) {
            const alarmProps = props.add4xxRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.error4xxRateMetric, common_1.ErrorType.ERROR, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add5xxCountAlarm) {
            const alarmProps = props.add5xxCountAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorCountAlarm(this.error5xxCountMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorCountAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.add5xxRateAlarm) {
            const alarmProps = props.add5xxRateAlarm[disambiguator];
            const createdAlarm = this.errorAlarmFactory.addErrorRateAlarm(this.error5xxRateMetric, common_1.ErrorType.FAULT, alarmProps, disambiguator);
            this.errorRateAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addLowTpsAlarm) {
            const alarmProps = props.addLowTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMinTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        for (const disambiguator in props.addHighTpsAlarm) {
            const alarmProps = props.addHighTpsAlarm[disambiguator];
            const createdAlarm = this.tpsAlarmFactory.addMaxTpsAlarm(this.tpsMetric, alarmProps, disambiguator);
            this.tpsAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
        props.useCreatedAlarms?.consume(this.createdAlarms());
    }
    summaryWidgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createLatencyWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
            this.createErrorRateWidget(common_1.ThirdWidth, common_1.DefaultSummaryWidgetHeight),
        ];
    }
    widgets() {
        return [
            this.createTitleWidget(),
            this.createTpsWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createLatencyWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorCountWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
            this.createErrorRateWidget(common_1.QuarterWidth, common_1.DefaultGraphWidgetHeight),
        ];
    }
    createTitleWidget() {
        return new dashboard_1.MonitoringHeaderWidget({
            family: "API Gateway V2 HTTP Endpoint",
            title: this.title,
        });
    }
    createTpsWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "TPS",
            left: [this.tpsMetric],
            leftYAxis: common_1.RateAxisFromZero,
            leftAnnotations: this.tpsAnnotations,
        });
    }
    createLatencyWidget(width, height) {
        const left = [];
        Array.from(new Set(this.latencyTypesToRender))
            .sort()
            .forEach((type) => {
            left.push(this.latencyMetrics[type]);
            left.push(this.integrationLatencyMetrics[type]);
        });
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Latency",
            left,
            leftYAxis: common_1.TimeAxisMillisFromZero,
            leftAnnotations: this.latencyAnnotations,
        });
    }
    createErrorCountWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors",
            left: [this.error4xxCountMetric, this.error5xxCountMetric],
            leftAnnotations: this.errorCountAnnotations,
            leftYAxis: common_1.CountAxisFromZero,
        });
    }
    createErrorRateWidget(width, height) {
        return new aws_cloudwatch_1.GraphWidget({
            width,
            height,
            title: "Errors (rate)",
            left: [this.error4xxRateMetric, this.error5xxRateMetric],
            leftYAxis: common_1.RateAxisFromZero,
            leftAnnotations: this.errorRateAnnotations,
        });
    }
}
exports.ApiGatewayV2HttpApiMonitoring = ApiGatewayV2HttpApiMonitoring;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayV2HttpApiMonitoring[_a] = { fqn: "cdk-monitoring-constructs.ApiGatewayV2HttpApiMonitoring", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQXBpR2F0ZXdheVYySHR0cEFwaU1vbml0b3JpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJBcGlHYXRld2F5VjJIdHRwQXBpTW9uaXRvcmluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLCtEQUtvQztBQUVwQyx5Q0F1QnNCO0FBQ3RCLCtDQUd5QjtBQUN6Qix5RkFHNEM7QUFFNUMsTUFBTSwyQkFBMkIsR0FBRztJQUNsQyxvQkFBVyxDQUFDLEdBQUc7SUFDZixvQkFBVyxDQUFDLEdBQUc7SUFDZixvQkFBVyxDQUFDLEdBQUc7Q0FDaEIsQ0FBQztBQTJERixNQUFhLDZCQUE4QixTQUFRLG1CQUFVO0lBd0IzRCxZQUNFLEtBQXNCLEVBQ3RCLEtBQXlDO1FBRXpDLEtBQUssQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFcEIsc0RBQXNEO1FBQ3RELE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzVDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxJQUFJLFVBQVUsQ0FBQyxDQUFDO1FBQ3JELElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUNuQixpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ3pDO1FBQ0QsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3JCLGlCQUFpQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDM0M7UUFFRCxNQUFNLGNBQWMsR0FBRyxJQUFJLG9DQUF3QixDQUFDO1lBQ2xELEdBQUcsS0FBSztZQUNSLGNBQWMsRUFBRSxLQUFLLENBQUMsR0FBRztZQUN6QixxQkFBcUIsRUFBRSxpQkFBaUI7aUJBQ3JDLElBQUksQ0FBQyxHQUFHLENBQUM7aUJBQ1QsT0FBTyxDQUFDLGlCQUFpQixFQUFFLEVBQUUsQ0FBQztZQUNqQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCLElBQUksaUJBQWlCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLGNBQWMsQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRXZELElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUN6QyxjQUFjLENBQUMsd0JBQXdCLEVBQUUsQ0FDMUMsQ0FBQztRQUVGLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLDBCQUFpQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNsRSxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksd0JBQWUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLG1CQUFtQixHQUFHLElBQUksNEJBQW1CLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXRFLElBQUksQ0FBQyxjQUFjLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxFQUFFLENBQUM7UUFDN0IsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEVBQUUsQ0FBQztRQUNoQyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsRUFBRSxDQUFDO1FBRS9CLE1BQU0sYUFBYSxHQUFHLElBQUksbUVBQWdDLENBQ3hELEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxFQUMzQixLQUFLLENBQ04sQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLEdBQUcsYUFBYSxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBRTNDLElBQUksQ0FBQyxjQUFjLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyx5QkFBeUIsR0FBRyxFQUFFLENBQUM7UUFDcEMsSUFBSSxDQUFDLG9CQUFvQixHQUFHO1lBQzFCLEdBQUcsQ0FBQyxLQUFLLENBQUMsb0JBQW9CLElBQUksMkJBQTJCLENBQUM7U0FDL0QsQ0FBQztRQUVGLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxhQUFhLENBQUMsY0FBYyxFQUFFLENBQUM7UUFDMUQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLGFBQWEsQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUV4RCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsYUFBYSxDQUFDLGNBQWMsRUFBRSxDQUFDO1FBQzFELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxhQUFhLENBQUMsYUFBYSxFQUFFLENBQUM7UUFFeEQsTUFBTSx1QkFBdUIsR0FBRztZQUM5QixDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUMzQyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUMvQyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM3QyxDQUFDLG9CQUFXLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUMvQyxDQUFDLG9CQUFXLENBQUMsTUFBTSxDQUFDLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtZQUNqRCxDQUFDLG9CQUFXLENBQUMsT0FBTyxDQUFDLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUNwRCxDQUFDO1FBRUYsTUFBTSxrQ0FBa0MsR0FBRztZQUN6QyxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLDZCQUE2QjtZQUN0RCxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLDZCQUE2QjtZQUN0RCxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLDZCQUE2QjtZQUN0RCxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLDZCQUE2QjtZQUN0RCxDQUFDLG9CQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsS0FBSyxDQUFDLDZCQUE2QjtZQUN0RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxDQUFDLCtCQUErQjtZQUMxRCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUN4RCxDQUFDLG9CQUFXLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxDQUFDLCtCQUErQjtZQUMxRCxDQUFDLG9CQUFXLENBQUMsTUFBTSxDQUFDLEVBQUUsS0FBSyxDQUFDLGdDQUFnQztZQUM1RCxDQUFDLG9CQUFXLENBQUMsT0FBTyxDQUFDLEVBQUUsS0FBSyxDQUFDLGlDQUFpQztTQUMvRCxDQUFDO1FBRUYsTUFBTSxDQUFDLE1BQU0sQ0FBQyxvQkFBVyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDakQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUM7Z0JBQzlCLGFBQWEsQ0FBQyxxQkFBcUIsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUNuRCxJQUFJLENBQUMseUJBQXlCLENBQUMsV0FBVyxDQUFDO2dCQUN6QyxhQUFhLENBQUMsZ0NBQWdDLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDaEUsQ0FBQyxDQUFDLENBQUM7UUFFSCxLQUFLLE1BQU0sQ0FBQyxXQUFXLEVBQUUsZUFBZSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FDekQsdUJBQXVCLENBQ3hCLEVBQUU7WUFDRCxLQUFLLE1BQU0sYUFBYSxJQUFJLGVBQWUsRUFBRTtnQkFDM0MsTUFBTSxVQUFVLEdBQUcsZUFBZSxDQUFDLGFBQWEsQ0FBQyxDQUFDO2dCQUNsRCxNQUFNLGVBQWUsR0FBRyxXQUEwQixDQUFDO2dCQUNuRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FBQyxDQUFDO2dCQUNwRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsZUFBZSxDQUMzRCxNQUFNLEVBQ04sZUFBZSxFQUNmLFVBQVUsRUFDVixhQUFhLENBQ2QsQ0FBQztnQkFDRixJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDdEQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztnQkFDaEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQzthQUM3QjtTQUNGO1FBRUQsS0FBSyxNQUFNLENBQUMsV0FBVyxFQUFFLGVBQWUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQ3pELGtDQUFrQyxDQUNuQyxFQUFFO1lBQ0QsS0FBSyxNQUFNLGFBQWEsSUFBSSxlQUFlLEVBQUU7Z0JBQzNDLE1BQU0sVUFBVSxHQUFHLGVBQWUsQ0FBQyxhQUFhLENBQUMsQ0FBQztnQkFDbEQsTUFBTSxlQUFlLEdBQUcsV0FBMEIsQ0FBQztnQkFDbkQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsQ0FBQyxDQUFDO2dCQUMvRCxNQUFNLFlBQVksR0FDaEIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLDBCQUEwQixDQUNqRCxNQUFNLEVBQ04sZUFBZSxFQUNmLFVBQVUsRUFDVixhQUFhLENBQ2QsQ0FBQztnQkFDSixJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDdEQsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztnQkFDaEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQzthQUM3QjtTQUNGO1FBRUQsS0FBSyxNQUFNLGFBQWEsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDbEQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3pELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FDNUQsSUFBSSxDQUFDLG1CQUFtQixFQUN4QixrQkFBUyxDQUFDLEtBQUssRUFDZixVQUFVLEVBQ1YsYUFBYSxDQUNkLENBQUM7WUFDRixJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUN6RCxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQzdCO1FBRUQsS0FBSyxNQUFNLGFBQWEsSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ2pELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDeEQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGlCQUFpQixDQUMzRCxJQUFJLENBQUMsa0JBQWtCLEVBQ3ZCLGtCQUFTLENBQUMsS0FBSyxFQUNmLFVBQVUsRUFDVixhQUFhLENBQ2QsQ0FBQztZQUNGLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3hELElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDN0I7UUFFRCxLQUFLLE1BQU0sYUFBYSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUNsRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDekQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGtCQUFrQixDQUM1RCxJQUFJLENBQUMsbUJBQW1CLEVBQ3hCLGtCQUFTLENBQUMsS0FBSyxFQUNmLFVBQVUsRUFDVixhQUFhLENBQ2QsQ0FBQztZQUNGLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ3pELElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDN0I7UUFFRCxLQUFLLE1BQU0sYUFBYSxJQUFJLEtBQUssQ0FBQyxlQUFlLEVBQUU7WUFDakQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN4RCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsaUJBQWlCLENBQzNELElBQUksQ0FBQyxrQkFBa0IsRUFDdkIsa0JBQVMsQ0FBQyxLQUFLLEVBQ2YsVUFBVSxFQUNWLGFBQWEsQ0FDZCxDQUFDO1lBQ0YsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDeEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM3QjtRQUVELEtBQUssTUFBTSxhQUFhLElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUNoRCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3ZELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUN0RCxJQUFJLENBQUMsU0FBUyxFQUNkLFVBQVUsRUFDVixhQUFhLENBQ2QsQ0FBQztZQUNGLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUNsRCxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQzdCO1FBRUQsS0FBSyxNQUFNLGFBQWEsSUFBSSxLQUFLLENBQUMsZUFBZSxFQUFFO1lBQ2pELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDeEQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQ3RELElBQUksQ0FBQyxTQUFTLEVBQ2QsVUFBVSxFQUNWLGFBQWEsQ0FDZCxDQUFDO1lBQ0YsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQ2xELElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDN0I7UUFFRCxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRCxjQUFjO1FBQ1osT0FBTztZQUNMLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUN4QixJQUFJLENBQUMsZUFBZSxDQUFDLG1CQUFVLEVBQUUsbUNBQTBCLENBQUM7WUFDNUQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLG1CQUFVLEVBQUUsbUNBQTBCLENBQUM7WUFDaEUsSUFBSSxDQUFDLHFCQUFxQixDQUFDLG1CQUFVLEVBQUUsbUNBQTBCLENBQUM7U0FDbkUsQ0FBQztJQUNKLENBQUM7SUFFRCxPQUFPO1FBQ0wsT0FBTztZQUNMLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUN4QixJQUFJLENBQUMsZUFBZSxDQUFDLHFCQUFZLEVBQUUsaUNBQXdCLENBQUM7WUFDNUQsSUFBSSxDQUFDLG1CQUFtQixDQUFDLHFCQUFZLEVBQUUsaUNBQXdCLENBQUM7WUFDaEUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLHFCQUFZLEVBQUUsaUNBQXdCLENBQUM7WUFDbkUsSUFBSSxDQUFDLHFCQUFxQixDQUFDLHFCQUFZLEVBQUUsaUNBQXdCLENBQUM7U0FDbkUsQ0FBQztJQUNKLENBQUM7SUFFRCxpQkFBaUI7UUFDZixPQUFPLElBQUksa0NBQXNCLENBQUM7WUFDaEMsTUFBTSxFQUFFLDhCQUE4QjtZQUN0QyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDbEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELGVBQWUsQ0FBQyxLQUFhLEVBQUUsTUFBYztRQUMzQyxPQUFPLElBQUksNEJBQVcsQ0FBQztZQUNyQixLQUFLO1lBQ0wsTUFBTTtZQUNOLEtBQUssRUFBRSxLQUFLO1lBQ1osSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUN0QixTQUFTLEVBQUUseUJBQWdCO1lBQzNCLGVBQWUsRUFBRSxJQUFJLENBQUMsY0FBYztTQUNyQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsbUJBQW1CLENBQUMsS0FBYSxFQUFFLE1BQWM7UUFDL0MsTUFBTSxJQUFJLEdBQWMsRUFBRSxDQUFDO1FBRTNCLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7YUFDM0MsSUFBSSxFQUFFO2FBQ04sT0FBTyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUU7WUFDaEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNsRCxDQUFDLENBQUMsQ0FBQztRQUVMLE9BQU8sSUFBSSw0QkFBVyxDQUFDO1lBQ3JCLEtBQUs7WUFDTCxNQUFNO1lBQ04sS0FBSyxFQUFFLFNBQVM7WUFDaEIsSUFBSTtZQUNKLFNBQVMsRUFBRSwrQkFBc0I7WUFDakMsZUFBZSxFQUFFLElBQUksQ0FBQyxrQkFBa0I7U0FDekMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELHNCQUFzQixDQUFDLEtBQWEsRUFBRSxNQUFjO1FBQ2xELE9BQU8sSUFBSSw0QkFBVyxDQUFDO1lBQ3JCLEtBQUs7WUFDTCxNQUFNO1lBQ04sS0FBSyxFQUFFLFFBQVE7WUFDZixJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDO1lBQzFELGVBQWUsRUFBRSxJQUFJLENBQUMscUJBQXFCO1lBQzNDLFNBQVMsRUFBRSwwQkFBaUI7U0FDN0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELHFCQUFxQixDQUFDLEtBQWEsRUFBRSxNQUFjO1FBQ2pELE9BQU8sSUFBSSw0QkFBVyxDQUFDO1lBQ3JCLEtBQUs7WUFDTCxNQUFNO1lBQ04sS0FBSyxFQUFFLGVBQWU7WUFDdEIsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQztZQUN4RCxTQUFTLEVBQUUseUJBQWdCO1lBQzNCLGVBQWUsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1NBQzNDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBN1RILHNFQThUQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIEdyYXBoV2lkZ2V0LFxuICBIb3Jpem9udGFsQW5ub3RhdGlvbixcbiAgSU1ldHJpYyxcbiAgSVdpZGdldCxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoXCI7XG5cbmltcG9ydCB7XG4gIEFsYXJtRmFjdG9yeSxcbiAgQmFzZU1vbml0b3JpbmdQcm9wcyxcbiAgQ291bnRBeGlzRnJvbVplcm8sXG4gIERlZmF1bHRHcmFwaFdpZGdldEhlaWdodCxcbiAgRGVmYXVsdFN1bW1hcnlXaWRnZXRIZWlnaHQsXG4gIEVycm9yQWxhcm1GYWN0b3J5LFxuICBFcnJvckNvdW50VGhyZXNob2xkLFxuICBFcnJvclJhdGVUaHJlc2hvbGQsXG4gIEVycm9yVHlwZSxcbiAgSGlnaFRwc1RocmVzaG9sZCxcbiAgTGF0ZW5jeUFsYXJtRmFjdG9yeSxcbiAgTGF0ZW5jeVRocmVzaG9sZCxcbiAgTGF0ZW5jeVR5cGUsXG4gIExvd1Rwc1RocmVzaG9sZCxcbiAgTWV0cmljV2l0aEFsYXJtU3VwcG9ydCxcbiAgTW9uaXRvcmluZyxcbiAgTW9uaXRvcmluZ1Njb3BlLFxuICBRdWFydGVyV2lkdGgsXG4gIFJhdGVBeGlzRnJvbVplcm8sXG4gIFRoaXJkV2lkdGgsXG4gIFRpbWVBeGlzTWlsbGlzRnJvbVplcm8sXG4gIFRwc0FsYXJtRmFjdG9yeSxcbn0gZnJvbSBcIi4uLy4uL2NvbW1vblwiO1xuaW1wb3J0IHtcbiAgTW9uaXRvcmluZ0hlYWRlcldpZGdldCxcbiAgTW9uaXRvcmluZ05hbWluZ1N0cmF0ZWd5LFxufSBmcm9tIFwiLi4vLi4vZGFzaGJvYXJkXCI7XG5pbXBvcnQge1xuICBBcGlHYXRld2F5VjJIdHRwQXBpTWV0cmljRmFjdG9yeSxcbiAgQXBpR2F0ZXdheVYySHR0cEFwaU1ldHJpY0ZhY3RvcnlQcm9wcyxcbn0gZnJvbSBcIi4vQXBpR2F0ZXdheVYySHR0cEFwaU1ldHJpY0ZhY3RvcnlcIjtcblxuY29uc3QgRGVmYXVsdExhdGVuY3lUeXBlc1RvUmVuZGVyID0gW1xuICBMYXRlbmN5VHlwZS5QNTAsXG4gIExhdGVuY3lUeXBlLlA5MCxcbiAgTGF0ZW5jeVR5cGUuUDk5LFxuXTtcblxuZXhwb3J0IGludGVyZmFjZSBBcGlHYXRld2F5VjJNb25pdG9yaW5nT3B0aW9ucyBleHRlbmRzIEJhc2VNb25pdG9yaW5nUHJvcHMge1xuICByZWFkb25seSBhZGQ0eHhDb3VudEFsYXJtPzogUmVjb3JkPHN0cmluZywgRXJyb3JDb3VudFRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZDR4eFJhdGVBbGFybT86IFJlY29yZDxzdHJpbmcsIEVycm9yUmF0ZVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZDV4eENvdW50QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBFcnJvckNvdW50VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkNXh4UmF0ZUFsYXJtPzogUmVjb3JkPHN0cmluZywgRXJyb3JSYXRlVGhyZXNob2xkPjtcblxuICByZWFkb25seSBhZGRMYXRlbmN5UDUwQWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVA3MEFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZExhdGVuY3lQOTBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRMYXRlbmN5UDk1QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVA5OUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZExhdGVuY3lQOTk5QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVA5OTk5QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVAxMDBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRMYXRlbmN5VE01MEFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZExhdGVuY3lUTTcwQWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVRNOTBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRMYXRlbmN5VE05NUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZExhdGVuY3lUTTk5QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVRNOTk5QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkTGF0ZW5jeVRNOTk5OUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZExhdGVuY3lBdmVyYWdlQWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcblxuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQNTBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQNzBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTVBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTlBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTk5QWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkSW50ZWdyYXRpb25MYXRlbmN5UDk5OTlBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lQMTAwQWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkSW50ZWdyYXRpb25MYXRlbmN5VE01MEFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZEludGVncmF0aW9uTGF0ZW5jeVRNNzBBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lUTTkwQWxhcm0/OiBSZWNvcmQ8c3RyaW5nLCBMYXRlbmN5VGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkSW50ZWdyYXRpb25MYXRlbmN5VE05NUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZEludGVncmF0aW9uTGF0ZW5jeVRNOTlBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuICByZWFkb25seSBhZGRJbnRlZ3JhdGlvbkxhdGVuY3lUTTk5OUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZEludGVncmF0aW9uTGF0ZW5jeVRNOTk5OUFsYXJtPzogUmVjb3JkPHN0cmluZywgTGF0ZW5jeVRocmVzaG9sZD47XG4gIHJlYWRvbmx5IGFkZEludGVncmF0aW9uTGF0ZW5jeUF2ZXJhZ2VBbGFybT86IFJlY29yZDxzdHJpbmcsIExhdGVuY3lUaHJlc2hvbGQ+O1xuXG4gIHJlYWRvbmx5IGFkZExvd1Rwc0FsYXJtPzogUmVjb3JkPHN0cmluZywgTG93VHBzVGhyZXNob2xkPjtcbiAgcmVhZG9ubHkgYWRkSGlnaFRwc0FsYXJtPzogUmVjb3JkPHN0cmluZywgSGlnaFRwc1RocmVzaG9sZD47XG5cbiAgLyoqXG4gICAqIFlvdSBjYW4gc3BlY2lmeSB3aGF0IGxhdGVuY3kgdHlwZXMgeW91IHdhbnQgdG8gYmUgcmVuZGVyZWQgaW4gdGhlIGRhc2hib2FyZHMuXG4gICAqIE5vdGU6IGFueSBsYXRlbmN5IHR5cGUgd2l0aCBhbiBhbGFybSB3aWxsIGJlIGFsc28gYWRkZWQgYXV0b21hdGljYWxseS5cbiAgICogSWYgdGhlIGxpc3QgaXMgdW5kZWZpbmVkLCBkZWZhdWx0IHZhbHVlcyB3aWxsIGJlIHNob3duLlxuICAgKiBJZiB0aGUgbGlzdCBpcyBlbXB0eSwgb25seSB0aGUgbGF0ZW5jeSB0eXBlcyB3aXRoIGFuIGFsYXJtIHdpbGwgYmUgc2hvd24gKGlmIGFueSkuXG4gICAqIEBkZWZhdWx0IC0gcDUwLCBwOTAsIHA5OSAoQHNlZSBEZWZhdWx0TGF0ZW5jeVR5cGVzU2hvd24pXG4gICAqL1xuICByZWFkb25seSBsYXRlbmN5VHlwZXNUb1JlbmRlcj86IExhdGVuY3lUeXBlW107XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQXBpR2F0ZXdheVYySHR0cEFwaU1vbml0b3JpbmdQcm9wc1xuICBleHRlbmRzIEFwaUdhdGV3YXlWMkh0dHBBcGlNZXRyaWNGYWN0b3J5UHJvcHMsXG4gICAgQXBpR2F0ZXdheVYyTW9uaXRvcmluZ09wdGlvbnMge31cblxuZXhwb3J0IGNsYXNzIEFwaUdhdGV3YXlWMkh0dHBBcGlNb25pdG9yaW5nIGV4dGVuZHMgTW9uaXRvcmluZyB7XG4gIHJlYWRvbmx5IHRpdGxlOiBzdHJpbmc7XG5cbiAgcmVhZG9ubHkgYWxhcm1GYWN0b3J5OiBBbGFybUZhY3Rvcnk7XG4gIHJlYWRvbmx5IGVycm9yQWxhcm1GYWN0b3J5OiBFcnJvckFsYXJtRmFjdG9yeTtcbiAgcmVhZG9ubHkgdHBzQWxhcm1GYWN0b3J5OiBUcHNBbGFybUZhY3Rvcnk7XG4gIHJlYWRvbmx5IGxhdGVuY3lBbGFybUZhY3Rvcnk6IExhdGVuY3lBbGFybUZhY3Rvcnk7XG5cbiAgcmVhZG9ubHkgdHBzQW5ub3RhdGlvbnM6IEhvcml6b250YWxBbm5vdGF0aW9uW107XG4gIHJlYWRvbmx5IGxhdGVuY3lBbm5vdGF0aW9uczogSG9yaXpvbnRhbEFubm90YXRpb25bXTtcbiAgcmVhZG9ubHkgZXJyb3JDb3VudEFubm90YXRpb25zOiBIb3Jpem9udGFsQW5ub3RhdGlvbltdO1xuICByZWFkb25seSBlcnJvclJhdGVBbm5vdGF0aW9uczogSG9yaXpvbnRhbEFubm90YXRpb25bXTtcblxuICByZWFkb25seSB0cHNNZXRyaWM6IE1ldHJpY1dpdGhBbGFybVN1cHBvcnQ7XG4gIHJlYWRvbmx5IGVycm9yNHh4Q291bnRNZXRyaWM6IE1ldHJpY1dpdGhBbGFybVN1cHBvcnQ7XG4gIHJlYWRvbmx5IGVycm9yNHh4UmF0ZU1ldHJpYzogTWV0cmljV2l0aEFsYXJtU3VwcG9ydDtcbiAgcmVhZG9ubHkgZXJyb3I1eHhDb3VudE1ldHJpYzogTWV0cmljV2l0aEFsYXJtU3VwcG9ydDtcbiAgcmVhZG9ubHkgZXJyb3I1eHhSYXRlTWV0cmljOiBNZXRyaWNXaXRoQWxhcm1TdXBwb3J0O1xuXG4gIC8vIGtleXMgYXJlIExhdGVuY3lUeXBlLCBidXQgSlNJSSBkb2Vzbid0IGxpa2Ugbm9uLXN0cmluZyB0eXBlc1xuICByZWFkb25seSBsYXRlbmN5TWV0cmljczogUmVjb3JkPHN0cmluZywgTWV0cmljV2l0aEFsYXJtU3VwcG9ydD47XG4gIHJlYWRvbmx5IGludGVncmF0aW9uTGF0ZW5jeU1ldHJpY3M6IFJlY29yZDxzdHJpbmcsIE1ldHJpY1dpdGhBbGFybVN1cHBvcnQ+O1xuICByZWFkb25seSBsYXRlbmN5VHlwZXNUb1JlbmRlcjogc3RyaW5nW107XG5cbiAgY29uc3RydWN0b3IoXG4gICAgc2NvcGU6IE1vbml0b3JpbmdTY29wZSxcbiAgICBwcm9wczogQXBpR2F0ZXdheVYySHR0cEFwaU1vbml0b3JpbmdQcm9wc1xuICApIHtcbiAgICBzdXBlcihzY29wZSwgcHJvcHMpO1xuXG4gICAgLy8gdXNlZCB3aGVuIGh1bWFuUmVhZGFibGVOYW1lIGlzIG5vdCBwcm92aWRlZCBieSB1c2VyXG4gICAgY29uc3QgZmFsbGJhY2tOYW1lQXJyYXkgPSBbcHJvcHMuYXBpLmFwaUlkXTtcbiAgICBmYWxsYmFja05hbWVBcnJheS5wdXNoKHByb3BzLmFwaVN0YWdlID8/IFwiJGRlZmF1bHRcIik7XG4gICAgaWYgKHByb3BzLmFwaU1ldGhvZCkge1xuICAgICAgZmFsbGJhY2tOYW1lQXJyYXkucHVzaChwcm9wcy5hcGlNZXRob2QpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuYXBpUmVzb3VyY2UpIHtcbiAgICAgIGZhbGxiYWNrTmFtZUFycmF5LnB1c2gocHJvcHMuYXBpUmVzb3VyY2UpO1xuICAgIH1cblxuICAgIGNvbnN0IG5hbWluZ1N0cmF0ZWd5ID0gbmV3IE1vbml0b3JpbmdOYW1pbmdTdHJhdGVneSh7XG4gICAgICAuLi5wcm9wcyxcbiAgICAgIG5hbWVkQ29uc3RydWN0OiBwcm9wcy5hcGksXG4gICAgICBmYWxsYmFja0NvbnN0cnVjdE5hbWU6IGZhbGxiYWNrTmFtZUFycmF5XG4gICAgICAgIC5qb2luKFwiLVwiKVxuICAgICAgICAucmVwbGFjZSgvW15hLXpBLVowLTktX10vZywgXCJcIiksXG4gICAgICBodW1hblJlYWRhYmxlTmFtZTogcHJvcHMuaHVtYW5SZWFkYWJsZU5hbWUgPz8gZmFsbGJhY2tOYW1lQXJyYXkuam9pbihcIiBcIiksXG4gICAgfSk7XG5cbiAgICB0aGlzLnRpdGxlID0gbmFtaW5nU3RyYXRlZ3kucmVzb2x2ZUh1bWFuUmVhZGFibGVOYW1lKCk7XG5cbiAgICB0aGlzLmFsYXJtRmFjdG9yeSA9IHRoaXMuY3JlYXRlQWxhcm1GYWN0b3J5KFxuICAgICAgbmFtaW5nU3RyYXRlZ3kucmVzb2x2ZUFsYXJtRnJpZW5kbHlOYW1lKClcbiAgICApO1xuXG4gICAgdGhpcy5lcnJvckFsYXJtRmFjdG9yeSA9IG5ldyBFcnJvckFsYXJtRmFjdG9yeSh0aGlzLmFsYXJtRmFjdG9yeSk7XG4gICAgdGhpcy50cHNBbGFybUZhY3RvcnkgPSBuZXcgVHBzQWxhcm1GYWN0b3J5KHRoaXMuYWxhcm1GYWN0b3J5KTtcbiAgICB0aGlzLmxhdGVuY3lBbGFybUZhY3RvcnkgPSBuZXcgTGF0ZW5jeUFsYXJtRmFjdG9yeSh0aGlzLmFsYXJtRmFjdG9yeSk7XG5cbiAgICB0aGlzLnRwc0Fubm90YXRpb25zID0gW107XG4gICAgdGhpcy5sYXRlbmN5QW5ub3RhdGlvbnMgPSBbXTtcbiAgICB0aGlzLmVycm9yQ291bnRBbm5vdGF0aW9ucyA9IFtdO1xuICAgIHRoaXMuZXJyb3JSYXRlQW5ub3RhdGlvbnMgPSBbXTtcblxuICAgIGNvbnN0IG1ldHJpY0ZhY3RvcnkgPSBuZXcgQXBpR2F0ZXdheVYySHR0cEFwaU1ldHJpY0ZhY3RvcnkoXG4gICAgICBzY29wZS5jcmVhdGVNZXRyaWNGYWN0b3J5KCksXG4gICAgICBwcm9wc1xuICAgICk7XG5cbiAgICB0aGlzLnRwc01ldHJpYyA9IG1ldHJpY0ZhY3RvcnkubWV0cmljVHBzKCk7XG5cbiAgICB0aGlzLmxhdGVuY3lNZXRyaWNzID0ge307XG4gICAgdGhpcy5pbnRlZ3JhdGlvbkxhdGVuY3lNZXRyaWNzID0ge307XG4gICAgdGhpcy5sYXRlbmN5VHlwZXNUb1JlbmRlciA9IFtcbiAgICAgIC4uLihwcm9wcy5sYXRlbmN5VHlwZXNUb1JlbmRlciA/PyBEZWZhdWx0TGF0ZW5jeVR5cGVzVG9SZW5kZXIpLFxuICAgIF07XG5cbiAgICB0aGlzLmVycm9yNHh4Q291bnRNZXRyaWMgPSBtZXRyaWNGYWN0b3J5Lm1ldHJpYzR4eENvdW50KCk7XG4gICAgdGhpcy5lcnJvcjR4eFJhdGVNZXRyaWMgPSBtZXRyaWNGYWN0b3J5Lm1ldHJpYzR4eFJhdGUoKTtcblxuICAgIHRoaXMuZXJyb3I1eHhDb3VudE1ldHJpYyA9IG1ldHJpY0ZhY3RvcnkubWV0cmljNXh4Q291bnQoKTtcbiAgICB0aGlzLmVycm9yNXh4UmF0ZU1ldHJpYyA9IG1ldHJpY0ZhY3RvcnkubWV0cmljNXh4UmF0ZSgpO1xuXG4gICAgY29uc3QgbGF0ZW5jeUFsYXJtRGVmaW5pdGlvbnMgPSB7XG4gICAgICBbTGF0ZW5jeVR5cGUuUDUwXTogcHJvcHMuYWRkTGF0ZW5jeVA1MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA3MF06IHByb3BzLmFkZExhdGVuY3lQNzBBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5QOTBdOiBwcm9wcy5hZGRMYXRlbmN5UDkwQWxhcm0sXG4gICAgICBbTGF0ZW5jeVR5cGUuUDk1XTogcHJvcHMuYWRkTGF0ZW5jeVA5NUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA5OV06IHByb3BzLmFkZExhdGVuY3lQOTlBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5QOTk5XTogcHJvcHMuYWRkTGF0ZW5jeVA5OTlBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5QOTk5OV06IHByb3BzLmFkZExhdGVuY3lQOTk5OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlAxMDBdOiBwcm9wcy5hZGRMYXRlbmN5UDEwMEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNNTBdOiBwcm9wcy5hZGRMYXRlbmN5VE01MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNNzBdOiBwcm9wcy5hZGRMYXRlbmN5VE03MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTBdOiBwcm9wcy5hZGRMYXRlbmN5VE05MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTVdOiBwcm9wcy5hZGRMYXRlbmN5VE05NUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTldOiBwcm9wcy5hZGRMYXRlbmN5VE05OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTk5XTogcHJvcHMuYWRkTGF0ZW5jeVRNOTk5QWxhcm0sXG4gICAgICBbTGF0ZW5jeVR5cGUuVE05OTk5XTogcHJvcHMuYWRkTGF0ZW5jeVRNOTk5OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLkFWRVJBR0VdOiBwcm9wcy5hZGRMYXRlbmN5QXZlcmFnZUFsYXJtLFxuICAgIH07XG5cbiAgICBjb25zdCBpbnRlZ3JhdGlvbkxhdGVuY3lBbGFybURlZmluaXRpb25zID0ge1xuICAgICAgW0xhdGVuY3lUeXBlLlA1MF06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVA1MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA3MF06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVA3MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA5MF06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVA5MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA5NV06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVA5NUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA5OV06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVA5OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlA5OTldOiBwcm9wcy5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTk5QWxhcm0sXG4gICAgICBbTGF0ZW5jeVR5cGUuUDk5OTldOiBwcm9wcy5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lQOTk5OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlAxMDBdOiBwcm9wcy5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lQMTAwQWxhcm0sXG4gICAgICBbTGF0ZW5jeVR5cGUuVE01MF06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVRNNTBBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5UTTcwXTogcHJvcHMuYWRkSW50ZWdyYXRpb25MYXRlbmN5VE03MEFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTBdOiBwcm9wcy5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lUTTkwQWxhcm0sXG4gICAgICBbTGF0ZW5jeVR5cGUuVE05NV06IHByb3BzLmFkZEludGVncmF0aW9uTGF0ZW5jeVRNOTVBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5UTTk5XTogcHJvcHMuYWRkSW50ZWdyYXRpb25MYXRlbmN5VE05OUFsYXJtLFxuICAgICAgW0xhdGVuY3lUeXBlLlRNOTk5XTogcHJvcHMuYWRkSW50ZWdyYXRpb25MYXRlbmN5VE05OTlBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5UTTk5OTldOiBwcm9wcy5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lUTTk5OTlBbGFybSxcbiAgICAgIFtMYXRlbmN5VHlwZS5BVkVSQUdFXTogcHJvcHMuYWRkSW50ZWdyYXRpb25MYXRlbmN5QXZlcmFnZUFsYXJtLFxuICAgIH07XG5cbiAgICBPYmplY3QudmFsdWVzKExhdGVuY3lUeXBlKS5mb3JFYWNoKChsYXRlbmN5VHlwZSkgPT4ge1xuICAgICAgdGhpcy5sYXRlbmN5TWV0cmljc1tsYXRlbmN5VHlwZV0gPVxuICAgICAgICBtZXRyaWNGYWN0b3J5Lm1ldHJpY0xhdGVuY3lJbk1pbGxpcyhsYXRlbmN5VHlwZSk7XG4gICAgICB0aGlzLmludGVncmF0aW9uTGF0ZW5jeU1ldHJpY3NbbGF0ZW5jeVR5cGVdID1cbiAgICAgICAgbWV0cmljRmFjdG9yeS5tZXRyaWNJbnRlZ3JhdGlvbkxhdGVuY3lJbk1pbGxpcyhsYXRlbmN5VHlwZSk7XG4gICAgfSk7XG5cbiAgICBmb3IgKGNvbnN0IFtsYXRlbmN5VHlwZSwgYWxhcm1EZWZpbml0aW9uXSBvZiBPYmplY3QuZW50cmllcyhcbiAgICAgIGxhdGVuY3lBbGFybURlZmluaXRpb25zXG4gICAgKSkge1xuICAgICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIGFsYXJtRGVmaW5pdGlvbikge1xuICAgICAgICBjb25zdCBhbGFybVByb3BzID0gYWxhcm1EZWZpbml0aW9uW2Rpc2FtYmlndWF0b3JdO1xuICAgICAgICBjb25zdCBsYXRlbmN5VHlwZUVudW0gPSBsYXRlbmN5VHlwZSBhcyBMYXRlbmN5VHlwZTtcbiAgICAgICAgY29uc3QgbWV0cmljID0gdGhpcy5sYXRlbmN5TWV0cmljc1tsYXRlbmN5VHlwZUVudW1dO1xuICAgICAgICBjb25zdCBjcmVhdGVkQWxhcm0gPSB0aGlzLmxhdGVuY3lBbGFybUZhY3RvcnkuYWRkTGF0ZW5jeUFsYXJtKFxuICAgICAgICAgIG1ldHJpYyxcbiAgICAgICAgICBsYXRlbmN5VHlwZUVudW0sXG4gICAgICAgICAgYWxhcm1Qcm9wcyxcbiAgICAgICAgICBkaXNhbWJpZ3VhdG9yXG4gICAgICAgICk7XG4gICAgICAgIHRoaXMubGF0ZW5jeUFubm90YXRpb25zLnB1c2goY3JlYXRlZEFsYXJtLmFubm90YXRpb24pO1xuICAgICAgICB0aGlzLmxhdGVuY3lUeXBlc1RvUmVuZGVyLnB1c2gobGF0ZW5jeVR5cGVFbnVtKTtcbiAgICAgICAgdGhpcy5hZGRBbGFybShjcmVhdGVkQWxhcm0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIGZvciAoY29uc3QgW2xhdGVuY3lUeXBlLCBhbGFybURlZmluaXRpb25dIG9mIE9iamVjdC5lbnRyaWVzKFxuICAgICAgaW50ZWdyYXRpb25MYXRlbmN5QWxhcm1EZWZpbml0aW9uc1xuICAgICkpIHtcbiAgICAgIGZvciAoY29uc3QgZGlzYW1iaWd1YXRvciBpbiBhbGFybURlZmluaXRpb24pIHtcbiAgICAgICAgY29uc3QgYWxhcm1Qcm9wcyA9IGFsYXJtRGVmaW5pdGlvbltkaXNhbWJpZ3VhdG9yXTtcbiAgICAgICAgY29uc3QgbGF0ZW5jeVR5cGVFbnVtID0gbGF0ZW5jeVR5cGUgYXMgTGF0ZW5jeVR5cGU7XG4gICAgICAgIGNvbnN0IG1ldHJpYyA9IHRoaXMuaW50ZWdyYXRpb25MYXRlbmN5TWV0cmljc1tsYXRlbmN5VHlwZUVudW1dO1xuICAgICAgICBjb25zdCBjcmVhdGVkQWxhcm0gPVxuICAgICAgICAgIHRoaXMubGF0ZW5jeUFsYXJtRmFjdG9yeS5hZGRJbnRlZ3JhdGlvbkxhdGVuY3lBbGFybShcbiAgICAgICAgICAgIG1ldHJpYyxcbiAgICAgICAgICAgIGxhdGVuY3lUeXBlRW51bSxcbiAgICAgICAgICAgIGFsYXJtUHJvcHMsXG4gICAgICAgICAgICBkaXNhbWJpZ3VhdG9yXG4gICAgICAgICAgKTtcbiAgICAgICAgdGhpcy5sYXRlbmN5QW5ub3RhdGlvbnMucHVzaChjcmVhdGVkQWxhcm0uYW5ub3RhdGlvbik7XG4gICAgICAgIHRoaXMubGF0ZW5jeVR5cGVzVG9SZW5kZXIucHVzaChsYXRlbmN5VHlwZUVudW0pO1xuICAgICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIHByb3BzLmFkZDR4eENvdW50QWxhcm0pIHtcbiAgICAgIGNvbnN0IGFsYXJtUHJvcHMgPSBwcm9wcy5hZGQ0eHhDb3VudEFsYXJtW2Rpc2FtYmlndWF0b3JdO1xuICAgICAgY29uc3QgY3JlYXRlZEFsYXJtID0gdGhpcy5lcnJvckFsYXJtRmFjdG9yeS5hZGRFcnJvckNvdW50QWxhcm0oXG4gICAgICAgIHRoaXMuZXJyb3I0eHhDb3VudE1ldHJpYyxcbiAgICAgICAgRXJyb3JUeXBlLkVSUk9SLFxuICAgICAgICBhbGFybVByb3BzLFxuICAgICAgICBkaXNhbWJpZ3VhdG9yXG4gICAgICApO1xuICAgICAgdGhpcy5lcnJvckNvdW50QW5ub3RhdGlvbnMucHVzaChjcmVhdGVkQWxhcm0uYW5ub3RhdGlvbik7XG4gICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIHByb3BzLmFkZDR4eFJhdGVBbGFybSkge1xuICAgICAgY29uc3QgYWxhcm1Qcm9wcyA9IHByb3BzLmFkZDR4eFJhdGVBbGFybVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGNvbnN0IGNyZWF0ZWRBbGFybSA9IHRoaXMuZXJyb3JBbGFybUZhY3RvcnkuYWRkRXJyb3JSYXRlQWxhcm0oXG4gICAgICAgIHRoaXMuZXJyb3I0eHhSYXRlTWV0cmljLFxuICAgICAgICBFcnJvclR5cGUuRVJST1IsXG4gICAgICAgIGFsYXJtUHJvcHMsXG4gICAgICAgIGRpc2FtYmlndWF0b3JcbiAgICAgICk7XG4gICAgICB0aGlzLmVycm9yUmF0ZUFubm90YXRpb25zLnB1c2goY3JlYXRlZEFsYXJtLmFubm90YXRpb24pO1xuICAgICAgdGhpcy5hZGRBbGFybShjcmVhdGVkQWxhcm0pO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgZGlzYW1iaWd1YXRvciBpbiBwcm9wcy5hZGQ1eHhDb3VudEFsYXJtKSB7XG4gICAgICBjb25zdCBhbGFybVByb3BzID0gcHJvcHMuYWRkNXh4Q291bnRBbGFybVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGNvbnN0IGNyZWF0ZWRBbGFybSA9IHRoaXMuZXJyb3JBbGFybUZhY3RvcnkuYWRkRXJyb3JDb3VudEFsYXJtKFxuICAgICAgICB0aGlzLmVycm9yNXh4Q291bnRNZXRyaWMsXG4gICAgICAgIEVycm9yVHlwZS5GQVVMVCxcbiAgICAgICAgYWxhcm1Qcm9wcyxcbiAgICAgICAgZGlzYW1iaWd1YXRvclxuICAgICAgKTtcbiAgICAgIHRoaXMuZXJyb3JDb3VudEFubm90YXRpb25zLnB1c2goY3JlYXRlZEFsYXJtLmFubm90YXRpb24pO1xuICAgICAgdGhpcy5hZGRBbGFybShjcmVhdGVkQWxhcm0pO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgZGlzYW1iaWd1YXRvciBpbiBwcm9wcy5hZGQ1eHhSYXRlQWxhcm0pIHtcbiAgICAgIGNvbnN0IGFsYXJtUHJvcHMgPSBwcm9wcy5hZGQ1eHhSYXRlQWxhcm1bZGlzYW1iaWd1YXRvcl07XG4gICAgICBjb25zdCBjcmVhdGVkQWxhcm0gPSB0aGlzLmVycm9yQWxhcm1GYWN0b3J5LmFkZEVycm9yUmF0ZUFsYXJtKFxuICAgICAgICB0aGlzLmVycm9yNXh4UmF0ZU1ldHJpYyxcbiAgICAgICAgRXJyb3JUeXBlLkZBVUxULFxuICAgICAgICBhbGFybVByb3BzLFxuICAgICAgICBkaXNhbWJpZ3VhdG9yXG4gICAgICApO1xuICAgICAgdGhpcy5lcnJvclJhdGVBbm5vdGF0aW9ucy5wdXNoKGNyZWF0ZWRBbGFybS5hbm5vdGF0aW9uKTtcbiAgICAgIHRoaXMuYWRkQWxhcm0oY3JlYXRlZEFsYXJtKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGRpc2FtYmlndWF0b3IgaW4gcHJvcHMuYWRkTG93VHBzQWxhcm0pIHtcbiAgICAgIGNvbnN0IGFsYXJtUHJvcHMgPSBwcm9wcy5hZGRMb3dUcHNBbGFybVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGNvbnN0IGNyZWF0ZWRBbGFybSA9IHRoaXMudHBzQWxhcm1GYWN0b3J5LmFkZE1pblRwc0FsYXJtKFxuICAgICAgICB0aGlzLnRwc01ldHJpYyxcbiAgICAgICAgYWxhcm1Qcm9wcyxcbiAgICAgICAgZGlzYW1iaWd1YXRvclxuICAgICAgKTtcbiAgICAgIHRoaXMudHBzQW5ub3RhdGlvbnMucHVzaChjcmVhdGVkQWxhcm0uYW5ub3RhdGlvbik7XG4gICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIHByb3BzLmFkZEhpZ2hUcHNBbGFybSkge1xuICAgICAgY29uc3QgYWxhcm1Qcm9wcyA9IHByb3BzLmFkZEhpZ2hUcHNBbGFybVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGNvbnN0IGNyZWF0ZWRBbGFybSA9IHRoaXMudHBzQWxhcm1GYWN0b3J5LmFkZE1heFRwc0FsYXJtKFxuICAgICAgICB0aGlzLnRwc01ldHJpYyxcbiAgICAgICAgYWxhcm1Qcm9wcyxcbiAgICAgICAgZGlzYW1iaWd1YXRvclxuICAgICAgKTtcbiAgICAgIHRoaXMudHBzQW5ub3RhdGlvbnMucHVzaChjcmVhdGVkQWxhcm0uYW5ub3RhdGlvbik7XG4gICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgfVxuXG4gICAgcHJvcHMudXNlQ3JlYXRlZEFsYXJtcz8uY29uc3VtZSh0aGlzLmNyZWF0ZWRBbGFybXMoKSk7XG4gIH1cblxuICBzdW1tYXJ5V2lkZ2V0cygpOiBJV2lkZ2V0W10ge1xuICAgIHJldHVybiBbXG4gICAgICB0aGlzLmNyZWF0ZVRpdGxlV2lkZ2V0KCksXG4gICAgICB0aGlzLmNyZWF0ZVRwc1dpZGdldChUaGlyZFdpZHRoLCBEZWZhdWx0U3VtbWFyeVdpZGdldEhlaWdodCksXG4gICAgICB0aGlzLmNyZWF0ZUxhdGVuY3lXaWRnZXQoVGhpcmRXaWR0aCwgRGVmYXVsdFN1bW1hcnlXaWRnZXRIZWlnaHQpLFxuICAgICAgdGhpcy5jcmVhdGVFcnJvclJhdGVXaWRnZXQoVGhpcmRXaWR0aCwgRGVmYXVsdFN1bW1hcnlXaWRnZXRIZWlnaHQpLFxuICAgIF07XG4gIH1cblxuICB3aWRnZXRzKCk6IElXaWRnZXRbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIHRoaXMuY3JlYXRlVGl0bGVXaWRnZXQoKSxcbiAgICAgIHRoaXMuY3JlYXRlVHBzV2lkZ2V0KFF1YXJ0ZXJXaWR0aCwgRGVmYXVsdEdyYXBoV2lkZ2V0SGVpZ2h0KSxcbiAgICAgIHRoaXMuY3JlYXRlTGF0ZW5jeVdpZGdldChRdWFydGVyV2lkdGgsIERlZmF1bHRHcmFwaFdpZGdldEhlaWdodCksXG4gICAgICB0aGlzLmNyZWF0ZUVycm9yQ291bnRXaWRnZXQoUXVhcnRlcldpZHRoLCBEZWZhdWx0R3JhcGhXaWRnZXRIZWlnaHQpLFxuICAgICAgdGhpcy5jcmVhdGVFcnJvclJhdGVXaWRnZXQoUXVhcnRlcldpZHRoLCBEZWZhdWx0R3JhcGhXaWRnZXRIZWlnaHQpLFxuICAgIF07XG4gIH1cblxuICBjcmVhdGVUaXRsZVdpZGdldCgpIHtcbiAgICByZXR1cm4gbmV3IE1vbml0b3JpbmdIZWFkZXJXaWRnZXQoe1xuICAgICAgZmFtaWx5OiBcIkFQSSBHYXRld2F5IFYyIEhUVFAgRW5kcG9pbnRcIixcbiAgICAgIHRpdGxlOiB0aGlzLnRpdGxlLFxuICAgIH0pO1xuICB9XG5cbiAgY3JlYXRlVHBzV2lkZ2V0KHdpZHRoOiBudW1iZXIsIGhlaWdodDogbnVtYmVyKSB7XG4gICAgcmV0dXJuIG5ldyBHcmFwaFdpZGdldCh7XG4gICAgICB3aWR0aCxcbiAgICAgIGhlaWdodCxcbiAgICAgIHRpdGxlOiBcIlRQU1wiLFxuICAgICAgbGVmdDogW3RoaXMudHBzTWV0cmljXSxcbiAgICAgIGxlZnRZQXhpczogUmF0ZUF4aXNGcm9tWmVybyxcbiAgICAgIGxlZnRBbm5vdGF0aW9uczogdGhpcy50cHNBbm5vdGF0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIGNyZWF0ZUxhdGVuY3lXaWRnZXQod2lkdGg6IG51bWJlciwgaGVpZ2h0OiBudW1iZXIpIHtcbiAgICBjb25zdCBsZWZ0OiBJTWV0cmljW10gPSBbXTtcblxuICAgIEFycmF5LmZyb20obmV3IFNldCh0aGlzLmxhdGVuY3lUeXBlc1RvUmVuZGVyKSlcbiAgICAgIC5zb3J0KClcbiAgICAgIC5mb3JFYWNoKCh0eXBlKSA9PiB7XG4gICAgICAgIGxlZnQucHVzaCh0aGlzLmxhdGVuY3lNZXRyaWNzW3R5cGVdKTtcbiAgICAgICAgbGVmdC5wdXNoKHRoaXMuaW50ZWdyYXRpb25MYXRlbmN5TWV0cmljc1t0eXBlXSk7XG4gICAgICB9KTtcblxuICAgIHJldHVybiBuZXcgR3JhcGhXaWRnZXQoe1xuICAgICAgd2lkdGgsXG4gICAgICBoZWlnaHQsXG4gICAgICB0aXRsZTogXCJMYXRlbmN5XCIsXG4gICAgICBsZWZ0LFxuICAgICAgbGVmdFlBeGlzOiBUaW1lQXhpc01pbGxpc0Zyb21aZXJvLFxuICAgICAgbGVmdEFubm90YXRpb25zOiB0aGlzLmxhdGVuY3lBbm5vdGF0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIGNyZWF0ZUVycm9yQ291bnRXaWRnZXQod2lkdGg6IG51bWJlciwgaGVpZ2h0OiBudW1iZXIpIHtcbiAgICByZXR1cm4gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgIHdpZHRoLFxuICAgICAgaGVpZ2h0LFxuICAgICAgdGl0bGU6IFwiRXJyb3JzXCIsXG4gICAgICBsZWZ0OiBbdGhpcy5lcnJvcjR4eENvdW50TWV0cmljLCB0aGlzLmVycm9yNXh4Q291bnRNZXRyaWNdLFxuICAgICAgbGVmdEFubm90YXRpb25zOiB0aGlzLmVycm9yQ291bnRBbm5vdGF0aW9ucyxcbiAgICAgIGxlZnRZQXhpczogQ291bnRBeGlzRnJvbVplcm8sXG4gICAgfSk7XG4gIH1cblxuICBjcmVhdGVFcnJvclJhdGVXaWRnZXQod2lkdGg6IG51bWJlciwgaGVpZ2h0OiBudW1iZXIpIHtcbiAgICByZXR1cm4gbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgIHdpZHRoLFxuICAgICAgaGVpZ2h0LFxuICAgICAgdGl0bGU6IFwiRXJyb3JzIChyYXRlKVwiLFxuICAgICAgbGVmdDogW3RoaXMuZXJyb3I0eHhSYXRlTWV0cmljLCB0aGlzLmVycm9yNXh4UmF0ZU1ldHJpY10sXG4gICAgICBsZWZ0WUF4aXM6IFJhdGVBeGlzRnJvbVplcm8sXG4gICAgICBsZWZ0QW5ub3RhdGlvbnM6IHRoaXMuZXJyb3JSYXRlQW5ub3RhdGlvbnMsXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==