"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoTableMetricFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const common_1 = require("../../common");
const DynamoDbNamespace = "AWS/DynamoDB";
const ProvisionedReadCapacityUnitsMetric = "ProvisionedReadCapacityUnits";
const ProvisionedWriteCapacityUnitsMetric = "ProvisionedWriteCapacityUnits";
const ProvisionedLabel = "Provisioned";
const ConsumedLabel = "Consumed";
const ReadThrottleEventsLabel = "Read";
const WriteThrottleEventsLabel = "Write";
const SystemErrorsLabel = "System Errors";
class DynamoTableMetricFactory {
    constructor(metricFactory, props) {
        this.metricFactory = metricFactory;
        this.table = props.table;
    }
    metricProvisionedReadCapacityUnits() {
        return this.metricFactory.adaptMetric(this.table.metric(ProvisionedReadCapacityUnitsMetric, {
            label: ProvisionedLabel,
            statistic: common_1.MetricStatistic.AVERAGE,
        }));
    }
    metricProvisionedWriteCapacityUnits() {
        return this.metricFactory.adaptMetric(this.table.metric(ProvisionedWriteCapacityUnitsMetric, {
            label: ProvisionedLabel,
            statistic: common_1.MetricStatistic.AVERAGE,
        }));
    }
    metricConsumedReadCapacityUnits() {
        return this.metricFactory.createMetricMath("consumed_rcu_sum/PERIOD(consumed_rcu_sum)", {
            consumed_rcu_sum: this.table.metricConsumedReadCapacityUnits({
                statistic: common_1.MetricStatistic.SUM,
            }),
        }, ConsumedLabel);
    }
    metricConsumedWriteCapacityUnits() {
        return this.metricFactory.createMetricMath("consumed_wcu_sum/PERIOD(consumed_wcu_sum)", {
            consumed_wcu_sum: this.table.metricConsumedWriteCapacityUnits({
                statistic: common_1.MetricStatistic.SUM,
            }),
        }, ConsumedLabel);
    }
    metricReadCapacityUtilizationPercentage() {
        return this.metricFactory.createMetricMath("100*(consumed_read_cap/provisioned_read_cap)", {
            consumed_read_cap: this.metricConsumedReadCapacityUnits(),
            provisioned_read_cap: this.metricProvisionedReadCapacityUnits(),
        }, "Utilization");
    }
    metricWriteCapacityUtilizationPercentage() {
        return this.metricFactory.createMetricMath("100*(consumed_write_cap/provisioned_write_cap)", {
            consumed_write_cap: this.metricConsumedWriteCapacityUnits(),
            provisioned_write_cap: this.metricProvisionedWriteCapacityUnits(),
        }, "Utilization");
    }
    metricSearchAverageSuccessfulRequestLatencyInMillis() {
        // searches for all used operations
        return this.metricFactory.createMetricSearch('MetricName="SuccessfulRequestLatency"', {
            TableName: this.table.tableName,
            Operation: undefined,
        }, common_1.MetricStatistic.AVERAGE, DynamoDbNamespace);
    }
    metricAverageSuccessfulRequestLatencyInMillis(operation) {
        return this.metricFactory.adaptMetric(this.table.metric("SuccessfulRequestLatency", {
            statistic: common_1.MetricStatistic.AVERAGE,
            label: operation,
            dimensionsMap: {
                TableName: this.table.tableName,
                Operation: operation,
            },
        }));
    }
    metricThrottledReadRequestCount() {
        const readThrottles = this.metricFactory.adaptMetric(this.table.metric("ReadThrottleEvents", {
            statistic: common_1.MetricStatistic.SUM,
            label: ReadThrottleEventsLabel,
        }));
        return this.metricFactory.createMetricMath("FILL(readThrottles,0)", { readThrottles }, ReadThrottleEventsLabel);
    }
    metricThrottledWriteRequestCount() {
        const writeThrottles = this.metricFactory.adaptMetric(this.table.metric("WriteThrottleEvents", {
            statistic: common_1.MetricStatistic.SUM,
            label: WriteThrottleEventsLabel,
        }));
        return this.metricFactory.createMetricMath("FILL(writeThrottles,0)", { writeThrottles }, WriteThrottleEventsLabel);
    }
    /**
     * This represents the number of requests that resulted in a 500 (server error) error code.
     * It summarizes across the basic CRUD operations:
     * GetItem, BatchGetItem, Scan, Query, GetRecords, PutItem, DeleteItem, UpdateItem, BatchWriteItem
     *
     * It’s usually equal to zero.
     */
    metricSystemErrorsCount() {
        const crudOperations = [
            aws_dynamodb_1.Operation.GET_ITEM,
            aws_dynamodb_1.Operation.BATCH_GET_ITEM,
            aws_dynamodb_1.Operation.SCAN,
            aws_dynamodb_1.Operation.QUERY,
            aws_dynamodb_1.Operation.GET_RECORDS,
            aws_dynamodb_1.Operation.PUT_ITEM,
            aws_dynamodb_1.Operation.DELETE_ITEM,
            aws_dynamodb_1.Operation.UPDATE_ITEM,
            aws_dynamodb_1.Operation.BATCH_WRITE_ITEM,
        ];
        const usingMetrics = {};
        crudOperations.forEach((operation) => {
            const metric = this.table.metric("SystemErrors", {
                dimensionsMap: {
                    TableName: this.table.tableName,
                    Operation: operation,
                },
                statistic: common_1.MetricStatistic.SUM,
            });
            const metricId = "systemError" + operation;
            usingMetrics[metricId] = this.metricFactory.adaptMetric(metric);
        });
        return this.metricFactory.createMetricMath(
        // the metric is not emitted until error happens
        Object.keys(usingMetrics).join("+"), usingMetrics, SystemErrorsLabel);
    }
}
exports.DynamoTableMetricFactory = DynamoTableMetricFactory;
_a = JSII_RTTI_SYMBOL_1;
DynamoTableMetricFactory[_a] = { fqn: "cdk-monitoring-constructs.DynamoTableMetricFactory", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,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