"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EC2MetricFactory = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("../../common");
const EC2Namespace = "AWS/EC2";
/**
 * Creates a single metric for the whole ASG.
 */
class AutoScalingGroupStrategy {
    constructor(autoScalingGroup) {
        this.autoScalingGroup = autoScalingGroup;
    }
    createMetrics(metricFactory, metricName, statistic) {
        return [
            metricFactory.createMetric(metricName, statistic, undefined, resolveDimensions(this.autoScalingGroup, undefined), undefined, EC2Namespace),
        ];
    }
}
/**
 * Creates multiple metrics (one for each instance) with an optional ASG filter.
 */
class SelectedInstancesStrategy {
    constructor(instanceIds, autoScalingGroup) {
        this.instanceIds = instanceIds;
        this.autoScalingGroup = autoScalingGroup;
    }
    createMetrics(metricFactory, metricName, statistic) {
        return this.instanceIds.map((instanceId) => {
            return metricFactory.createMetric(metricName, statistic, `${metricName} (${instanceId})`, resolveDimensions(this.autoScalingGroup, instanceId), undefined, EC2Namespace);
        });
    }
}
/**
 * Creates a single metric search expression for all instances.
 */
class AllInstancesStrategy {
    createMetrics(metricFactory, metricName, statistic) {
        return [
            metricFactory.createMetricSearch(`MetricName="${metricName}"`, { InstanceId: undefined }, statistic, EC2Namespace),
        ];
    }
}
function resolveDimensions(autoScalingGroup, instanceId) {
    const dimensions = {};
    if (autoScalingGroup) {
        dimensions.AutoScalingGroupName = autoScalingGroup.autoScalingGroupName;
    }
    if (instanceId) {
        dimensions.InstanceId = instanceId;
    }
    return dimensions;
}
function resolveStrategy(props) {
    if (props.instanceIds) {
        // instance filter + optional ASG
        return new SelectedInstancesStrategy(props.instanceIds, props.autoScalingGroup);
    }
    else if (props.autoScalingGroup) {
        // ASG only
        return new AutoScalingGroupStrategy(props.autoScalingGroup);
    }
    else {
        // all instances
        return new AllInstancesStrategy();
    }
}
class EC2MetricFactory {
    constructor(metricFactory, props) {
        this.metricFactory = metricFactory;
        this.strategy = resolveStrategy(props);
    }
    /**
     * The percentage of allocated EC2 compute units that are currently in use on the instance.
     * This metric identifies the processing power required to run an application on a selected instance.
     * Depending on the instance type, tools in your operating system can show a lower percentage than
     * CloudWatch when the instance is not allocated a full processor core.
     */
    metricAverageCpuUtilisationPercent() {
        return this.createMetrics("CPUUtilization", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * Bytes read from all instance store volumes available to the instance.
     * This metric is used to determine the volume of the data the application reads from the hard disk of the instance.
     * This can be used to determine the speed of the application.
     */
    metricAverageDiskReadBytes() {
        return this.createMetrics("DiskReadBytes", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * Bytes written to all instance store volumes available to the instance.
     * This metric is used to determine the volume of the data the application writes onto the hard disk of the instance.
     * This can be used to determine the speed of the application.
     */
    metricAverageDiskWriteBytes() {
        return this.createMetrics("DiskWriteBytes", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * Completed read operations from all instance store volumes available to the instance in a specified period of time.
     */
    metricAverageDiskReadOps() {
        return this.createMetrics("DiskReadOps", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * Completed write operations to all instance store volumes available to the instance in a specified period of time.
     */
    metricAverageDiskWriteOps() {
        return this.createMetrics("DiskWriteOps", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * The number of bytes received on all network interfaces by the instance.
     * This metric identifies the volume of incoming network traffic to a single instance.
     */
    metricAverageNetworkInRateBytes() {
        return this.createMetrics("NetworkIn", common_1.MetricStatistic.AVERAGE);
    }
    /**
     * The number of bytes sent out on all network interfaces by the instance.
     * This metric identifies the volume of outgoing network traffic from a single instance.
     */
    metricAverageNetworkOutRateBytes() {
        return this.createMetrics("NetworkOut", common_1.MetricStatistic.AVERAGE);
    }
    createMetrics(metricName, statistic) {
        return this.strategy.createMetrics(this.metricFactory, metricName, statistic);
    }
}
exports.EC2MetricFactory = EC2MetricFactory;
_a = JSII_RTTI_SYMBOL_1;
EC2MetricFactory[_a] = { fqn: "cdk-monitoring-constructs.EC2MetricFactory", version: "1.27.0" };
//# sourceMappingURL=data:application/json;base64,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