"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomMonitoring = exports.AxisPosition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const common_1 = require("../../common");
const dashboard_1 = require("../../dashboard");
var AxisPosition;
(function (AxisPosition) {
    AxisPosition["LEFT"] = "left";
    AxisPosition["RIGHT"] = "right";
})(AxisPosition = exports.AxisPosition || (exports.AxisPosition = {}));
/**
 * Custom monitoring is a construct allowing you to monitor your own custom metrics.
 * The entire construct consists of metric groups.
 * Each metric group represents a single graph widget with multiple metrics.
 * Each metric inside the metric group represents a single metric inside a graph.
 * The widgets will be sized automatically to waste as little space as possible.
 */
class CustomMonitoring extends common_1.Monitoring {
    constructor(scope, props) {
        super(scope, props);
        const namingStrategy = new dashboard_1.MonitoringNamingStrategy({ ...props });
        this.title = namingStrategy.resolveHumanReadableName();
        this.description = props.description;
        this.descriptionWidgetHeight = props.descriptionWidgetHeight;
        this.height = props.height;
        const alarmFactory = this.createAlarmFactory(namingStrategy.resolveAlarmFriendlyName());
        this.customAlarmFactory = new common_1.CustomAlarmFactory(alarmFactory);
        this.anomalyDetectingAlarmFactory = new common_1.AnomalyDetectingAlarmFactory(alarmFactory);
        this.metricGroups = props.metricGroups.map((metricGroup) => {
            const metricGroupWithAnnotation = {
                metricGroup,
                annotations: [],
                rightAnnotations: [],
                titleAddons: [],
            };
            if (metricGroup.horizontalAnnotations) {
                metricGroupWithAnnotation.annotations.push(...metricGroup.horizontalAnnotations);
            }
            if (metricGroup.horizontalRightAnnotations) {
                metricGroupWithAnnotation.rightAnnotations.push(...metricGroup.horizontalRightAnnotations);
            }
            metricGroup.metrics.forEach((metric) => {
                if (this.hasAlarm(metric) && this.hasAnomalyDetection(metric)) {
                    throw new Error("Adding both a regular alarm and an anomaly detection alarm at the same time is not supported");
                }
                if (this.hasAlarm(metric)) {
                    this.setupAlarm(metricGroupWithAnnotation, metric);
                }
                else if (this.hasAnomalyDetection(metric)) {
                    this.setupAnomalyDetectionAlarm(metricGroupWithAnnotation, metric);
                }
            });
            return metricGroupWithAnnotation;
        });
        props.useCreatedAlarms?.consume(this.createdAlarms());
    }
    summaryWidgets() {
        return this.getAllWidgets(true);
    }
    widgets() {
        return this.getAllWidgets(false);
    }
    getAllWidgets(summary) {
        const filteredMetricGroups = summary
            ? this.metricGroups.filter((group) => group.metricGroup.important ?? false)
            : this.metricGroups;
        if (filteredMetricGroups.length < 1) {
            // short-circuit if there are no metrics specified
            return [];
        }
        const rows = [];
        // header and description
        rows.push(new aws_cloudwatch_1.Row(new dashboard_1.MonitoringHeaderWidget({ title: this.title })));
        if (this.description && !summary) {
            rows.push(new aws_cloudwatch_1.Row(this.createDescriptionWidget(this.description, this.descriptionWidgetHeight)));
        }
        // graphs
        rows.push(new aws_cloudwatch_1.Row(...this.createCustomMetricGroupWidgets(filteredMetricGroups, summary)));
        return rows;
    }
    createDescriptionWidget(markdown, descriptionWidgetHeight) {
        return new aws_cloudwatch_1.TextWidget({
            markdown,
            width: common_1.FullWidth,
            height: descriptionWidgetHeight ?? 1,
        });
    }
    createCustomMetricGroupWidgets(annotatedGroups, summary) {
        const widgets = [];
        const metricGroupWidgetWidth = common_1.recommendedWidgetWidth(annotatedGroups.length);
        const metricGroupWidgetHeightDefault = summary
            ? common_1.DefaultSummaryWidgetHeight
            : common_1.DefaultGraphWidgetHeight;
        const metricGroupWidgetHeight = this.height ?? metricGroupWidgetHeightDefault;
        annotatedGroups.forEach((annotatedGroup) => {
            const metrics = annotatedGroup.metricGroup.metrics;
            const left = this.toMetrics(metrics.filter((metric) => (metric.position ?? AxisPosition.LEFT) == AxisPosition.LEFT));
            const right = this.toMetrics(metrics.filter((metric) => (metric.position ?? AxisPosition.LEFT) ==
                AxisPosition.RIGHT));
            const hasOneMetricOnly = metrics.length === 1;
            const hasAnomalyDetection = metrics.filter((metric) => this.hasAnomalyDetection(metric)).length > 0;
            const useAnomalyDetectionWidget = hasOneMetricOnly && hasAnomalyDetection;
            let title = annotatedGroup.metricGroup.title;
            if (annotatedGroup.titleAddons.length > 0) {
                title = `${title} (${annotatedGroup.titleAddons.join(", ")})`;
            }
            const graphWidgetProps = {
                title,
                width: metricGroupWidgetWidth,
                height: metricGroupWidgetHeight,
                left,
                right,
                leftAnnotations: annotatedGroup.annotations,
                rightAnnotations: annotatedGroup.rightAnnotations,
                leftYAxis: annotatedGroup.metricGroup.graphWidgetAxis,
                rightYAxis: annotatedGroup.metricGroup.graphWidgetRightAxis,
                legendPosition: annotatedGroup.metricGroup.graphWidgetLegend,
            };
            const widget = useAnomalyDetectionWidget
                ? new AnomalyDetectionGraphWidget(graphWidgetProps)
                : common_1.createGraphWidget(annotatedGroup.metricGroup.graphWidgetType ?? common_1.GraphWidgetType.LINE, graphWidgetProps);
            widgets.push(widget);
        });
        return widgets;
    }
    toMetrics(metrics) {
        const metricFactory = this.createMetricFactory();
        return metrics.map((metric) => {
            if (this.hasAlarm(metric)) {
                // metric with alarm
                return metricFactory.adaptMetricPreservingPeriod(metric.metric);
            }
            else if (this.hasAnomalyDetection(metric)) {
                // metric with anomaly detection
                return metricFactory.createMetricAnomalyDetection(metric.metric, metric.anomalyDetectionStandardDeviationToRender, `Expected (stdev = ${metric.anomalyDetectionStandardDeviationToRender})`, undefined, 
                // needs to be unique in the whole widget and start with lowercase
                AnomalyDetectionMetricIdPrefix +
                    common_1.getHashForMetricExpressionId(metric.alarmFriendlyName), 
                // preserve the most specific metric period
                metric.period ?? metric.metric.period);
            }
            else if (this.isSearch(metric)) {
                // metric search
                return metricFactory.createMetricSearch(metric.searchQuery, metric.dimensionsMap, metric.statistic, metric.namespace, metric.label, metric.period);
            }
            else {
                // general metric
                return metricFactory.adaptMetricPreservingPeriod(metric);
            }
        });
    }
    hasAlarm(metric) {
        // type guard
        return metric.addAlarm !== undefined;
    }
    hasAnomalyDetection(metric) {
        // type guard
        return (metric
            .anomalyDetectionStandardDeviationToRender !== undefined);
    }
    isSearch(metric) {
        // type guard
        return metric.searchQuery !== undefined;
    }
    setupAlarm(metricGroup, metric) {
        if (this.isSearch(metric)) {
            throw new Error("Alarming on search queries is not supported by CloudWatch");
        }
        for (const disambiguator in metric.addAlarm) {
            const alarmProps = metric.addAlarm[disambiguator];
            const createdAlarm = this.customAlarmFactory.addCustomAlarm(metric.metric, metric.alarmFriendlyName, disambiguator, alarmProps);
            const targetAnnotations = (metric.position ?? AxisPosition.LEFT) == AxisPosition.LEFT
                ? metricGroup.annotations
                : metricGroup.rightAnnotations;
            targetAnnotations.push(createdAlarm.annotation);
            this.addAlarm(createdAlarm);
        }
    }
    setupAnomalyDetectionAlarm(metricGroup, metric) {
        if (this.isSearch(metric)) {
            throw new Error("Alarming on search queries is not supported by CloudWatch");
        }
        const alarmStDevs = new Set();
        const metricFactory = this.createMetricFactory();
        for (const disambiguator in metric.addAlarmOnAnomaly) {
            const alarmProps = metric.addAlarmOnAnomaly[disambiguator];
            if (alarmProps.alarmWhenAboveTheBand ||
                alarmProps.alarmWhenBelowTheBand) {
                const anomalyMetric = metricFactory.createMetricAnomalyDetection(
                // Because the metric was provided to us, we use metricFactory.overrideNamespace() to
                // confirm it aligns with any namespace overrides requested for this MonitoringFacade
                metricFactory.adaptMetricPreservingPeriod(metric.metric), alarmProps.standardDeviationForAlarm, `Band (stdev ${alarmProps.standardDeviationForAlarm})`, undefined, 
                // expression ID needs to be unique across the whole widget; needs to start with a lowercase letter
                AnomalyDetectionAlarmIdPrefix +
                    common_1.getHashForMetricExpressionId(metric.alarmFriendlyName + "_" + disambiguator), 
                // preserve the most-specific metric period
                metric.period ?? metric.metric.period);
                const createdAlarm = this.anomalyDetectingAlarmFactory.addAlarmWhenOutOfBand(anomalyMetric, metric.alarmFriendlyName, disambiguator, alarmProps);
                // no need to add annotation since the bands are rendered automatically
                this.addAlarm(createdAlarm);
                alarmStDevs.add(alarmProps.standardDeviationForAlarm);
            }
        }
        if (alarmStDevs.size > 0) {
            const alarmStDevsString = Array.from(alarmStDevs).sort().join(", ");
            metricGroup.titleAddons.push(`alarms with stdev ${alarmStDevsString}`);
        }
    }
}
exports.CustomMonitoring = CustomMonitoring;
_a = JSII_RTTI_SYMBOL_1;
CustomMonitoring[_a] = { fqn: "cdk-monitoring-constructs.CustomMonitoring", version: "1.27.0" };
const AnomalyDetectionAlarmIdPrefix = "alarm_";
const AnomalyDetectionMetricIdPrefix = "anomaly_";
const AnomalyBandMetricIdSuffix = "_band";
/**
 * INTERNAL - PLEASE DO NOT USE
 * This is a hacky solution to make band visible in GraphWidget (default widget only renders lines, not the band).
 * The class makes assumptions about the internal JSON structure but found no other way :(.
 * Ideally, we want to remove this hack once the anomaly detection rendering in CDK gets improved
 */
class AnomalyDetectionGraphWidget extends aws_cloudwatch_1.GraphWidget {
    constructor(props) {
        super(props);
    }
    toJson() {
        const json = super.toJson();
        if (json.length !== 1 || !json?.[0]?.properties?.metrics) {
            throw new Error("The JSON is expected to have exactly one element with properties.metrics property.");
        }
        const metrics = json[0].properties.metrics;
        if (metrics.length < 2) {
            throw new Error("The number of metrics must be at least two (metric + anomaly detection math).");
        }
        const anomalyDetectionMetricPart = metrics[0]?.value;
        if (!anomalyDetectionMetricPart ||
            anomalyDetectionMetricPart.length !== 1) {
            throw new Error("First metric must be a math expression.");
        }
        const evaluatedMetricPart = metrics[1]?.value;
        if (!evaluatedMetricPart ||
            evaluatedMetricPart.length < 1 ||
            !evaluatedMetricPart[evaluatedMetricPart.length - 1].id) {
            throw new Error("Second metric must have an ID.");
        }
        // band rendering requires ID to be set
        anomalyDetectionMetricPart[0].id =
            evaluatedMetricPart[evaluatedMetricPart.length - 1].id +
                AnomalyBandMetricIdSuffix;
        // band rendering requires the evaluated metric to be visible
        evaluatedMetricPart[evaluatedMetricPart.length - 1].visible = true;
        return json;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiQ3VzdG9tTW9uaXRvcmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIkN1c3RvbU1vbml0b3JpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSwrREFXb0M7QUFFcEMseUNBaUJzQjtBQUN0QiwrQ0FHeUI7QUFFekIsSUFBWSxZQUdYO0FBSEQsV0FBWSxZQUFZO0lBQ3RCLDZCQUFhLENBQUE7SUFDYiwrQkFBZSxDQUFBO0FBQ2pCLENBQUMsRUFIVyxZQUFZLEdBQVosb0JBQVksS0FBWixvQkFBWSxRQUd2QjtBQXNMRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLG1CQUFVO0lBUzlDLFlBQVksS0FBc0IsRUFBRSxLQUE0QjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXBCLE1BQU0sY0FBYyxHQUFHLElBQUksb0NBQXdCLENBQUMsRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLEtBQUssR0FBRyxjQUFjLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztRQUV2RCxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxDQUFDLHVCQUF1QixHQUFHLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQztRQUM3RCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFFM0IsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUMxQyxjQUFjLENBQUMsd0JBQXdCLEVBQUUsQ0FDMUMsQ0FBQztRQUNGLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLDJCQUFrQixDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxJQUFJLHFDQUE0QixDQUNsRSxZQUFZLENBQ2IsQ0FBQztRQUVGLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUN6RCxNQUFNLHlCQUF5QixHQUFxQztnQkFDbEUsV0FBVztnQkFDWCxXQUFXLEVBQUUsRUFBRTtnQkFDZixnQkFBZ0IsRUFBRSxFQUFFO2dCQUNwQixXQUFXLEVBQUUsRUFBRTthQUNoQixDQUFDO1lBRUYsSUFBSSxXQUFXLENBQUMscUJBQXFCLEVBQUU7Z0JBQ3JDLHlCQUF5QixDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQ3hDLEdBQUcsV0FBVyxDQUFDLHFCQUFxQixDQUNyQyxDQUFDO2FBQ0g7WUFDRCxJQUFJLFdBQVcsQ0FBQywwQkFBMEIsRUFBRTtnQkFDMUMseUJBQXlCLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUM3QyxHQUFHLFdBQVcsQ0FBQywwQkFBMEIsQ0FDMUMsQ0FBQzthQUNIO1lBRUQsV0FBVyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDckMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsRUFBRTtvQkFDN0QsTUFBTSxJQUFJLEtBQUssQ0FDYiw4RkFBOEYsQ0FDL0YsQ0FBQztpQkFDSDtnQkFFRCxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7b0JBQ3pCLElBQUksQ0FBQyxVQUFVLENBQUMseUJBQXlCLEVBQUUsTUFBTSxDQUFDLENBQUM7aUJBQ3BEO3FCQUFNLElBQUksSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxFQUFFO29CQUMzQyxJQUFJLENBQUMsMEJBQTBCLENBQUMseUJBQXlCLEVBQUUsTUFBTSxDQUFDLENBQUM7aUJBQ3BFO1lBQ0gsQ0FBQyxDQUFDLENBQUM7WUFFSCxPQUFPLHlCQUF5QixDQUFDO1FBQ25DLENBQUMsQ0FBQyxDQUFDO1FBRUgsS0FBSyxDQUFDLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQsY0FBYztRQUNaLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQsT0FBTztRQUNMLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRU8sYUFBYSxDQUFDLE9BQWdCO1FBQ3BDLE1BQU0sb0JBQW9CLEdBQUcsT0FBTztZQUNsQyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQ3RCLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQ2hEO1lBQ0gsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUM7UUFFdEIsSUFBSSxvQkFBb0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ25DLGtEQUFrRDtZQUNsRCxPQUFPLEVBQUUsQ0FBQztTQUNYO1FBRUQsTUFBTSxJQUFJLEdBQVUsRUFBRSxDQUFDO1FBRXZCLHlCQUF5QjtRQUN6QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksb0JBQUcsQ0FBQyxJQUFJLGtDQUFzQixDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0RSxJQUFJLElBQUksQ0FBQyxXQUFXLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDaEMsSUFBSSxDQUFDLElBQUksQ0FDUCxJQUFJLG9CQUFHLENBQ0wsSUFBSSxDQUFDLHVCQUF1QixDQUMxQixJQUFJLENBQUMsV0FBVyxFQUNoQixJQUFJLENBQUMsdUJBQXVCLENBQzdCLENBQ0YsQ0FDRixDQUFDO1NBQ0g7UUFFRCxTQUFTO1FBQ1QsSUFBSSxDQUFDLElBQUksQ0FDUCxJQUFJLG9CQUFHLENBQ0wsR0FBRyxJQUFJLENBQUMsOEJBQThCLENBQUMsb0JBQW9CLEVBQUUsT0FBTyxDQUFDLENBQ3RFLENBQ0YsQ0FBQztRQUVGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVPLHVCQUF1QixDQUM3QixRQUFnQixFQUNoQix1QkFBZ0M7UUFFaEMsT0FBTyxJQUFJLDJCQUFVLENBQUM7WUFDcEIsUUFBUTtZQUNSLEtBQUssRUFBRSxrQkFBUztZQUNoQixNQUFNLEVBQUUsdUJBQXVCLElBQUksQ0FBQztTQUNyQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sOEJBQThCLENBQ3BDLGVBQW1ELEVBQ25ELE9BQWdCO1FBRWhCLE1BQU0sT0FBTyxHQUFjLEVBQUUsQ0FBQztRQUM5QixNQUFNLHNCQUFzQixHQUFHLCtCQUFzQixDQUNuRCxlQUFlLENBQUMsTUFBTSxDQUN2QixDQUFDO1FBQ0YsTUFBTSw4QkFBOEIsR0FBRyxPQUFPO1lBQzVDLENBQUMsQ0FBQyxtQ0FBMEI7WUFDNUIsQ0FBQyxDQUFDLGlDQUF3QixDQUFDO1FBQzdCLE1BQU0sdUJBQXVCLEdBQzNCLElBQUksQ0FBQyxNQUFNLElBQUksOEJBQThCLENBQUM7UUFFaEQsZUFBZSxDQUFDLE9BQU8sQ0FBQyxDQUFDLGNBQWMsRUFBRSxFQUFFO1lBQ3pDLE1BQU0sT0FBTyxHQUFHLGNBQWMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDO1lBQ25ELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQ3pCLE9BQU8sQ0FBQyxNQUFNLENBQ1osQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNULENBQUUsTUFBYyxDQUFDLFFBQVEsSUFBSSxZQUFZLENBQUMsSUFBSSxDQUFDLElBQUksWUFBWSxDQUFDLElBQUksQ0FDdkUsQ0FDRixDQUFDO1lBQ0YsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FDMUIsT0FBTyxDQUFDLE1BQU0sQ0FDWixDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ1QsQ0FBRSxNQUFjLENBQUMsUUFBUSxJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUM7Z0JBQy9DLFlBQVksQ0FBQyxLQUFLLENBQ3JCLENBQ0YsQ0FBQztZQUNGLE1BQU0sZ0JBQWdCLEdBQUcsT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUM7WUFDOUMsTUFBTSxtQkFBbUIsR0FDdkIsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztZQUMxRSxNQUFNLHlCQUF5QixHQUFHLGdCQUFnQixJQUFJLG1CQUFtQixDQUFDO1lBQzFFLElBQUksS0FBSyxHQUFHLGNBQWMsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDO1lBRTdDLElBQUksY0FBYyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUN6QyxLQUFLLEdBQUcsR0FBRyxLQUFLLEtBQUssY0FBYyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQzthQUMvRDtZQUVELE1BQU0sZ0JBQWdCLEdBQXFCO2dCQUN6QyxLQUFLO2dCQUNMLEtBQUssRUFBRSxzQkFBc0I7Z0JBQzdCLE1BQU0sRUFBRSx1QkFBdUI7Z0JBQy9CLElBQUk7Z0JBQ0osS0FBSztnQkFDTCxlQUFlLEVBQUUsY0FBYyxDQUFDLFdBQVc7Z0JBQzNDLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxnQkFBZ0I7Z0JBQ2pELFNBQVMsRUFBRSxjQUFjLENBQUMsV0FBVyxDQUFDLGVBQWU7Z0JBQ3JELFVBQVUsRUFBRSxjQUFjLENBQUMsV0FBVyxDQUFDLG9CQUFvQjtnQkFDM0QsY0FBYyxFQUFFLGNBQWMsQ0FBQyxXQUFXLENBQUMsaUJBQWlCO2FBQzdELENBQUM7WUFFRixNQUFNLE1BQU0sR0FBRyx5QkFBeUI7Z0JBQ3RDLENBQUMsQ0FBQyxJQUFJLDJCQUEyQixDQUFDLGdCQUFnQixDQUFDO2dCQUNuRCxDQUFDLENBQUMsMEJBQWlCLENBQ2YsY0FBYyxDQUFDLFdBQVcsQ0FBQyxlQUFlLElBQUksd0JBQWUsQ0FBQyxJQUFJLEVBQ2xFLGdCQUFnQixDQUNqQixDQUFDO1lBRU4sT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN2QixDQUFDLENBQUMsQ0FBQztRQUVILE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFTyxTQUFTLENBQUMsT0FBdUI7UUFDdkMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFakQsT0FBTyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7WUFDNUIsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUN6QixvQkFBb0I7Z0JBQ3BCLE9BQU8sYUFBYSxDQUFDLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQzthQUNqRTtpQkFBTSxJQUFJLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDM0MsZ0NBQWdDO2dCQUNoQyxPQUFPLGFBQWEsQ0FBQyw0QkFBNEIsQ0FDL0MsTUFBTSxDQUFDLE1BQU0sRUFDYixNQUFNLENBQUMseUNBQXlDLEVBQ2hELHFCQUFxQixNQUFNLENBQUMseUNBQXlDLEdBQUcsRUFDeEUsU0FBUztnQkFDVCxrRUFBa0U7Z0JBQ2xFLDhCQUE4QjtvQkFDNUIscUNBQTRCLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDO2dCQUN4RCwyQ0FBMkM7Z0JBQzNDLE1BQU0sQ0FBQyxNQUFNLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQ3RDLENBQUM7YUFDSDtpQkFBTSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ2hDLGdCQUFnQjtnQkFDaEIsT0FBTyxhQUFhLENBQUMsa0JBQWtCLENBQ3JDLE1BQU0sQ0FBQyxXQUFXLEVBQ2xCLE1BQU0sQ0FBQyxhQUFhLEVBQ3BCLE1BQU0sQ0FBQyxTQUFTLEVBQ2hCLE1BQU0sQ0FBQyxTQUFTLEVBQ2hCLE1BQU0sQ0FBQyxLQUFLLEVBQ1osTUFBTSxDQUFDLE1BQU0sQ0FDZCxDQUFDO2FBQ0g7aUJBQU07Z0JBQ0wsaUJBQWlCO2dCQUNqQixPQUFPLGFBQWEsQ0FBQywyQkFBMkIsQ0FBQyxNQUFNLENBQUMsQ0FBQzthQUMxRDtRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLFFBQVEsQ0FBQyxNQUFvQjtRQUNuQyxhQUFhO1FBQ2IsT0FBUSxNQUFnQyxDQUFDLFFBQVEsS0FBSyxTQUFTLENBQUM7SUFDbEUsQ0FBQztJQUVPLG1CQUFtQixDQUN6QixNQUFvQjtRQUVwQixhQUFhO1FBQ2IsT0FBTyxDQUNKLE1BQTJDO2FBQ3pDLHlDQUF5QyxLQUFLLFNBQVMsQ0FDM0QsQ0FBQztJQUNKLENBQUM7SUFFTyxRQUFRLENBQUMsTUFBb0I7UUFDbkMsYUFBYTtRQUNiLE9BQVEsTUFBNkIsQ0FBQyxXQUFXLEtBQUssU0FBUyxDQUFDO0lBQ2xFLENBQUM7SUFFTyxVQUFVLENBQ2hCLFdBQTZDLEVBQzdDLE1BQTZCO1FBRTdCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUNiLDJEQUEyRCxDQUM1RCxDQUFDO1NBQ0g7UUFFRCxLQUFLLE1BQU0sYUFBYSxJQUFJLE1BQU0sQ0FBQyxRQUFRLEVBQUU7WUFDM0MsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUNsRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsY0FBYyxDQUN6RCxNQUFNLENBQUMsTUFBTSxFQUNiLE1BQU0sQ0FBQyxpQkFBaUIsRUFDeEIsYUFBYSxFQUNiLFVBQVUsQ0FDWCxDQUFDO1lBQ0YsTUFBTSxpQkFBaUIsR0FDckIsQ0FBQyxNQUFNLENBQUMsUUFBUSxJQUFJLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxZQUFZLENBQUMsSUFBSTtnQkFDekQsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxXQUFXO2dCQUN6QixDQUFDLENBQUMsV0FBVyxDQUFDLGdCQUFnQixDQUFDO1lBQ25DLGlCQUFpQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDaEQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM3QjtJQUNILENBQUM7SUFFTywwQkFBMEIsQ0FDaEMsV0FBNkMsRUFDN0MsTUFBd0M7UUFFeEMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQ2IsMkRBQTJELENBQzVELENBQUM7U0FDSDtRQUVELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxFQUFVLENBQUM7UUFDdEMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFFakQsS0FBSyxNQUFNLGFBQWEsSUFBSSxNQUFNLENBQUMsaUJBQWlCLEVBQUU7WUFDcEQsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLGlCQUFpQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQzNELElBQ0UsVUFBVSxDQUFDLHFCQUFxQjtnQkFDaEMsVUFBVSxDQUFDLHFCQUFxQixFQUNoQztnQkFDQSxNQUFNLGFBQWEsR0FBRyxhQUFhLENBQUMsNEJBQTRCO2dCQUM5RCxxRkFBcUY7Z0JBQ3JGLHFGQUFxRjtnQkFDckYsYUFBYSxDQUFDLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFDeEQsVUFBVSxDQUFDLHlCQUF5QixFQUNwQyxlQUFlLFVBQVUsQ0FBQyx5QkFBeUIsR0FBRyxFQUN0RCxTQUFTO2dCQUNULG1HQUFtRztnQkFDbkcsNkJBQTZCO29CQUMzQixxQ0FBNEIsQ0FDMUIsTUFBTSxDQUFDLGlCQUFpQixHQUFHLEdBQUcsR0FBRyxhQUFhLENBQy9DO2dCQUNILDJDQUEyQztnQkFDM0MsTUFBTSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FDdEMsQ0FBQztnQkFFRixNQUFNLFlBQVksR0FDaEIsSUFBSSxDQUFDLDRCQUE0QixDQUFDLHFCQUFxQixDQUNyRCxhQUFhLEVBQ2IsTUFBTSxDQUFDLGlCQUFpQixFQUN4QixhQUFhLEVBQ2IsVUFBVSxDQUNYLENBQUM7Z0JBRUosdUVBQXVFO2dCQUN2RSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUM1QixXQUFXLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO2FBQ3ZEO1NBQ0Y7UUFFRCxJQUFJLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxFQUFFO1lBQ3hCLE1BQU0saUJBQWlCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEUsV0FBVyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMscUJBQXFCLGlCQUFpQixFQUFFLENBQUMsQ0FBQztTQUN4RTtJQUNILENBQUM7O0FBcFVILDRDQXFVQzs7O0FBRUQsTUFBTSw2QkFBNkIsR0FBRyxRQUFRLENBQUM7QUFDL0MsTUFBTSw4QkFBOEIsR0FBRyxVQUFVLENBQUM7QUFDbEQsTUFBTSx5QkFBeUIsR0FBRyxPQUFPLENBQUM7QUFFMUM7Ozs7O0dBS0c7QUFDSCxNQUFNLDJCQUE0QixTQUFRLDRCQUFXO0lBQ25ELFlBQVksS0FBdUI7UUFDakMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2YsQ0FBQztJQUVELE1BQU07UUFDSixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDNUIsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLFVBQVUsRUFBRSxPQUFPLEVBQUU7WUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FDYixvRkFBb0YsQ0FDckYsQ0FBQztTQUNIO1FBQ0QsTUFBTSxPQUFPLEdBQVUsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUM7UUFDbEQsSUFBSSxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUNiLCtFQUErRSxDQUNoRixDQUFDO1NBQ0g7UUFDRCxNQUFNLDBCQUEwQixHQUFVLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLENBQUM7UUFDNUQsSUFDRSxDQUFDLDBCQUEwQjtZQUMzQiwwQkFBMEIsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUN2QztZQUNBLE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLENBQUMsQ0FBQztTQUM1RDtRQUNELE1BQU0sbUJBQW1CLEdBQVUsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQztRQUNyRCxJQUNFLENBQUMsbUJBQW1CO1lBQ3BCLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDO1lBQzlCLENBQUMsbUJBQW1CLENBQUMsbUJBQW1CLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFDdkQ7WUFDQSxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7U0FDbkQ7UUFDRCx1Q0FBdUM7UUFDdkMsMEJBQTBCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5QixtQkFBbUIsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDdEQseUJBQXlCLENBQUM7UUFDNUIsNkRBQTZEO1FBQzdELG1CQUFtQixDQUFDLG1CQUFtQixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxPQUFPLEdBQUcsSUFBSSxDQUFDO1FBQ25FLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7XG4gIERpbWVuc2lvbnNNYXAsXG4gIEdyYXBoV2lkZ2V0LFxuICBHcmFwaFdpZGdldFByb3BzLFxuICBIb3Jpem9udGFsQW5ub3RhdGlvbixcbiAgSU1ldHJpYyxcbiAgSVdpZGdldCxcbiAgTGVnZW5kUG9zaXRpb24sXG4gIFJvdyxcbiAgVGV4dFdpZGdldCxcbiAgWUF4aXNQcm9wcyxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoXCI7XG5cbmltcG9ydCB7XG4gIEFub21hbHlEZXRlY3RpbmdBbGFybUZhY3RvcnksXG4gIEFub21hbHlEZXRlY3Rpb25UaHJlc2hvbGQsXG4gIEJhc2VNb25pdG9yaW5nUHJvcHMsXG4gIGNyZWF0ZUdyYXBoV2lkZ2V0LFxuICBDdXN0b21BbGFybUZhY3RvcnksXG4gIEN1c3RvbVRocmVzaG9sZCxcbiAgRGVmYXVsdEdyYXBoV2lkZ2V0SGVpZ2h0LFxuICBEZWZhdWx0U3VtbWFyeVdpZGdldEhlaWdodCxcbiAgRnVsbFdpZHRoLFxuICBnZXRIYXNoRm9yTWV0cmljRXhwcmVzc2lvbklkLFxuICBHcmFwaFdpZGdldFR5cGUsXG4gIE1ldHJpY1N0YXRpc3RpYyxcbiAgTWV0cmljV2l0aEFsYXJtU3VwcG9ydCxcbiAgTW9uaXRvcmluZyxcbiAgTW9uaXRvcmluZ1Njb3BlLFxuICByZWNvbW1lbmRlZFdpZGdldFdpZHRoLFxufSBmcm9tIFwiLi4vLi4vY29tbW9uXCI7XG5pbXBvcnQge1xuICBNb25pdG9yaW5nSGVhZGVyV2lkZ2V0LFxuICBNb25pdG9yaW5nTmFtaW5nU3RyYXRlZ3ksXG59IGZyb20gXCIuLi8uLi9kYXNoYm9hcmRcIjtcblxuZXhwb3J0IGVudW0gQXhpc1Bvc2l0aW9uIHtcbiAgTEVGVCA9IFwibGVmdFwiLFxuICBSSUdIVCA9IFwicmlnaHRcIixcbn1cblxuLyoqXG4gKiBDdXN0b20gbWV0cmljIHdpdGggYW4gYWxhcm0gZGVmaW5lZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDdXN0b21NZXRyaWNXaXRoQWxhcm0ge1xuICAvKipcbiAgICogbWV0cmljIHRvIGFsYXJtIG9uXG4gICAqL1xuICByZWFkb25seSBtZXRyaWM6IE1ldHJpY1dpdGhBbGFybVN1cHBvcnQ7XG4gIC8qKlxuICAgKiBhbGFybSBmcmllbmRseSBuYW1lXG4gICAqL1xuICByZWFkb25seSBhbGFybUZyaWVuZGx5TmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogYWxhcm0gZGVmaW5pdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGFkZEFsYXJtOiBSZWNvcmQ8c3RyaW5nLCBDdXN0b21UaHJlc2hvbGQ+O1xuICAvKipcbiAgICogYXhpcyAocmlnaHQgb3IgbGVmdCkgb24gd2hpY2ggdG8gZ3JhcGggbWV0cmljXG4gICAqIGRlZmF1bHQ6IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAqL1xuICByZWFkb25seSBwb3NpdGlvbj86IEF4aXNQb3NpdGlvbjtcbn1cblxuLyoqXG4gKiBDdXN0b20gbWV0cmljIHdpdGggYW5vbWFseSBkZXRlY3Rpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tTWV0cmljV2l0aEFub21hbHlEZXRlY3Rpb24ge1xuICAvKipcbiAgICogbWV0cmljIHRvIGFsYXJtIG9uXG4gICAqL1xuICByZWFkb25seSBtZXRyaWM6IE1ldHJpY1dpdGhBbGFybVN1cHBvcnQ7XG4gIC8qKlxuICAgKiBhbm9tYWx5IGRldGVjdGlvbiBwZXJpb2RcbiAgICogQGRlZmF1bHQgLSBtZXRyaWMgcGVyaW9kIChpZiBkZWZpbmVkKSBvciBnbG9iYWwgZGVmYXVsdFxuICAgKi9cbiAgcmVhZG9ubHkgcGVyaW9kPzogRHVyYXRpb247XG4gIC8qKlxuICAgKiBhbGFybSBmcmllbmRseSBuYW1lXG4gICAqL1xuICByZWFkb25seSBhbGFybUZyaWVuZGx5TmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogc3RhbmRhcmQgZGV2aWF0aW9uIGZvciB0aGUgYW5vbWFseSBkZXRlY3Rpb24gdG8gYmUgcmVuZGVyZWQgb24gdGhlIGdyYXBoIHdpZGdldFxuICAgKi9cbiAgcmVhZG9ubHkgYW5vbWFseURldGVjdGlvblN0YW5kYXJkRGV2aWF0aW9uVG9SZW5kZXI6IG51bWJlcjtcbiAgLyoqXG4gICAqIGFkZHMgYWxhcm0gb24gYSBkZXRlY3RlZCBhbm9tYWx5XG4gICAqL1xuICByZWFkb25seSBhZGRBbGFybU9uQW5vbWFseT86IFJlY29yZDxzdHJpbmcsIEFub21hbHlEZXRlY3Rpb25UaHJlc2hvbGQ+O1xufVxuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgc2VhcmNoLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU1ldHJpY1NlYXJjaCB7XG4gIC8qKlxuICAgKiBtZXRyaWMgbmFtZXNwYWNlXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZXNwYWNlPzogc3RyaW5nO1xuICAvKipcbiAgICogc2VhcmNoIHF1ZXJ5IChjYW4gYmUgZW1wdHkpXG4gICAqL1xuICByZWFkb25seSBzZWFyY2hRdWVyeTogc3RyaW5nO1xuICAvKipcbiAgICogY3VzdG9tIGxhYmVsIGZvciB0aGUgbWV0cmljc1xuICAgKiBAZGVmYXVsdCAtIFwiIFwiXG4gICAqL1xuICByZWFkb25seSBsYWJlbD86IHN0cmluZztcbiAgLyoqXG4gICAqIHNlYXJjaCBkaW1lbnNpb25zIChjYW4gYmUgZW1wdHkpXG4gICAqL1xuICByZWFkb25seSBkaW1lbnNpb25zTWFwOiBEaW1lbnNpb25zTWFwO1xuICAvKipcbiAgICogbWV0cmljIHN0YXRpc3RpY1xuICAgKi9cbiAgcmVhZG9ubHkgc3RhdGlzdGljOiBNZXRyaWNTdGF0aXN0aWM7XG4gIC8qKlxuICAgKiBtZXRyaWMgcGVyaW9kXG4gICAqIEBkZWZhdWx0IC0gZ2xvYmFsIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IHBlcmlvZD86IER1cmF0aW9uO1xuICAvKipcbiAgICogYXhpcyAocmlnaHQgb3IgbGVmdCkgb24gd2hpY2ggdG8gZ3JhcGggbWV0cmljXG4gICAqIGRlZmF1bHQ6IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAqL1xuICByZWFkb25seSBwb3NpdGlvbj86IEF4aXNQb3NpdGlvbjtcbn1cblxuLyoqXG4gKiBFYWNoIGN1c3RvbSBtZXRyaWMgY2FuIGJlIG9mIGZvdXIgdHlwZXM6XG4gKiBAc2VlIE1ldHJpY1dpdGhBbGFybVN1cHBvcnQgZm9yIGEgc3RhbmRhcmQgbWV0cmljXG4gKiBAc2VlIEN1c3RvbU1ldHJpY1NlYXJjaCBmb3IgYSBzZWFyY2hcbiAqIEBzZWUgQ3VzdG9tTWV0cmljV2l0aEFsYXJtIGZvciBhIG1ldHJpYyB3aXRoIGFuIGFsYXJtXG4gKiBAc2VlIEN1c3RvbU1ldHJpY1dpdGhBbm9tYWx5RGV0ZWN0aW9uIGZvciBhIG1ldHJpYyB3aXRoIGFuIGFub21hbHkgZGV0ZWN0aW5nIGFsYXJtXG4gKi9cbmV4cG9ydCB0eXBlIEN1c3RvbU1ldHJpYyA9XG4gIHwgTWV0cmljV2l0aEFsYXJtU3VwcG9ydFxuICB8IEN1c3RvbU1ldHJpY1NlYXJjaFxuICB8IEN1c3RvbU1ldHJpY1dpdGhBbGFybVxuICB8IEN1c3RvbU1ldHJpY1dpdGhBbm9tYWx5RGV0ZWN0aW9uO1xuXG4vKipcbiAqIEN1c3RvbSBtZXRyaWMgZ3JvdXAgcmVwcmVzZW50cyBhIHNpbmdsZSB3aWRnZXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tTWV0cmljR3JvdXAge1xuICAvKipcbiAgICogdGl0bGUgb2YgdGhlIHdob2xlIGdyb3VwXG4gICAqL1xuICByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICAvKipcbiAgICogdHlwZSBvZiB0aGUgd2lkZ2V0XG4gICAqIEBkZWZhdWx0IGxpbmVcbiAgICovXG4gIHJlYWRvbmx5IGdyYXBoV2lkZ2V0VHlwZT86IEdyYXBoV2lkZ2V0VHlwZTtcbiAgLyoqXG4gICAqIG9wdGlvbmFsIGF4aXNcbiAgICogQGRlZmF1bHQgdW5kZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBncmFwaFdpZGdldEF4aXM/OiBZQXhpc1Byb3BzO1xuICAvKipcbiAgICogb3B0aW9uYWwgcmlnaHQgYXhpc1xuICAgKiBAZGVmYXVsdCB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IGdyYXBoV2lkZ2V0UmlnaHRBeGlzPzogWUF4aXNQcm9wcztcbiAgLyoqXG4gICAqIGdyYXBoIHdpZGdldCBsZWdlbmRcbiAgICogQGRlZmF1bHQgQk9UVE9NXG4gICAqL1xuICByZWFkb25seSBncmFwaFdpZGdldExlZ2VuZD86IExlZ2VuZFBvc2l0aW9uO1xuICAvKipcbiAgICogRmxhZyBpbmRpY2F0aW5nLCB3aGV0aGVyIHRoaXMgaXMgYW4gaW1wb3J0YW50IG1ldHJpYyBncm91cCB0aGF0IHNob3VsZCBiZSBpbmNsdWRlZCBpbiB0aGUgc3VtbWFyeSBhcyB3ZWxsLlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgaW1wb3J0YW50PzogYm9vbGVhbjtcbiAgLyoqXG4gICAqIGxpc3Qgb2YgbWV0cmljcyBpbiB0aGUgZ3JvdXAgKGNhbiBiZSBkZWZpbmVkIGluIGRpZmZlcmVudCB3YXlzLCBzZWUgdGhlIHR5cGUgZG9jdW1lbnRhdGlvbilcbiAgICovXG4gIHJlYWRvbmx5IG1ldHJpY3M6IEN1c3RvbU1ldHJpY1tdO1xuICAvKipcbiAgICogb3B0aW9uYWwgY3VzdG9tIGhvcml6b250YWwgYW5ub3RhdGlvbnMgd2hpY2ggd2lsbCBiZSBkaXNwbGF5ZWQgb3ZlciB0aGUgbWV0cmljcyBvbiB0aGUgbGVmdCBheGlzXG4gICAqIChpZiB0aGVyZSBhcmUgYW55IGFsYXJtcywgYW55IGV4aXN0aW5nIGFubm90YXRpb25zIHdpbGwgYmUgbWVyZ2VkIHRvZ2V0aGVyKVxuICAgKi9cbiAgcmVhZG9ubHkgaG9yaXpvbnRhbEFubm90YXRpb25zPzogSG9yaXpvbnRhbEFubm90YXRpb25bXTtcbiAgLyoqXG4gICAqIG9wdGlvbmFsIGN1c3RvbSBob3Jpem9udGFsIGFubm90YXRpb25zIHdoaWNoIHdpbGwgYmUgZGlzcGxheWVkIG92ZXIgdGhlIG1ldHJpY3Mgb24gdGhlIHJpZ2h0IGF4aXNcbiAgICogKGlmIHRoZXJlIGFyZSBhbnkgYWxhcm1zLCBhbnkgZXhpc3RpbmcgYW5ub3RhdGlvbnMgd2lsbCBiZSBtZXJnZWQgdG9nZXRoZXIpXG4gICAqL1xuICByZWFkb25seSBob3Jpem9udGFsUmlnaHRBbm5vdGF0aW9ucz86IEhvcml6b250YWxBbm5vdGF0aW9uW107XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tTW9uaXRvcmluZ1Byb3BzIGV4dGVuZHMgQmFzZU1vbml0b3JpbmdQcm9wcyB7XG4gIC8qKlxuICAgKiBvcHRpb25hbCBkZXNjcmlwdGlvbiBvZiB0aGUgd2hvbGUgc2VjdGlvbiwgaW4gbWFya2Rvd25cbiAgICogQGRlZmF1bHQgbm8gZGVzY3JpcHRpb25cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogb3B0aW9uYWwgaGVpZ2h0IG9mIHRoZSBkZXNjcmlwdGlvbiB3aWRnZXQsIHNvIHRoZSBjb250ZW50IGZpdHNcbiAgICogQGRlZmF1bHQgbWluaW11bSBoZWlnaHQgKHNob3VsZCBmaXQgb25lIG9yIHR3byBsaW5lcyBvZiB0ZXh0KVxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBoZWlnaHQgb3ZlcnJpZGVcbiAgICogQGRlZmF1bHQgZGVmYXVsdCBoZWlnaHRcbiAgICovXG4gIHJlYWRvbmx5IGhlaWdodD86IG51bWJlcjtcbiAgLyoqXG4gICAqIGRlZmluZSBtZXRyaWMgZ3JvdXBzIGFuZCBtZXRyaWNzIGluc2lkZSB0aGVtIChlYWNoIG1ldHJpYyBncm91cCByZXByZXNlbnRzIGEgd2lkZ2V0KVxuICAgKi9cbiAgcmVhZG9ubHkgbWV0cmljR3JvdXBzOiBDdXN0b21NZXRyaWNHcm91cFtdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU1ldHJpY0dyb3VwV2l0aEFubm90YXRpb25zIHtcbiAgcmVhZG9ubHkgbWV0cmljR3JvdXA6IEN1c3RvbU1ldHJpY0dyb3VwO1xuICByZWFkb25seSBhbm5vdGF0aW9uczogSG9yaXpvbnRhbEFubm90YXRpb25bXTtcbiAgcmVhZG9ubHkgcmlnaHRBbm5vdGF0aW9uczogSG9yaXpvbnRhbEFubm90YXRpb25bXTtcbiAgcmVhZG9ubHkgdGl0bGVBZGRvbnM6IHN0cmluZ1tdO1xuICByZWFkb25seSBoZWlnaHQ/OiBudW1iZXI7XG59XG5cbi8qKlxuICogQ3VzdG9tIG1vbml0b3JpbmcgaXMgYSBjb25zdHJ1Y3QgYWxsb3dpbmcgeW91IHRvIG1vbml0b3IgeW91ciBvd24gY3VzdG9tIG1ldHJpY3MuXG4gKiBUaGUgZW50aXJlIGNvbnN0cnVjdCBjb25zaXN0cyBvZiBtZXRyaWMgZ3JvdXBzLlxuICogRWFjaCBtZXRyaWMgZ3JvdXAgcmVwcmVzZW50cyBhIHNpbmdsZSBncmFwaCB3aWRnZXQgd2l0aCBtdWx0aXBsZSBtZXRyaWNzLlxuICogRWFjaCBtZXRyaWMgaW5zaWRlIHRoZSBtZXRyaWMgZ3JvdXAgcmVwcmVzZW50cyBhIHNpbmdsZSBtZXRyaWMgaW5zaWRlIGEgZ3JhcGguXG4gKiBUaGUgd2lkZ2V0cyB3aWxsIGJlIHNpemVkIGF1dG9tYXRpY2FsbHkgdG8gd2FzdGUgYXMgbGl0dGxlIHNwYWNlIGFzIHBvc3NpYmxlLlxuICovXG5leHBvcnQgY2xhc3MgQ3VzdG9tTW9uaXRvcmluZyBleHRlbmRzIE1vbml0b3Jpbmcge1xuICByZWFkb25seSB0aXRsZTogc3RyaW5nO1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgcmVhZG9ubHkgZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ/OiBudW1iZXI7XG4gIHJlYWRvbmx5IGhlaWdodD86IG51bWJlcjtcbiAgcmVhZG9ubHkgY3VzdG9tQWxhcm1GYWN0b3J5OiBDdXN0b21BbGFybUZhY3Rvcnk7XG4gIHJlYWRvbmx5IGFub21hbHlEZXRlY3RpbmdBbGFybUZhY3Rvcnk6IEFub21hbHlEZXRlY3RpbmdBbGFybUZhY3Rvcnk7XG4gIHJlYWRvbmx5IG1ldHJpY0dyb3VwczogQ3VzdG9tTWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbnNbXTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogTW9uaXRvcmluZ1Njb3BlLCBwcm9wczogQ3VzdG9tTW9uaXRvcmluZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIHByb3BzKTtcblxuICAgIGNvbnN0IG5hbWluZ1N0cmF0ZWd5ID0gbmV3IE1vbml0b3JpbmdOYW1pbmdTdHJhdGVneSh7IC4uLnByb3BzIH0pO1xuICAgIHRoaXMudGl0bGUgPSBuYW1pbmdTdHJhdGVneS5yZXNvbHZlSHVtYW5SZWFkYWJsZU5hbWUoKTtcblxuICAgIHRoaXMuZGVzY3JpcHRpb24gPSBwcm9wcy5kZXNjcmlwdGlvbjtcbiAgICB0aGlzLmRlc2NyaXB0aW9uV2lkZ2V0SGVpZ2h0ID0gcHJvcHMuZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ7XG4gICAgdGhpcy5oZWlnaHQgPSBwcm9wcy5oZWlnaHQ7XG5cbiAgICBjb25zdCBhbGFybUZhY3RvcnkgPSB0aGlzLmNyZWF0ZUFsYXJtRmFjdG9yeShcbiAgICAgIG5hbWluZ1N0cmF0ZWd5LnJlc29sdmVBbGFybUZyaWVuZGx5TmFtZSgpXG4gICAgKTtcbiAgICB0aGlzLmN1c3RvbUFsYXJtRmFjdG9yeSA9IG5ldyBDdXN0b21BbGFybUZhY3RvcnkoYWxhcm1GYWN0b3J5KTtcbiAgICB0aGlzLmFub21hbHlEZXRlY3RpbmdBbGFybUZhY3RvcnkgPSBuZXcgQW5vbWFseURldGVjdGluZ0FsYXJtRmFjdG9yeShcbiAgICAgIGFsYXJtRmFjdG9yeVxuICAgICk7XG5cbiAgICB0aGlzLm1ldHJpY0dyb3VwcyA9IHByb3BzLm1ldHJpY0dyb3Vwcy5tYXAoKG1ldHJpY0dyb3VwKSA9PiB7XG4gICAgICBjb25zdCBtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uOiBDdXN0b21NZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9ucyA9IHtcbiAgICAgICAgbWV0cmljR3JvdXAsXG4gICAgICAgIGFubm90YXRpb25zOiBbXSxcbiAgICAgICAgcmlnaHRBbm5vdGF0aW9uczogW10sXG4gICAgICAgIHRpdGxlQWRkb25zOiBbXSxcbiAgICAgIH07XG5cbiAgICAgIGlmIChtZXRyaWNHcm91cC5ob3Jpem9udGFsQW5ub3RhdGlvbnMpIHtcbiAgICAgICAgbWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbi5hbm5vdGF0aW9ucy5wdXNoKFxuICAgICAgICAgIC4uLm1ldHJpY0dyb3VwLmhvcml6b250YWxBbm5vdGF0aW9uc1xuICAgICAgICApO1xuICAgICAgfVxuICAgICAgaWYgKG1ldHJpY0dyb3VwLmhvcml6b250YWxSaWdodEFubm90YXRpb25zKSB7XG4gICAgICAgIG1ldHJpY0dyb3VwV2l0aEFubm90YXRpb24ucmlnaHRBbm5vdGF0aW9ucy5wdXNoKFxuICAgICAgICAgIC4uLm1ldHJpY0dyb3VwLmhvcml6b250YWxSaWdodEFubm90YXRpb25zXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIG1ldHJpY0dyb3VwLm1ldHJpY3MuZm9yRWFjaCgobWV0cmljKSA9PiB7XG4gICAgICAgIGlmICh0aGlzLmhhc0FsYXJtKG1ldHJpYykgJiYgdGhpcy5oYXNBbm9tYWx5RGV0ZWN0aW9uKG1ldHJpYykpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgICBcIkFkZGluZyBib3RoIGEgcmVndWxhciBhbGFybSBhbmQgYW4gYW5vbWFseSBkZXRlY3Rpb24gYWxhcm0gYXQgdGhlIHNhbWUgdGltZSBpcyBub3Qgc3VwcG9ydGVkXCJcbiAgICAgICAgICApO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHRoaXMuaGFzQWxhcm0obWV0cmljKSkge1xuICAgICAgICAgIHRoaXMuc2V0dXBBbGFybShtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uLCBtZXRyaWMpO1xuICAgICAgICB9IGVsc2UgaWYgKHRoaXMuaGFzQW5vbWFseURldGVjdGlvbihtZXRyaWMpKSB7XG4gICAgICAgICAgdGhpcy5zZXR1cEFub21hbHlEZXRlY3Rpb25BbGFybShtZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9uLCBtZXRyaWMpO1xuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgICAgcmV0dXJuIG1ldHJpY0dyb3VwV2l0aEFubm90YXRpb247XG4gICAgfSk7XG5cbiAgICBwcm9wcy51c2VDcmVhdGVkQWxhcm1zPy5jb25zdW1lKHRoaXMuY3JlYXRlZEFsYXJtcygpKTtcbiAgfVxuXG4gIHN1bW1hcnlXaWRnZXRzKCk6IElXaWRnZXRbXSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0QWxsV2lkZ2V0cyh0cnVlKTtcbiAgfVxuXG4gIHdpZGdldHMoKTogSVdpZGdldFtdIHtcbiAgICByZXR1cm4gdGhpcy5nZXRBbGxXaWRnZXRzKGZhbHNlKTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0QWxsV2lkZ2V0cyhzdW1tYXJ5OiBib29sZWFuKTogSVdpZGdldFtdIHtcbiAgICBjb25zdCBmaWx0ZXJlZE1ldHJpY0dyb3VwcyA9IHN1bW1hcnlcbiAgICAgID8gdGhpcy5tZXRyaWNHcm91cHMuZmlsdGVyKFxuICAgICAgICAgIChncm91cCkgPT4gZ3JvdXAubWV0cmljR3JvdXAuaW1wb3J0YW50ID8/IGZhbHNlXG4gICAgICAgIClcbiAgICAgIDogdGhpcy5tZXRyaWNHcm91cHM7XG5cbiAgICBpZiAoZmlsdGVyZWRNZXRyaWNHcm91cHMubGVuZ3RoIDwgMSkge1xuICAgICAgLy8gc2hvcnQtY2lyY3VpdCBpZiB0aGVyZSBhcmUgbm8gbWV0cmljcyBzcGVjaWZpZWRcbiAgICAgIHJldHVybiBbXTtcbiAgICB9XG5cbiAgICBjb25zdCByb3dzOiBSb3dbXSA9IFtdO1xuXG4gICAgLy8gaGVhZGVyIGFuZCBkZXNjcmlwdGlvblxuICAgIHJvd3MucHVzaChuZXcgUm93KG5ldyBNb25pdG9yaW5nSGVhZGVyV2lkZ2V0KHsgdGl0bGU6IHRoaXMudGl0bGUgfSkpKTtcbiAgICBpZiAodGhpcy5kZXNjcmlwdGlvbiAmJiAhc3VtbWFyeSkge1xuICAgICAgcm93cy5wdXNoKFxuICAgICAgICBuZXcgUm93KFxuICAgICAgICAgIHRoaXMuY3JlYXRlRGVzY3JpcHRpb25XaWRnZXQoXG4gICAgICAgICAgICB0aGlzLmRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgdGhpcy5kZXNjcmlwdGlvbldpZGdldEhlaWdodFxuICAgICAgICAgIClcbiAgICAgICAgKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICAvLyBncmFwaHNcbiAgICByb3dzLnB1c2goXG4gICAgICBuZXcgUm93KFxuICAgICAgICAuLi50aGlzLmNyZWF0ZUN1c3RvbU1ldHJpY0dyb3VwV2lkZ2V0cyhmaWx0ZXJlZE1ldHJpY0dyb3Vwcywgc3VtbWFyeSlcbiAgICAgIClcbiAgICApO1xuXG4gICAgcmV0dXJuIHJvd3M7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZURlc2NyaXB0aW9uV2lkZ2V0KFxuICAgIG1hcmtkb3duOiBzdHJpbmcsXG4gICAgZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQ/OiBudW1iZXJcbiAgKSB7XG4gICAgcmV0dXJuIG5ldyBUZXh0V2lkZ2V0KHtcbiAgICAgIG1hcmtkb3duLFxuICAgICAgd2lkdGg6IEZ1bGxXaWR0aCxcbiAgICAgIGhlaWdodDogZGVzY3JpcHRpb25XaWRnZXRIZWlnaHQgPz8gMSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQ3VzdG9tTWV0cmljR3JvdXBXaWRnZXRzKFxuICAgIGFubm90YXRlZEdyb3VwczogQ3VzdG9tTWV0cmljR3JvdXBXaXRoQW5ub3RhdGlvbnNbXSxcbiAgICBzdW1tYXJ5OiBib29sZWFuXG4gICkge1xuICAgIGNvbnN0IHdpZGdldHM6IElXaWRnZXRbXSA9IFtdO1xuICAgIGNvbnN0IG1ldHJpY0dyb3VwV2lkZ2V0V2lkdGggPSByZWNvbW1lbmRlZFdpZGdldFdpZHRoKFxuICAgICAgYW5ub3RhdGVkR3JvdXBzLmxlbmd0aFxuICAgICk7XG4gICAgY29uc3QgbWV0cmljR3JvdXBXaWRnZXRIZWlnaHREZWZhdWx0ID0gc3VtbWFyeVxuICAgICAgPyBEZWZhdWx0U3VtbWFyeVdpZGdldEhlaWdodFxuICAgICAgOiBEZWZhdWx0R3JhcGhXaWRnZXRIZWlnaHQ7XG4gICAgY29uc3QgbWV0cmljR3JvdXBXaWRnZXRIZWlnaHQgPVxuICAgICAgdGhpcy5oZWlnaHQgPz8gbWV0cmljR3JvdXBXaWRnZXRIZWlnaHREZWZhdWx0O1xuXG4gICAgYW5ub3RhdGVkR3JvdXBzLmZvckVhY2goKGFubm90YXRlZEdyb3VwKSA9PiB7XG4gICAgICBjb25zdCBtZXRyaWNzID0gYW5ub3RhdGVkR3JvdXAubWV0cmljR3JvdXAubWV0cmljcztcbiAgICAgIGNvbnN0IGxlZnQgPSB0aGlzLnRvTWV0cmljcyhcbiAgICAgICAgbWV0cmljcy5maWx0ZXIoXG4gICAgICAgICAgKG1ldHJpYykgPT5cbiAgICAgICAgICAgICgobWV0cmljIGFzIGFueSkucG9zaXRpb24gPz8gQXhpc1Bvc2l0aW9uLkxFRlQpID09IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAgICAgIClcbiAgICAgICk7XG4gICAgICBjb25zdCByaWdodCA9IHRoaXMudG9NZXRyaWNzKFxuICAgICAgICBtZXRyaWNzLmZpbHRlcihcbiAgICAgICAgICAobWV0cmljKSA9PlxuICAgICAgICAgICAgKChtZXRyaWMgYXMgYW55KS5wb3NpdGlvbiA/PyBBeGlzUG9zaXRpb24uTEVGVCkgPT1cbiAgICAgICAgICAgIEF4aXNQb3NpdGlvbi5SSUdIVFxuICAgICAgICApXG4gICAgICApO1xuICAgICAgY29uc3QgaGFzT25lTWV0cmljT25seSA9IG1ldHJpY3MubGVuZ3RoID09PSAxO1xuICAgICAgY29uc3QgaGFzQW5vbWFseURldGVjdGlvbiA9XG4gICAgICAgIG1ldHJpY3MuZmlsdGVyKChtZXRyaWMpID0+IHRoaXMuaGFzQW5vbWFseURldGVjdGlvbihtZXRyaWMpKS5sZW5ndGggPiAwO1xuICAgICAgY29uc3QgdXNlQW5vbWFseURldGVjdGlvbldpZGdldCA9IGhhc09uZU1ldHJpY09ubHkgJiYgaGFzQW5vbWFseURldGVjdGlvbjtcbiAgICAgIGxldCB0aXRsZSA9IGFubm90YXRlZEdyb3VwLm1ldHJpY0dyb3VwLnRpdGxlO1xuXG4gICAgICBpZiAoYW5ub3RhdGVkR3JvdXAudGl0bGVBZGRvbnMubGVuZ3RoID4gMCkge1xuICAgICAgICB0aXRsZSA9IGAke3RpdGxlfSAoJHthbm5vdGF0ZWRHcm91cC50aXRsZUFkZG9ucy5qb2luKFwiLCBcIil9KWA7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGdyYXBoV2lkZ2V0UHJvcHM6IEdyYXBoV2lkZ2V0UHJvcHMgPSB7XG4gICAgICAgIHRpdGxlLFxuICAgICAgICB3aWR0aDogbWV0cmljR3JvdXBXaWRnZXRXaWR0aCxcbiAgICAgICAgaGVpZ2h0OiBtZXRyaWNHcm91cFdpZGdldEhlaWdodCxcbiAgICAgICAgbGVmdCxcbiAgICAgICAgcmlnaHQsXG4gICAgICAgIGxlZnRBbm5vdGF0aW9uczogYW5ub3RhdGVkR3JvdXAuYW5ub3RhdGlvbnMsXG4gICAgICAgIHJpZ2h0QW5ub3RhdGlvbnM6IGFubm90YXRlZEdyb3VwLnJpZ2h0QW5ub3RhdGlvbnMsXG4gICAgICAgIGxlZnRZQXhpczogYW5ub3RhdGVkR3JvdXAubWV0cmljR3JvdXAuZ3JhcGhXaWRnZXRBeGlzLFxuICAgICAgICByaWdodFlBeGlzOiBhbm5vdGF0ZWRHcm91cC5tZXRyaWNHcm91cC5ncmFwaFdpZGdldFJpZ2h0QXhpcyxcbiAgICAgICAgbGVnZW5kUG9zaXRpb246IGFubm90YXRlZEdyb3VwLm1ldHJpY0dyb3VwLmdyYXBoV2lkZ2V0TGVnZW5kLFxuICAgICAgfTtcblxuICAgICAgY29uc3Qgd2lkZ2V0ID0gdXNlQW5vbWFseURldGVjdGlvbldpZGdldFxuICAgICAgICA/IG5ldyBBbm9tYWx5RGV0ZWN0aW9uR3JhcGhXaWRnZXQoZ3JhcGhXaWRnZXRQcm9wcylcbiAgICAgICAgOiBjcmVhdGVHcmFwaFdpZGdldChcbiAgICAgICAgICAgIGFubm90YXRlZEdyb3VwLm1ldHJpY0dyb3VwLmdyYXBoV2lkZ2V0VHlwZSA/PyBHcmFwaFdpZGdldFR5cGUuTElORSxcbiAgICAgICAgICAgIGdyYXBoV2lkZ2V0UHJvcHNcbiAgICAgICAgICApO1xuXG4gICAgICB3aWRnZXRzLnB1c2god2lkZ2V0KTtcbiAgICB9KTtcblxuICAgIHJldHVybiB3aWRnZXRzO1xuICB9XG5cbiAgcHJpdmF0ZSB0b01ldHJpY3MobWV0cmljczogQ3VzdG9tTWV0cmljW10pOiBJTWV0cmljW10ge1xuICAgIGNvbnN0IG1ldHJpY0ZhY3RvcnkgPSB0aGlzLmNyZWF0ZU1ldHJpY0ZhY3RvcnkoKTtcblxuICAgIHJldHVybiBtZXRyaWNzLm1hcCgobWV0cmljKSA9PiB7XG4gICAgICBpZiAodGhpcy5oYXNBbGFybShtZXRyaWMpKSB7XG4gICAgICAgIC8vIG1ldHJpYyB3aXRoIGFsYXJtXG4gICAgICAgIHJldHVybiBtZXRyaWNGYWN0b3J5LmFkYXB0TWV0cmljUHJlc2VydmluZ1BlcmlvZChtZXRyaWMubWV0cmljKTtcbiAgICAgIH0gZWxzZSBpZiAodGhpcy5oYXNBbm9tYWx5RGV0ZWN0aW9uKG1ldHJpYykpIHtcbiAgICAgICAgLy8gbWV0cmljIHdpdGggYW5vbWFseSBkZXRlY3Rpb25cbiAgICAgICAgcmV0dXJuIG1ldHJpY0ZhY3RvcnkuY3JlYXRlTWV0cmljQW5vbWFseURldGVjdGlvbihcbiAgICAgICAgICBtZXRyaWMubWV0cmljLFxuICAgICAgICAgIG1ldHJpYy5hbm9tYWx5RGV0ZWN0aW9uU3RhbmRhcmREZXZpYXRpb25Ub1JlbmRlcixcbiAgICAgICAgICBgRXhwZWN0ZWQgKHN0ZGV2ID0gJHttZXRyaWMuYW5vbWFseURldGVjdGlvblN0YW5kYXJkRGV2aWF0aW9uVG9SZW5kZXJ9KWAsXG4gICAgICAgICAgdW5kZWZpbmVkLFxuICAgICAgICAgIC8vIG5lZWRzIHRvIGJlIHVuaXF1ZSBpbiB0aGUgd2hvbGUgd2lkZ2V0IGFuZCBzdGFydCB3aXRoIGxvd2VyY2FzZVxuICAgICAgICAgIEFub21hbHlEZXRlY3Rpb25NZXRyaWNJZFByZWZpeCArXG4gICAgICAgICAgICBnZXRIYXNoRm9yTWV0cmljRXhwcmVzc2lvbklkKG1ldHJpYy5hbGFybUZyaWVuZGx5TmFtZSksXG4gICAgICAgICAgLy8gcHJlc2VydmUgdGhlIG1vc3Qgc3BlY2lmaWMgbWV0cmljIHBlcmlvZFxuICAgICAgICAgIG1ldHJpYy5wZXJpb2QgPz8gbWV0cmljLm1ldHJpYy5wZXJpb2RcbiAgICAgICAgKTtcbiAgICAgIH0gZWxzZSBpZiAodGhpcy5pc1NlYXJjaChtZXRyaWMpKSB7XG4gICAgICAgIC8vIG1ldHJpYyBzZWFyY2hcbiAgICAgICAgcmV0dXJuIG1ldHJpY0ZhY3RvcnkuY3JlYXRlTWV0cmljU2VhcmNoKFxuICAgICAgICAgIG1ldHJpYy5zZWFyY2hRdWVyeSxcbiAgICAgICAgICBtZXRyaWMuZGltZW5zaW9uc01hcCxcbiAgICAgICAgICBtZXRyaWMuc3RhdGlzdGljLFxuICAgICAgICAgIG1ldHJpYy5uYW1lc3BhY2UsXG4gICAgICAgICAgbWV0cmljLmxhYmVsLFxuICAgICAgICAgIG1ldHJpYy5wZXJpb2RcbiAgICAgICAgKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIGdlbmVyYWwgbWV0cmljXG4gICAgICAgIHJldHVybiBtZXRyaWNGYWN0b3J5LmFkYXB0TWV0cmljUHJlc2VydmluZ1BlcmlvZChtZXRyaWMpO1xuICAgICAgfVxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBoYXNBbGFybShtZXRyaWM6IEN1c3RvbU1ldHJpYyk6IG1ldHJpYyBpcyBDdXN0b21NZXRyaWNXaXRoQWxhcm0ge1xuICAgIC8vIHR5cGUgZ3VhcmRcbiAgICByZXR1cm4gKG1ldHJpYyBhcyBDdXN0b21NZXRyaWNXaXRoQWxhcm0pLmFkZEFsYXJtICE9PSB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIGhhc0Fub21hbHlEZXRlY3Rpb24oXG4gICAgbWV0cmljOiBDdXN0b21NZXRyaWNcbiAgKTogbWV0cmljIGlzIEN1c3RvbU1ldHJpY1dpdGhBbm9tYWx5RGV0ZWN0aW9uIHtcbiAgICAvLyB0eXBlIGd1YXJkXG4gICAgcmV0dXJuIChcbiAgICAgIChtZXRyaWMgYXMgQ3VzdG9tTWV0cmljV2l0aEFub21hbHlEZXRlY3Rpb24pXG4gICAgICAgIC5hbm9tYWx5RGV0ZWN0aW9uU3RhbmRhcmREZXZpYXRpb25Ub1JlbmRlciAhPT0gdW5kZWZpbmVkXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgaXNTZWFyY2gobWV0cmljOiBDdXN0b21NZXRyaWMpOiBtZXRyaWMgaXMgQ3VzdG9tTWV0cmljU2VhcmNoIHtcbiAgICAvLyB0eXBlIGd1YXJkXG4gICAgcmV0dXJuIChtZXRyaWMgYXMgQ3VzdG9tTWV0cmljU2VhcmNoKS5zZWFyY2hRdWVyeSAhPT0gdW5kZWZpbmVkO1xuICB9XG5cbiAgcHJpdmF0ZSBzZXR1cEFsYXJtKFxuICAgIG1ldHJpY0dyb3VwOiBDdXN0b21NZXRyaWNHcm91cFdpdGhBbm5vdGF0aW9ucyxcbiAgICBtZXRyaWM6IEN1c3RvbU1ldHJpY1dpdGhBbGFybVxuICApIHtcbiAgICBpZiAodGhpcy5pc1NlYXJjaChtZXRyaWMpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiQWxhcm1pbmcgb24gc2VhcmNoIHF1ZXJpZXMgaXMgbm90IHN1cHBvcnRlZCBieSBDbG91ZFdhdGNoXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIG1ldHJpYy5hZGRBbGFybSkge1xuICAgICAgY29uc3QgYWxhcm1Qcm9wcyA9IG1ldHJpYy5hZGRBbGFybVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGNvbnN0IGNyZWF0ZWRBbGFybSA9IHRoaXMuY3VzdG9tQWxhcm1GYWN0b3J5LmFkZEN1c3RvbUFsYXJtKFxuICAgICAgICBtZXRyaWMubWV0cmljLFxuICAgICAgICBtZXRyaWMuYWxhcm1GcmllbmRseU5hbWUsXG4gICAgICAgIGRpc2FtYmlndWF0b3IsXG4gICAgICAgIGFsYXJtUHJvcHNcbiAgICAgICk7XG4gICAgICBjb25zdCB0YXJnZXRBbm5vdGF0aW9ucyA9XG4gICAgICAgIChtZXRyaWMucG9zaXRpb24gPz8gQXhpc1Bvc2l0aW9uLkxFRlQpID09IEF4aXNQb3NpdGlvbi5MRUZUXG4gICAgICAgICAgPyBtZXRyaWNHcm91cC5hbm5vdGF0aW9uc1xuICAgICAgICAgIDogbWV0cmljR3JvdXAucmlnaHRBbm5vdGF0aW9ucztcbiAgICAgIHRhcmdldEFubm90YXRpb25zLnB1c2goY3JlYXRlZEFsYXJtLmFubm90YXRpb24pO1xuICAgICAgdGhpcy5hZGRBbGFybShjcmVhdGVkQWxhcm0pO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgc2V0dXBBbm9tYWx5RGV0ZWN0aW9uQWxhcm0oXG4gICAgbWV0cmljR3JvdXA6IEN1c3RvbU1ldHJpY0dyb3VwV2l0aEFubm90YXRpb25zLFxuICAgIG1ldHJpYzogQ3VzdG9tTWV0cmljV2l0aEFub21hbHlEZXRlY3Rpb25cbiAgKSB7XG4gICAgaWYgKHRoaXMuaXNTZWFyY2gobWV0cmljKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIkFsYXJtaW5nIG9uIHNlYXJjaCBxdWVyaWVzIGlzIG5vdCBzdXBwb3J0ZWQgYnkgQ2xvdWRXYXRjaFwiXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IGFsYXJtU3REZXZzID0gbmV3IFNldDxudW1iZXI+KCk7XG4gICAgY29uc3QgbWV0cmljRmFjdG9yeSA9IHRoaXMuY3JlYXRlTWV0cmljRmFjdG9yeSgpO1xuXG4gICAgZm9yIChjb25zdCBkaXNhbWJpZ3VhdG9yIGluIG1ldHJpYy5hZGRBbGFybU9uQW5vbWFseSkge1xuICAgICAgY29uc3QgYWxhcm1Qcm9wcyA9IG1ldHJpYy5hZGRBbGFybU9uQW5vbWFseVtkaXNhbWJpZ3VhdG9yXTtcbiAgICAgIGlmIChcbiAgICAgICAgYWxhcm1Qcm9wcy5hbGFybVdoZW5BYm92ZVRoZUJhbmQgfHxcbiAgICAgICAgYWxhcm1Qcm9wcy5hbGFybVdoZW5CZWxvd1RoZUJhbmRcbiAgICAgICkge1xuICAgICAgICBjb25zdCBhbm9tYWx5TWV0cmljID0gbWV0cmljRmFjdG9yeS5jcmVhdGVNZXRyaWNBbm9tYWx5RGV0ZWN0aW9uKFxuICAgICAgICAgIC8vIEJlY2F1c2UgdGhlIG1ldHJpYyB3YXMgcHJvdmlkZWQgdG8gdXMsIHdlIHVzZSBtZXRyaWNGYWN0b3J5Lm92ZXJyaWRlTmFtZXNwYWNlKCkgdG9cbiAgICAgICAgICAvLyBjb25maXJtIGl0IGFsaWducyB3aXRoIGFueSBuYW1lc3BhY2Ugb3ZlcnJpZGVzIHJlcXVlc3RlZCBmb3IgdGhpcyBNb25pdG9yaW5nRmFjYWRlXG4gICAgICAgICAgbWV0cmljRmFjdG9yeS5hZGFwdE1ldHJpY1ByZXNlcnZpbmdQZXJpb2QobWV0cmljLm1ldHJpYyksXG4gICAgICAgICAgYWxhcm1Qcm9wcy5zdGFuZGFyZERldmlhdGlvbkZvckFsYXJtLFxuICAgICAgICAgIGBCYW5kIChzdGRldiAke2FsYXJtUHJvcHMuc3RhbmRhcmREZXZpYXRpb25Gb3JBbGFybX0pYCxcbiAgICAgICAgICB1bmRlZmluZWQsXG4gICAgICAgICAgLy8gZXhwcmVzc2lvbiBJRCBuZWVkcyB0byBiZSB1bmlxdWUgYWNyb3NzIHRoZSB3aG9sZSB3aWRnZXQ7IG5lZWRzIHRvIHN0YXJ0IHdpdGggYSBsb3dlcmNhc2UgbGV0dGVyXG4gICAgICAgICAgQW5vbWFseURldGVjdGlvbkFsYXJtSWRQcmVmaXggK1xuICAgICAgICAgICAgZ2V0SGFzaEZvck1ldHJpY0V4cHJlc3Npb25JZChcbiAgICAgICAgICAgICAgbWV0cmljLmFsYXJtRnJpZW5kbHlOYW1lICsgXCJfXCIgKyBkaXNhbWJpZ3VhdG9yXG4gICAgICAgICAgICApLFxuICAgICAgICAgIC8vIHByZXNlcnZlIHRoZSBtb3N0LXNwZWNpZmljIG1ldHJpYyBwZXJpb2RcbiAgICAgICAgICBtZXRyaWMucGVyaW9kID8/IG1ldHJpYy5tZXRyaWMucGVyaW9kXG4gICAgICAgICk7XG5cbiAgICAgICAgY29uc3QgY3JlYXRlZEFsYXJtID1cbiAgICAgICAgICB0aGlzLmFub21hbHlEZXRlY3RpbmdBbGFybUZhY3RvcnkuYWRkQWxhcm1XaGVuT3V0T2ZCYW5kKFxuICAgICAgICAgICAgYW5vbWFseU1ldHJpYyxcbiAgICAgICAgICAgIG1ldHJpYy5hbGFybUZyaWVuZGx5TmFtZSxcbiAgICAgICAgICAgIGRpc2FtYmlndWF0b3IsXG4gICAgICAgICAgICBhbGFybVByb3BzXG4gICAgICAgICAgKTtcblxuICAgICAgICAvLyBubyBuZWVkIHRvIGFkZCBhbm5vdGF0aW9uIHNpbmNlIHRoZSBiYW5kcyBhcmUgcmVuZGVyZWQgYXV0b21hdGljYWxseVxuICAgICAgICB0aGlzLmFkZEFsYXJtKGNyZWF0ZWRBbGFybSk7XG4gICAgICAgIGFsYXJtU3REZXZzLmFkZChhbGFybVByb3BzLnN0YW5kYXJkRGV2aWF0aW9uRm9yQWxhcm0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChhbGFybVN0RGV2cy5zaXplID4gMCkge1xuICAgICAgY29uc3QgYWxhcm1TdERldnNTdHJpbmcgPSBBcnJheS5mcm9tKGFsYXJtU3REZXZzKS5zb3J0KCkuam9pbihcIiwgXCIpO1xuICAgICAgbWV0cmljR3JvdXAudGl0bGVBZGRvbnMucHVzaChgYWxhcm1zIHdpdGggc3RkZXYgJHthbGFybVN0RGV2c1N0cmluZ31gKTtcbiAgICB9XG4gIH1cbn1cblxuY29uc3QgQW5vbWFseURldGVjdGlvbkFsYXJtSWRQcmVmaXggPSBcImFsYXJtX1wiO1xuY29uc3QgQW5vbWFseURldGVjdGlvbk1ldHJpY0lkUHJlZml4ID0gXCJhbm9tYWx5X1wiO1xuY29uc3QgQW5vbWFseUJhbmRNZXRyaWNJZFN1ZmZpeCA9IFwiX2JhbmRcIjtcblxuLyoqXG4gKiBJTlRFUk5BTCAtIFBMRUFTRSBETyBOT1QgVVNFXG4gKiBUaGlzIGlzIGEgaGFja3kgc29sdXRpb24gdG8gbWFrZSBiYW5kIHZpc2libGUgaW4gR3JhcGhXaWRnZXQgKGRlZmF1bHQgd2lkZ2V0IG9ubHkgcmVuZGVycyBsaW5lcywgbm90IHRoZSBiYW5kKS5cbiAqIFRoZSBjbGFzcyBtYWtlcyBhc3N1bXB0aW9ucyBhYm91dCB0aGUgaW50ZXJuYWwgSlNPTiBzdHJ1Y3R1cmUgYnV0IGZvdW5kIG5vIG90aGVyIHdheSA6KC5cbiAqIElkZWFsbHksIHdlIHdhbnQgdG8gcmVtb3ZlIHRoaXMgaGFjayBvbmNlIHRoZSBhbm9tYWx5IGRldGVjdGlvbiByZW5kZXJpbmcgaW4gQ0RLIGdldHMgaW1wcm92ZWRcbiAqL1xuY2xhc3MgQW5vbWFseURldGVjdGlvbkdyYXBoV2lkZ2V0IGV4dGVuZHMgR3JhcGhXaWRnZXQge1xuICBjb25zdHJ1Y3Rvcihwcm9wczogR3JhcGhXaWRnZXRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxuXG4gIHRvSnNvbigpIHtcbiAgICBjb25zdCBqc29uID0gc3VwZXIudG9Kc29uKCk7XG4gICAgaWYgKGpzb24ubGVuZ3RoICE9PSAxIHx8ICFqc29uPy5bMF0/LnByb3BlcnRpZXM/Lm1ldHJpY3MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJUaGUgSlNPTiBpcyBleHBlY3RlZCB0byBoYXZlIGV4YWN0bHkgb25lIGVsZW1lbnQgd2l0aCBwcm9wZXJ0aWVzLm1ldHJpY3MgcHJvcGVydHkuXCJcbiAgICAgICk7XG4gICAgfVxuICAgIGNvbnN0IG1ldHJpY3M6IGFueVtdID0ganNvblswXS5wcm9wZXJ0aWVzLm1ldHJpY3M7XG4gICAgaWYgKG1ldHJpY3MubGVuZ3RoIDwgMikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcIlRoZSBudW1iZXIgb2YgbWV0cmljcyBtdXN0IGJlIGF0IGxlYXN0IHR3byAobWV0cmljICsgYW5vbWFseSBkZXRlY3Rpb24gbWF0aCkuXCJcbiAgICAgICk7XG4gICAgfVxuICAgIGNvbnN0IGFub21hbHlEZXRlY3Rpb25NZXRyaWNQYXJ0OiBhbnlbXSA9IG1ldHJpY3NbMF0/LnZhbHVlO1xuICAgIGlmIChcbiAgICAgICFhbm9tYWx5RGV0ZWN0aW9uTWV0cmljUGFydCB8fFxuICAgICAgYW5vbWFseURldGVjdGlvbk1ldHJpY1BhcnQubGVuZ3RoICE9PSAxXG4gICAgKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJGaXJzdCBtZXRyaWMgbXVzdCBiZSBhIG1hdGggZXhwcmVzc2lvbi5cIik7XG4gICAgfVxuICAgIGNvbnN0IGV2YWx1YXRlZE1ldHJpY1BhcnQ6IGFueVtdID0gbWV0cmljc1sxXT8udmFsdWU7XG4gICAgaWYgKFxuICAgICAgIWV2YWx1YXRlZE1ldHJpY1BhcnQgfHxcbiAgICAgIGV2YWx1YXRlZE1ldHJpY1BhcnQubGVuZ3RoIDwgMSB8fFxuICAgICAgIWV2YWx1YXRlZE1ldHJpY1BhcnRbZXZhbHVhdGVkTWV0cmljUGFydC5sZW5ndGggLSAxXS5pZFxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiU2Vjb25kIG1ldHJpYyBtdXN0IGhhdmUgYW4gSUQuXCIpO1xuICAgIH1cbiAgICAvLyBiYW5kIHJlbmRlcmluZyByZXF1aXJlcyBJRCB0byBiZSBzZXRcbiAgICBhbm9tYWx5RGV0ZWN0aW9uTWV0cmljUGFydFswXS5pZCA9XG4gICAgICBldmFsdWF0ZWRNZXRyaWNQYXJ0W2V2YWx1YXRlZE1ldHJpY1BhcnQubGVuZ3RoIC0gMV0uaWQgK1xuICAgICAgQW5vbWFseUJhbmRNZXRyaWNJZFN1ZmZpeDtcbiAgICAvLyBiYW5kIHJlbmRlcmluZyByZXF1aXJlcyB0aGUgZXZhbHVhdGVkIG1ldHJpYyB0byBiZSB2aXNpYmxlXG4gICAgZXZhbHVhdGVkTWV0cmljUGFydFtldmFsdWF0ZWRNZXRyaWNQYXJ0Lmxlbmd0aCAtIDFdLnZpc2libGUgPSB0cnVlO1xuICAgIHJldHVybiBqc29uO1xuICB9XG59XG4iXX0=