from flask import Response, request, jsonify
from logging import Logger
from typing import Any

from .iam_common import (
    IAM_SERVERS, IamServer,
    _service_login, _service_logout,
    _service_callback, _service_token, _log_init
)


# @flask_app.route(rule=<login_endpoint>,  # JUSBR_LOGIN_ENDPOINT: /iam/jusbr:login
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,  # KEYCLOAK_LOGIN_ENDPOINT: /iam/keycloak:logout
#                  methods=["GET"])
def service_login() -> Response:
    """
    Entry point for the JusBR login service.

    Redirect the request to the JusBR authentication page, with the appropriate parameters.

    :return: the response from the redirect operation
    """
    # retrieve logger and registry
    registry: dict[str, Any] = __get_iam_registry(endpoint=request.endpoint)
    logger: Logger = registry["logger"]

    # log the request
    if logger:
        logger.debug(msg=_log_init(request=request))

    # obtain the login URL
    login_data: dict[str,  str] = _service_login(registry=registry,
                                                 args=request.args,
                                                 logger=logger)
    result = jsonify(login_data)

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<logout_endpoint>,  # JUSBR_LOGOUT_ENDPOINT: /iam/jusbr:logout
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,  # KEYCLOAK_LOGOUT_ENDPOINT: /iam/keycloak:logout
#                  methods=["GET"])
def service_logout() -> Response:
    """
    Entry point for the JusBR logout service.

    Remove all data associating the user with JusBR from the registry.

    :return: response *OK*
    """
    # retrieve logger and registry
    registry: dict[str, Any] = __get_iam_registry(endpoint=request.endpoint)
    logger: Logger = registry["logger"]

    # log the request
    if logger:
        logger.debug(msg=_log_init(request=request))

    # logout the user
    _service_logout(registry=registry,
                    args=request.args,
                    logger=logger)

    result: Response = Response(status=200)

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<callback_endpoint>,  # JUSBR_CALLBACK_ENDPOINT: /iam/jusbr:callback
#                  methods=["GET", "POST"])
# @flask_app.route(rule=<callback_endpoint>,  # KEYCLOAK_CALLBACK_ENDPOINT: /iam/keycloak:callback
#                  methods=["POST"])
def service_callback() -> Response:
    """
    Entry point for the callback from JusBR on authentication operation.

    This callback is typically invoked from a front-end application after a successful login at the
    JusBR login page, forwarding the data received.

    :return: the response containing the token, or *BAD REQUEST*
    """
    # retrieve logger and registry
    registry: dict[str, Any] = __get_iam_registry(endpoint=request.endpoint)
    logger: Logger = registry["logger"]

    # log the request
    if logger:
        logger.debug(msg=_log_init(request=request))

    # process the callback operation
    errors: list[str] = []
    token_data: tuple[str, str] = _service_callback(registry=registry,
                                                    args=request.args,
                                                    errors=errors,
                                                    logger=logger)
    result: Response
    if errors:
        result = jsonify({"errors": "; ".join(errors)})
        result.status_code = 400
    else:
        result = jsonify({
            "user_id": token_data[0],
            "access_token": token_data[1]})

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<token_endpoint>,  # JUSBR_TOKEN_ENDPOINT: /iam/jusbr:get-token
#                  methods=["GET"])
# @flask_app.route(rule=<token_endpoint>,  # JUSBR_TOKEN_ENDPOINT: /iam/jusbr:get-token
#                  methods=["GET"])
def service_token() -> Response:
    """
    Entry point for retrieving the JusBR token.

    :return: the response containing the token, or *UNAUTHORIZED*
    """
    # retrieve logger and registry
    registry: dict[str, Any] = __get_iam_registry(endpoint=request.endpoint)
    logger: Logger = registry["logger"]

    # log the request
    if logger:
        logger.debug(msg=_log_init(request=request))

    # retrieve the token
    errors: list[str] = []
    token: str = _service_token(registry=registry,
                                args=request.args,
                                errors=errors,
                                logger=logger)
    result: Response
    if token:
        result = jsonify({"token": token})
    else:
        result = Response("; ".join(errors))
        result.status_code = 401

    # log the response
    if logger:
        logger.debug(msg=f"Response {result}")

    return result


def __get_iam_registry(endpoint: str) -> dict[str, Any]:
    """
    Retrieve the registry associated the the IAM identifies by *endpoint*.

    :param endpoint: the service enpoint identifying the IAM.
    :return: the tuple (*logger*, *registry*) associated with *endpoint*
    """
    # initialize the return variable
    result: dict[str, Any] | None = None

    if endpoint.startswith("jusbr-"):
        result = IAM_SERVERS[IamServer.IAM_JUSRBR]
    elif endpoint.startswith("keycloak-"):
        result = IAM_SERVERS[IamServer.IAM_KEYCLOAK]

    return result

