"""
OnionMeta: A Python metaclass for dynamic abstract method resolution
Implements the onion architecture pattern, decoupling core business logic from implementations
"""

import abc
import warnings
from typing import Dict, List, Type, Any, Optional


class OnionMeta(abc.ABCMeta):
    """
    Onion Architecture Metaclass
    
    This metaclass implements the "Construction & Binding" (C&B) process, dynamically
    collecting and integrating methods from all implementation submodules at first instantiation.
    """
    
    # Attribute section: all lowercase and private
    __onion_subs__ = []  # Registry of subclasses
    __onion_built__ = False  # Initialization flag
    
    # Internal methods: private methods starting with _
    # Note: impls loading is user responsibility, no auto-loading provided
    
    def __onion_get__onion_subs__(cls) -> List[Type]:
        """Retrieve all registered non-abstract subclasses"""
        subs = list(cls.__onion_subs__)
        # [DEBUG] found {len(subs)} subclasses
        return subs
    
    def __onion_get_meths(cls, subs: List[Type], warns: List[str]) -> Dict[str, Any]:
        """Collect method implementations from subclasses"""
        meths = {}
        srcs = {}  # Track source of each method

        for sub in subs:
            for meth_name in cls.__abstractmethods__:
                if hasattr(sub, meth_name):
                    meth = getattr(sub, meth_name)
                    if callable(meth):
                        # Check if method is defined in this subclass
                        if meth.__qualname__.startswith(sub.__name__ + '.'):
                            if meth_name in meths:
                                warns.append(
                                    f"Method conflict: {meth_name} implemented in both {sub.__name__} "
                                    f"and {srcs[meth_name]}, using {sub.__name__}'s implementation"
                                )
                            else:
                                srcs[meth_name] = sub.__name__
                            
                            meths[meth_name] = meth
        
        # [DEBUG] collected {len(meths)} methods
        return meths
    
    def __onion_merge_meths(cls, meths: Dict[str, Any], warns: List[str]):
        """Merge method implementations into core class"""
        for meth_name, meth in meths.items():
            setattr(cls, meth_name, meth)
            # [DEBUG] merged method: {meth_name}
    
    def __onion_handle_warns(cls, warns: List[str]):
        """Handle warning messages"""
        for warn in warns:
            warnings.warn(warn, RuntimeWarning)
            # [DEBUG] warning: {warn}
    
    def __onion_raise_errs(cls, errs: List[str]):
        """Handle compilation errors"""
        err_msg = f"Failed to compile class '{cls.__name__}':\n" + "\n".join(f"- {err}" for err in errs)
        # [DEBUG] compile errors: {err_msg}
        raise TypeError(err_msg)
    
    def __onion_compile(cls):
        """
        Core implementation of Construction & Binding (C&B) process
        
        Note: impls module is controlled by user, not auto-loaded here
        """
        errs = []
        warns = []
        
        # No auto-loading of impls module, user controlled
        # [DEBUG] compiling {cls.__name__}, impls loading skipped (user responsibility)
        
        subs = cls.__onion_get__onion_subs__()
        if not subs:
            errs.append(f"No implementation subclasses found for {cls.__name__}")
            cls.__onion_raise_errs(errs)
            return
        
        meths = cls.__onion_get_meths(subs, warns)
        
        # Check if all abstract methods are implemented
        miss_meths = cls.__abstractmethods__ - set(meths.keys())
        if miss_meths:
            errs.append(f"Missing implementations for abstract methods: {miss_meths}")
            cls.__onion_raise_errs(errs)
            return
        
        cls.__onion_merge_meths(meths, warns)
        
        if warns:
            cls.__onion_handle_warns(warns)
    
    # Public API: methods for external use
    def onionCompile(cls):
        """
        Manually trigger Construction & Binding (C&B) process
        
        Should be called after user loads impls modules, or rely on auto-trigger at first instantiation
        
        :param: None
        
        :raises: TypeError: If compilation fails (missing implementation subclasses or abstract methods)
        :warns: RuntimeWarning: For warning situations like method conflicts
        
        :usage:
            # User responsible for loading impls
            import impls
            # Manually trigger compilation
            MyCoreClass.onionCompile()
        """
        if not getattr(cls, '__onion_built__', False):
            cls.__onion_compile()
            cls.__onion_built__ = True
            
            # Recalculate abstract methods
            if hasattr(abc, 'update_abstractmethods'):
                abc.update_abstractmethods(cls)
                # [DEBUG] manually updated abstractmethods for {cls.__name__}
        else:
            # [DEBUG] {cls.__name__} already compiled, skipping
            pass
    
    # Magic methods section
    def __init__(cls, name, bases, namespace, **kwargs):
        super().__init__(name, bases, namespace, **kwargs)

        # Initialize subclass registry (only on core class)
        if not hasattr(cls, '__onion_subs__'):
            cls.__onion_subs__ = []

        # Register non-abstract subclasses (concrete implementations)
        for base in bases:
            if isinstance(base, OnionMeta) and hasattr(base, '__onion_subs__'):
                if not getattr(cls, '__abstractmethods__', None):  # Only register non-abstract classes
                    base.__onion_subs__.append(cls)
                    # [DEBUG] registered subclass: {cls.__name__} to {base.__name__}
    
    def __call__(cls, *args, **kwargs):
        """
        Intercept first instantiation to trigger C&B process
        
        Uses __onion_built__ flag to ensure C&B runs only once
        """
        if not getattr(cls, '__onion_built__', False):
            cls.__onion_compile()
            cls.__onion_built__ = True
            
            # Recalculate abstract methods
            if hasattr(abc, 'update_abstractmethods'):
                abc.update_abstractmethods(cls)
                # [DEBUG] updated abstractmethods for {cls.__name__}
        
        return super().__call__(*args, **kwargs)
