import * as iam from '@aws-cdk/aws-iam';
import * as lambda from '@aws-cdk/aws-lambda';
import * as sfn from '@aws-cdk/aws-stepfunctions';
import { Construct } from 'constructs';
/**
 * Properties for invoking a Lambda function with LambdaInvoke.
 *
 * @stability stable
 */
export interface LambdaInvokeProps extends sfn.TaskStateBaseProps {
    /**
     * Lambda function to invoke.
     *
     * @stability stable
     */
    readonly lambdaFunction: lambda.IFunction;
    /**
     * The JSON that will be supplied as input to the Lambda function.
     *
     * @default - The state input (JSON path '$')
     * @stability stable
     */
    readonly payload?: sfn.TaskInput;
    /**
     * Invocation type of the Lambda function.
     *
     * @default InvocationType.REQUEST_RESPONSE
     * @stability stable
     */
    readonly invocationType?: LambdaInvocationType;
    /**
     * Up to 3583 bytes of base64-encoded data about the invoking client to pass to the function.
     *
     * @default - No context
     * @stability stable
     */
    readonly clientContext?: string;
    /**
     * Version or alias to invoke a published version of the function.
     *
     * You only need to supply this if you want the version of the Lambda Function to depend
     * on data in the state machine state. If not, you can pass the appropriate Alias or Version object
     * directly as the `lambdaFunction` argument.
     *
     * @default - Version or alias inherent to the `lambdaFunction` object.
     * @stability stable
     */
    readonly qualifier?: string;
    /**
     * Invoke the Lambda in a way that only returns the payload response without additional metadata.
     *
     * The `payloadResponseOnly` property cannot be used if `integrationPattern`, `invocationType`,
     * `clientContext`, or `qualifier` are specified.
     * It always uses the REQUEST_RESPONSE behavior.
     *
     * @default false
     * @stability stable
     */
    readonly payloadResponseOnly?: boolean;
    /**
     * Whether to retry on Lambda service exceptions.
     *
     * This handles `Lambda.ServiceException`, `Lambda.AWSLambdaException` and
     * `Lambda.SdkClientException` with an interval of 2 seconds, a back-off rate
     * of 2 and 6 maximum attempts.
     *
     * @default true
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
     * @stability stable
     */
    readonly retryOnServiceExceptions?: boolean;
}
/**
 * Invoke a Lambda function as a Task.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @stability stable
 */
export declare class LambdaInvoke extends sfn.TaskStateBase {
    private readonly props;
    private static readonly SUPPORTED_INTEGRATION_PATTERNS;
    /**
     * @stability stable
     */
    protected readonly taskMetrics?: sfn.TaskMetricsConfig;
    /**
     * @stability stable
     */
    protected readonly taskPolicies?: iam.PolicyStatement[];
    private readonly integrationPattern;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: LambdaInvokeProps);
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    protected _renderTask(): any;
}
/**
 * Invocation type of a Lambda.
 *
 * @stability stable
 */
export declare enum LambdaInvocationType {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     *
     * @stability stable
     */
    REQUEST_RESPONSE = "RequestResponse",
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     *
     * @stability stable
     */
    EVENT = "Event",
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     *
     * @stability stable
     */
    DRY_RUN = "DryRun"
}
