from .download_selenium_driver import DriverType
from typing import Optional, TYPE_CHECKING, Union
import os
import pathlib

if TYPE_CHECKING:
    from selenium import webdriver

def selenium_driver_options(driver_type: DriverType) -> Union['webdriver.ChromeOptions', 'webdriver.FirefoxOptions']:
    """
    Returns the selenium driver. 
    Args:
        driver_type (DriverType): The type of selenium driver to load ('chrome' or 'firefox').
    
    Returns:
        pathlib.Path: The selenium webdriver instance.
    """
    if os.environ.get('HOME') is None:
        os.environ['HOME'] = '/tmp/home'
        if not os.path.exists(os.environ['HOME']):
            os.makedirs(os.environ['HOME'])
    if driver_type == 'chrome':
        # make executable
        from selenium import webdriver
        options = webdriver.ChromeOptions()
        headless = os.environ.get('DATALOG_GUI', '0') == '0'
        if headless:
            options.add_argument('--headless=new')
        options.add_argument("--disable-gpu")
        options.add_argument("--no-sandbox")
        return options
        
    elif driver_type == 'firefox':
        from selenium import webdriver
        options = webdriver.FirefoxOptions()
        options.binary_location="/usr/bin/firefox"
        headless = os.environ.get('DATALOG_GUI', '0') == '0'
        if headless:
            options.add_argument('--headless')
        return options
    raise ValueError(f"Unsupported driver type: {driver_type}")

def selenium_driver(driver_type: DriverType) -> Union['webdriver.Chrome', 'webdriver.Firefox']:
    """
    Returns the selenium driver. 
    Args:
        driver_type (DriverType): The type of selenium driver to load ('chrome' or 'firefox').
    
    Returns:
        pathlib.Path: The selenium webdriver instance.
    """
    options = selenium_driver_options(driver_type)
    if driver_type == 'chrome':
        driver = "/usr/bin/chromedriver"        
        from selenium import webdriver
        from selenium.webdriver.chrome.service import Service as ChromeService
        service = ChromeService(executable_path=str(driver))
        return webdriver.Chrome(service=service, options=options)
    elif driver_type == 'firefox':
        driver = "/usr/bin/geckodriver"
        from selenium import webdriver
        from selenium.webdriver.firefox.service import Service as GeckoService
        service = GeckoService(executable_path=str(driver))
        return webdriver.Firefox(service=service, options=options)
        
class DatallogSeleniumDriver():
    def __init__(self, driver_type: DriverType):
        self.driver_type = driver_type
        self.driver: Optional[pathlib.Path] = None

    def __enter__(self):
        self.driver = selenium_driver(self.driver_type)
        return self.driver

    def __exit__(self, exc_type, exc_value, traceback):
        if self.driver:
            self.driver.quit()