# lib/hba.sh
# shellcheck shell=bash
[[ ${__HBA_LIB_LOADED:-0} -eq 1 ]] && return 0
__HBA_LIB_LOADED=1

# ----- Rule formatter ---------------------------------------------------------
hba_emit_rule() {
	(($# >= 5)) || {
		err "hba_emit_rule needs ≥5 args"
		return 2
	}
	local type="$1" db="$2" user="$3" addr="$4" method="$5"
	shift 5
	if [[ "$type" == "local" ]]; then
		printf "%-7s %-15s %-15s %s%s\n" "$type" "$db" "$user" "$method" "${*:+ $*}"
	else
		printf "%-7s %-15s %-15s %-39s %s%s\n" "$type" "$db" "$user" "$addr" "$method" "${*:+ $*}"
	fi
}

# Split comma/space-separated lists into array (empty-safe)
_hba_parse_list() {
	local s="${1:-}"
	s="${s//,/ }"
	# shellcheck disable=SC2206
	local arr=($s)
	printf '%s\n' "${arr[@]}"
}

# ----- Policy composer --------------------------------------------------------
hba_policy_block() {
	local begin="# pgprovision:hba begin (managed)"
	local end="# pgprovision:hba end"
	local proto="host"
	[[ "${ENABLE_TLS:-false}" == "true" ]] && proto="hostssl"

	local allowed_v4 allowed_v6
	mapfile -t allowed_v4 < <(_hba_parse_list "${ALLOWED_CIDR:-}")
	mapfile -t allowed_v6 < <(_hba_parse_list "${ALLOWED_CIDR_V6:-}")

	printf '%s\n' "$begin"
	printf '# Generated by pgprovision; first-match-wins.\n'
	printf '# Local socket policy\n'
	hba_emit_rule local all postgres "" peer
	hba_emit_rule local all all "" peer "map=${LOCAL_PEER_MAP}"

	if [[ "${SOCKET_ONLY:-false}" == "true" ]]; then
		printf '# Loopback TCP explicitly rejected (socket-only)\n'
		hba_emit_rule host all all "127.0.0.1/32" reject
		hba_emit_rule host all all "::1/128" reject
	else
		printf '# Loopback TCP allowed\n'
		hba_emit_rule host all all "127.0.0.1/32" scram-sha-256
		hba_emit_rule host all all "::1/128" scram-sha-256
	fi

	if [[ "${ALLOW_NETWORK:-false}" == "true" || ${#allowed_v4[@]} -gt 0 || ${#allowed_v6[@]} -gt 0 ]]; then
		printf '# Network access\n'
		local cidr
		for cidr in "${allowed_v4[@]}"; do
			[[ -n "$cidr" ]] && hba_emit_rule "$proto" all all "$cidr" scram-sha-256
		done
		for cidr in "${allowed_v6[@]}"; do
			[[ -n "$cidr" ]] && hba_emit_rule "$proto" all all "$cidr" scram-sha-256
		done
	fi

	printf '%s\n' "$end"
}

apply_hba_policy() {
	local hba_file="$1"
	local begin="# pgprovision:hba begin (managed)"
	local end="# pgprovision:hba end"
	local block
	block="$(hba_policy_block)"
	replace_managed_block_top "$hba_file" "$begin" "$end" "$block" "${CONF_FILE:-}" "0600"
}
