import click
from rich.console import Console
from rich.prompt import Prompt, Confirm
from rich.panel import Panel
from rich.progress import Progress, SpinnerColumn, TextColumn
from pathlib import Path
import sys

from .generator import ProjectGenerator
from .config import AVAILABLE_TEMPLATES

console = Console()


@click.group()
@click.version_option(version="0.1.0", prog_name="Projex")
def main():
    """
    🔨 Projex - Instantly generate Python project boilerplates!
    """
    pass


@main.command()
@click.argument("project_name", required=False)
@click.option(
    "--template",
    "-t",
    type=click.Choice(["fastapi", "django", "flask"]),
    help="Choose the project template type",
)
@click.option("--path", "-p", default=".", help="Directory path to create the project")
@click.option("--author", "-a", help="Author name")
@click.option("--description", "-d", help="Project description")
@click.option("--no-git", is_flag=True, help="Skip git initialization")
@click.option("--no-venv", is_flag=True, help="Skip virtual environment creation")
def create(project_name, template, path, author, description, no_git, no_venv):
    """Create a new Python project using Projex templates."""

    console.print(
        Panel.fit(
            "[bold cyan]🔨 Projex[/bold cyan]\n"
            "[dim]Generate production-ready Python projects in seconds[/dim]",
            border_style="cyan",
        )
    )

    # Interactive mode if not provided via CLI
    if not project_name:
        project_name = Prompt.ask("[cyan]Project name[/cyan]")

    if not template:
        console.print("\n[yellow]Available templates:[/yellow]")
        for i, (key, value) in enumerate(AVAILABLE_TEMPLATES.items(), 1):
            console.print(f"  {i}. [cyan]{key}[/cyan] - {value['description']}")

        template_choice = Prompt.ask(
            "\n[cyan]Choose template[/cyan]",
            choices=["1", "2", "3", "fastapi", "django", "flask"],
            default="1",
        )

        template_map = {"1": "fastapi", "2": "django", "3": "flask"}
        template = template_map.get(template_choice, template_choice)

    if not author:
        author = Prompt.ask("[cyan]Author name[/cyan]", default="Developer")

    if not description:
        description = Prompt.ask(
            "[cyan]Project description[/cyan]",
            default=f"A {template} project generated by Projex",
        )

    # Show summary
    console.print("\n[bold yellow]Project Configuration:[/bold yellow]")
    console.print(f"  Name: [cyan]{project_name}[/cyan]")
    console.print(f"  Template: [cyan]{template}[/cyan]")
    console.print(f"  Path: [cyan]{Path(path).absolute() / project_name}[/cyan]")
    console.print(f"  Author: [cyan]{author}[/cyan]")
    console.print(f"  Git Init: [cyan]{'No' if no_git else 'Yes'}[/cyan]")
    console.print(f"  Virtual Env: [cyan]{'No' if no_venv else 'Yes'}[/cyan]")

    if not Confirm.ask("\n[cyan]Proceed with project creation?[/cyan]", default=True):
        console.print("[yellow]Cancelled by user.[/yellow]")
        return

    # Project generation logic
    try:
        generator = ProjectGenerator(
            project_name=project_name,
            template_type=template,
            base_path=path,
            author=author,
            description=description,
            init_git=not no_git,
            create_venv=not no_venv,
        )

        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            console=console,
        ) as progress:
            task = progress.add_task("[cyan]Creating project...", total=None)
            project_path = generator.generate()
            progress.update(task, completed=True)

        console.print(
            f"\n[bold green]✓[/bold green] Project created successfully at: [cyan]{project_path}[/cyan]"
        )

        # Post-creation guidance
        console.print("\n[yellow]Next steps:[/yellow]")
        console.print(f"  cd {project_name}")
        if not no_venv:
            console.print("  source venv/bin/activate  # (Windows: venv\\Scripts\\activate)")
        console.print("  pip install -r requirements.txt")

        if template == "fastapi":
            console.print("  uvicorn app.main:app --reload")
        elif template == "django":
            console.print("  python manage.py migrate")
            console.print("  python manage.py runserver")
        elif template == "flask":
            console.print("  flask run")

    except Exception as e:
        console.print(f"[bold red]✗ Error:[/bold red] {str(e)}")
        sys.exit(1)


@main.command()
def list():
    """List all available Projex templates."""
    console.print("\n[bold cyan]Available Templates:[/bold cyan]\n")

    for name, details in AVAILABLE_TEMPLATES.items():
        console.print(
            Panel(
                f"[bold]{details['description']}[/bold]\n"
                f"[dim]Features: {', '.join(details['features'])}[/dim]",
                title=f"[cyan]{name}[/cyan]",
                border_style="cyan",
            )
        )


if __name__ == "__main__":
    main()
