import pytest
import pytest_mock
import csle_common.constants.constants as constants
from csle_common.dao.training.experiment_config import ExperimentConfig
from csle_common.dao.training.agent_type import AgentType
from csle_common.dao.training.hparam import HParam
from csle_common.dao.training.player_type import PlayerType
from csle_agents.agents.lp_cmdp.linear_programming_cmdp_agent import LinearProgrammingCMDPAgent
import csle_agents.constants.constants as agents_constants
from csle_common.dao.simulation_config.simulation_env_config import SimulationEnvConfig


class TestLPCMDPSuite:
    """
    Test suite for the LPCMDPAgent
    """

    @pytest.fixture
    def experiment_config(self) -> ExperimentConfig:
        """
        Fixture, which is run before every test. It sets up an example experiment config

        :return: the example experiment config
        """
        states = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20]
        actions = [0, 1]
        cost_tensor = [0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0, 13.0, 14.0, 15.0, 16.0,
                       17.0, 18.0, 19.0, 20.0]
        transition_tensor = [[[0.3334508872586958, 0.22230056675992774, 0.14820034223004194, 0.0988001769098325,
                               0.06586671086586718, 0.04391103446259648, 0.029273870273497236, 0.019515693774385355,
                               0.013010146303196206, 0.00867297583049061, 0.0057813290757716745, 0.003853277094528227,
                               0.002567495419038641, 0.0017097123282656694, 0.0011370002729613705,
                               0.0007539594793629763, 0.0004968249832973096, 0.0003228492041396098,
                               0.00020319183612257222, 0.0001181339794087175, 5.3821658572311436e-05],
                              [0.18813544683753372, 0.27073152680570955, 0.18048766444576408, 0.12032508071852965,
                               0.08021667887397595, 0.05347772604531107, 0.03565173120818217, 0.02376769682579245,
                               0.015844952807436988, 0.010563045135631477, 0.007041660641064936, 0.004693908780742312,
                               0.00312850746802111, 0.0020845707161778423, 0.0013881295468582925, 0.0009231399034740852,
                               0.0006121459225244671, 0.00040337630344630036, 0.00026212393280247714,
                               0.0001649731283673018, 9.59139526539745e-05],
                              [0.11563176668871723, 0.16639697253428382, 0.2394493285943973, 0.15963286795971943,
                               0.10642188640185729, 0.07094788747019631, 0.047298538693997395, 0.03153228292896068,
                               0.02102141229844815, 0.014014117070574896, 0.009342517642780606, 0.006228018334571864,
                               0.004151542006552854, 0.0027670180180301577, 0.0018437049585087316,
                               0.0012277354319203482, 0.0008164739167751211, 0.0005414143372207746,
                               0.00035676740781070173, 0.00023183631569853943, 0.0001459109889777146],
                              [0.07438772220577869, 0.10704580210896521, 0.15404155712146422, 0.22166958248456367,
                               0.14777970520595754, 0.09851977979808524, 0.06567981913334213, 0.043786497067980136,
                               0.02919092750302829, 0.01946051682327376, 0.012973531803825735, 0.008648810992232216,
                               0.0057655714970463925, 0.003843279029687918, 0.0025615596100625757,
                               0.0017068049878295338, 0.0011365728281338642, 0.0007558485684779336,
                               0.0005012128903738927, 0.0003302764839548134, 0.0002146218559364012],
                              [0.049162304168785295, 0.07074583398246416, 0.10180503341090695, 0.14649996132922058,
                               0.21081704099009022, 0.14054467834835788, 0.09369642971865087, 0.062464254081498596,
                               0.041642789433393015, 0.027761792533593132, 0.018507765146831254, 0.012338371171262654,
                               0.00822538086204015, 0.005483299553328506, 0.0036551190454580905, 0.002436150288462454,
                               0.0016232429052660217, 0.0010809282798807324, 0.0007188435907944644,
                               0.00047667441455154366, 0.00031410674516343907],
                              [0.033044997683374135, 0.04755267255661902, 0.06842953224005026, 0.09847181697960951,
                               0.1417033804340889, 0.20391464332386247, 0.13594308041644296, 0.09062869850116223,
                               0.06041910099724174, 0.040279355570312786, 0.026852838821434844, 0.017901799166298485,
                               0.011934398399528428, 0.007956072226460506, 0.00530376988218631, 0.0035354461525521513,
                               0.002356387865146525, 0.001570096024973951, 0.0010455374177314103, 0.0006953078068740571,
                               0.00046106753404938513],
                              [0.022457996920655955, 0.03231777390347747, 0.046506177271246035, 0.0669235983141657,
                               0.09630474020619414, 0.13858490335229856, 0.19942707824320685, 0.1329513706957011,
                               0.08863422583323731, 0.059089453241591275, 0.039392924725610914, 0.02626188635305246,
                               0.017507832909093492, 0.011671757185325397, 0.007780982339237772, 0.005187049414586878,
                               0.0034576413199019834, 0.002304530657994815, 0.0015355428022131952, 0.001022528196177475,
                               0.0006800061150309672],
                              [0.01537671267856714, 0.022127713735322557, 0.03184248586485868, 0.045822224538458044,
                               0.0659393725480269, 0.0948884145886253, 0.1365467757543641, 0.196494162592866,
                               0.1309960938131298, 0.08733070822473518, 0.058220441953309776, 0.038813584515353466,
                               0.02587566047956096, 0.01725035033654127, 0.011500104069723556, 0.007666549710134389,
                               0.005110764997540058, 0.003406790801357864, 0.002270638600341137, 0.0015129600238059065,
                               0.001007490173377851],
                              [0.010582834338694029, 0.015229326872787666, 0.02191561957793586, 0.031537275607288154,
                               0.04538301847228617, 0.06530734359887898, 0.09397890904373891, 0.13523797477772892,
                               0.19461076585585865, 0.12974049612822774, 0.086493643302599, 0.05766239896131926,
                               0.03844155627052851, 0.025627642249034212, 0.017085005712012182, 0.011389875561179815,
                               0.0075930658238059666, 0.005061778310134374, 0.00337413670825771, 0.002248874527177974,
                               0.0014984583005258987],
                              [0.007310979003003242, 0.010521229082101385, 0.0151406732512998, 0.021788043417875495,
                               0.031353689443710385, 0.04511883286676163, 0.06492717364336123, 0.09343183492156659,
                               0.13445072158348875, 0.19347788918194087, 0.12898524509635506, 0.08599014273566616,
                               0.05732673209079375, 0.038217778607373805, 0.025478457500781686, 0.016985549731970338,
                               0.011323572321041276, 0.007548864738025943, 0.005032312465576006, 0.003354495036561985,
                               0.0022357832807445963],
                              [0.005065528329928695, 0.007290244438371869, 0.010491389972412423, 0.015097733001032008,
                               0.02172625065465547, 0.031264767686455204, 0.044990871980020775, 0.06474303460019239,
                               0.09316685420981607, 0.13406940778471269, 0.1929291692640836, 0.12861943185850483,
                               0.08574626730236491, 0.05716414855291822, 0.038109389703469596, 0.025406198406132313,
                               0.016937377253486638, 0.011291457696987209, 0.007527455509010271, 0.005018040395033897,
                               0.003344981400411117],
                              [0.0035186391047108447, 0.005064645709559135, 0.007288974182275902, 0.010489561947003914,
                               0.015095102363946797, 0.021722465061103733, 0.03125932009658653, 0.0449830327463876,
                               0.06473175373916398, 0.09315062076723368, 0.13404604745928586, 0.19289555318228832,
                               0.12859702113980273, 0.0857313268268201, 0.05715418824105434, 0.038102749502994346,
                               0.025401771616513177, 0.01693442607580147, 0.011289490267349808, 0.007526143921130294,
                               0.00501716604898755],
                              [0.0024501703288988277, 0.0035277169814481365, 0.005077712190121997, 0.007307779296184691,
                               0.010516624384369424, 0.015134046817901528, 0.021778507711226844, 0.03133996725768597,
                               0.04509908625866583, 0.064898757760874, 0.09339094375238867, 0.13439187817955708,
                               0.19339321207905222, 0.1289287937007133, 0.08595250848094778, 0.05730164260066106,
                               0.03820105229934449, 0.0254673066557091, 0.01697811587403258, 0.011318616471550929,
                               0.007545560918665412],
                              [0.0017107185943459945, 0.002464575380631061, 0.0035484571500050206, 0.005107565097047439,
                               0.007350743223046294, 0.010578453766808068, 0.015223022969787212, 0.021906548005622237,
                               0.03152422132537617, 0.04536423299682115, 0.06528031081123817, 0.09394000818281127,
                               0.1351819954766017, 0.1945302102672204, 0.1296867924084472, 0.08645784083134798,
                               0.057638530659531395, 0.03842564408715099, 0.02561703415241789, 0.017077933684491176,
                               0.011385160929251249],
                              [0.0011980955153864682, 0.0017283464930347272, 0.002489971307970659,
                               0.0035850218096447396, 0.005160195401280078, 0.007426488092629766, 0.010687458200325689,
                               0.015379886820764561, 0.02213228145479341, 0.03184905895887217, 0.04583168340392085,
                               0.06595298410137061, 0.09490800195603523, 0.13657496246058032, 0.19653472395510693,
                               0.13102313471828034, 0.08734873549050395, 0.05823246012425562, 0.03882159662034748,
                               0.025881001869982368, 0.017253911244914082],
                              [0.0008419665618246663, 0.0012181038141205682, 0.001757210028958671, 0.00253155404417942,
                               0.0036448919839459086, 0.0052463711108034035, 0.007550511086117593, 0.010865939676663862,
                               0.015636732260927357, 0.02250189246938088, 0.032380940998213176, 0.046597076480896484,
                               0.06705440464034985, 0.09649297379760023, 0.13885577614648112, 0.19981687084407954,
                               0.1332112324006892, 0.08880746692826931, 0.059204947245044, 0.039469920641712404,
                               0.026313216839742422],
                              [0.0005936031801186476, 0.0008642068294948989, 0.0012502796226437271,
                               0.0018036261494660752, 0.0025984242052012486, 0.003741170597643305, 0.005384952264844461,
                               0.007749955333925246, 0.011152959871829205, 0.016049771360980368, 0.0230962725009599,
                               0.03323627282238156, 0.04782792281206141, 0.06882562451449496, 0.09904180371895084,
                               0.14252360545120202, 0.20509496726036705, 0.13672996295318576, 0.09115328673294146,
                               0.06076882630370667, 0.04051250551360108],
                              [0.0004188845483121686, 0.0006184077774106705, 0.000900319005272583,
                               0.0013025244278954467, 0.0018789933675020802, 0.0027070032495233824,
                               0.0038975010102545566, 0.005609970554570496, 0.008073798816453015, 0.01161900299204888,
                               0.016720435078042307, 0.024061384814237315, 0.034625100225028224, 0.04982648414798624,
                               0.07170160624207401, 0.10318041371735605, 0.14847916761162988, 0.21366516742084657,
                               0.14244342909083196, 0.09496226324280956, 0.06330814265991462],
                              [0.00029329174583083556, 0.00044706451716239543, 0.0006600104385123725,
                               0.0009608869150369251, 0.0013901502378057208, 0.00200540045217337, 0.00288911373212868,
                               0.004159700839551793, 0.0059873747727864236, 0.008616954203939608, 0.012400657850663098,
                               0.01784528282322657, 0.025680086380792156, 0.03695446341044255, 0.053178502685944276,
                               0.07652524807501908, 0.11012175556494724, 0.158467930231456, 0.22803924205921408,
                               0.15202614444968954, 0.10135073861367745],
                              [0.00019774950816067177, 0.0003263050003290338, 0.0004973866107500853,
                               0.0007343019686621634, 0.001069045445650327, 0.0015466271391975563,
                               0.0022311306216701503, 0.0032143156795711976, 0.004627921525624897, 0.006661320528028022,
                               0.009586888428744068, 0.013796490087295988, 0.019853968276567973, 0.028570666287237215,
                               0.04111410009560504, 0.05916433579567245, 0.08513901755933147, 0.12251718637380078,
                               0.17630526177896985, 0.25370760007028204, 0.1691383812188491],
                              [0.00011301679225330372, 0.00023797844872850134, 0.00039268647751863417,
                               0.0005985718758322202, 0.0008836838332792373, 0.0012865254591148008,
                               0.001861263427510434, 0.0026850180776394777, 0.0038682162411576727, 0.005569397343889257,
                               0.0080164585009001, 0.01153718589556255, 0.01660316295808527, 0.023892937158274008,
                               0.034382906462917945, 0.04947809909934483, 0.0712003634480162, 0.10245917430336184,
                               0.147441327298413, 0.21217171709343308, 0.30532030980476776]],
                             [
                                 [0.0, 0.3334688351044833, 0.22231253198912077, 0.14820831904758727, 0.0988054947854379,
                                  0.06587025611230173, 0.04391339795450793, 0.029275445926552447, 0.019516744197928017,
                                  0.013010846568537952, 0.008673442649559542, 0.0057816402532393715,
                                  0.0038534844954550545, 0.0025676336133383165, 0.0017098043527717696,
                                  0.001137061471495665, 0.0007540000608968173, 0.0004968517246811963,
                                  0.0003228665813545158, 0.00020320277283283254, 0.00011814033791764851],
                                 [0.0, 0.1881534933827897, 0.2707574962273834, 0.18050497739161941, 0.12033662267965989,
                                  0.08022437351073855, 0.053482855797409815, 0.035655151034650376, 0.023769976698212066,
                                  0.015846472705269773, 0.010564058376226407, 0.007042336099356143,
                                  0.004694359035272562, 0.003128807564321698, 0.002084770674773637,
                                  0.001388262700621231, 0.0009232284539643074, 0.0006122046414914657,
                                  0.00040341499657318156, 0.0002621490765565943, 0.00016498895310993882],
                                 [0.0, 0.11564864109631355, 0.16642125522422052, 0.23948427198136674,
                                  0.1596561635484392, 0.10643741679060545, 0.07095824105732511, 0.047305441077689074,
                                  0.03153688450696838, 0.021024480001117857, 0.014016162182660642, 0.00934388101770079,
                                  0.0062289272035003975, 0.004152147850552111, 0.0027674218152841444,
                                  0.0018439740145808484, 0.0012279145981537452, 0.0008165930666770723,
                                  0.0005414933470505677, 0.000356819471692703, 0.0002318701481011633],
                                 [0.0, 0.07440369086400044, 0.10706878140954416, 0.15407462490341367,
                                  0.22171716783461728, 0.1478114287691286, 0.09854092883511752, 0.06569391848405091,
                                  0.043795896624596105, 0.029197193858962432, 0.019464694372105136,
                                  0.012976316805021646, 0.008650667614570744, 0.005766809180335506,
                                  0.0038441040584353517, 0.0025621094947574524, 0.0017071713841203952,
                                  0.0011368168138683162, 0.0007560108249242869, 0.0005013204847067393,
                                  0.00033034738372341186],
                                 [0.0, 0.049177751232159284, 0.07076806270830299, 0.10183702110614373,
                                  0.14654599241391444, 0.21088328085104774, 0.14058883825074714, 0.09372586964650315,
                                  0.06248388068988727, 0.04165587382433692, 0.02777051543980944, 0.018513580387308035,
                                  0.012342247954595672, 0.008227965321446588, 0.005485022435872986,
                                  0.0036562675037431383, 0.0024369157401338245, 0.0016237529370160173,
                                  0.0010812679134256681, 0.0007190694553606342, 0.0004768241882453286],
                                 [0.0, 0.03306024068702278, 0.04757460766365787, 0.06846109742942036,
                                  0.0985172400805548, 0.14176874539993572, 0.2040087051144542, 0.136005788269857,
                                  0.09067050372672653, 0.06044697113316286, 0.04029793564012566, 0.02686522550470998,
                                  0.017910056912073622, 0.011939903501392603, 0.007959742205170713,
                                  0.0053062164062998955, 0.0035370769839148677, 0.0023574748202484803,
                                  0.0015708202792065196, 0.0010460197034566502, 0.000695628538608968],
                                 [0.0, 0.022473278887724407, 0.032339765141532376, 0.046537823275653714,
                                  0.06696913773734545, 0.09637027258085634, 0.1386792060604473, 0.19956278215540513,
                                  0.13304183996042918, 0.08869453866189726, 0.05912966177317676, 0.03941973038332445,
                                  0.026279756748342865, 0.017519746443809076, 0.011679699452374737,
                                  0.007786277055248667, 0.00519057904007468, 0.003459994137073165,
                                  0.0023060988192938007, 0.0015365876912395195, 0.0010232239947509522],
                                 [0.0, 0.015392220189154182, 0.022150029672557685, 0.03187459921034346,
                                  0.04586843653753781, 0.06600587281627451, 0.09498411014622467, 0.13668448402887642,
                                  0.19669232818068685, 0.13112820419030422, 0.08741878178835562, 0.05827915763193669,
                                  0.0388527282572816, 0.025901756244450472, 0.017267747422385696, 0.011511701996363746,
                                  0.007674281473306481, 0.00511591923589802, 0.003410226571117257, 0.002272928553523602,
                                  0.0015144858534209477],
                                 [0.0, 0.010598716072768187, 0.015252181630953697, 0.02194850850268587,
                                  0.03158460381904963, 0.04545112508794244, 0.06540535078966857, 0.09411994385485326,
                                  0.13544092735956179, 0.1949028196036897, 0.12993519860510805, 0.08662344492266351,
                                  0.05774893332981383, 0.03849924578493894, 0.025666101832367064, 0.017110645300490057,
                                  0.011406968427704145, 0.007604460791199563, 0.005069374556517062,
                                  0.0033792002989948785, 0.0022522494290297583],
                                 [0.0, 0.007327361395102385, 0.010544804980776116, 0.01517460036909701,
                                  0.02183686591759792, 0.03142394657808468, 0.04521993484103558, 0.06507266201312371,
                                  0.09364119634273861, 0.13475199784732272, 0.19391143312206038, 0.12927427435759417,
                                  0.0861828288635265, 0.057455189442741854, 0.03830341674613019, 0.025535549455318517,
                                  0.01702361083645639, 0.011348946104996899, 0.0075657801828645805,
                                  0.005043588837173108, 0.003362011766258653],
                                 [0.0, 0.005082529295890494, 0.00731471201400809, 0.010526601257829407,
                                  0.015148404131096438, 0.02179916846772444, 0.03136969874529472, 0.045141870697884966,
                                  0.06496032568136119, 0.09347954153757823, 0.13451937258400115, 0.1935766796570899,
                                  0.1290511054057897, 0.08603404959807252, 0.057356003317216235, 0.038237292736475186,
                                  0.02549146688874436, 0.016994222612038354, 0.011329354175984548, 0.007552719214304648,
                                  0.005034881981615667],
                                 [0.0, 0.0035363817190077105, 0.0050901840079469055, 0.007325728578986491,
                                  0.010542455195282659, 0.015171218888274808, 0.02183199982942945, 0.03141694412199836,
                                  0.04520985811158458, 0.06505816133743572, 0.09362032950592585, 0.13472196995298671,
                                  0.19386822224491307, 0.12924546710936938, 0.0861636240359912, 0.05744238623102547,
                                  0.038294881281208434, 0.025529859159121756, 0.017019817325446673,
                                  0.011346417126132692, 0.00756409423793219],
                                 [0.0, 0.002468798800645023, 0.0035545379641946763, 0.005116317676831776,
                                  0.007363339825402097, 0.010596581535877997, 0.015249109905650034, 0.02194408816528255,
                                  0.031578242812532346, 0.04544197142229702, 0.06539217842679765, 0.09410098849357357,
                                  0.13541365012580014, 0.19486356699463872, 0.12990903020198716, 0.08660599932477477,
                                  0.05773730293725355, 0.03849149219857921, 0.025660932787285342, 0.017107199288411038,
                                  0.011404671112185268],
                                 [0.0, 0.001730419701118374, 0.0024929581099021794, 0.003589322160428426,
                                  0.005166385224248009, 0.007435396407720974, 0.010700278155597296,
                                  0.015398335497468493, 0.02215882984946226, 0.03188726294560523, 0.04588666000548949,
                                  0.06603209686049076, 0.09502184720503531, 0.13673878858996935, 0.19677047377729998,
                                  0.13118030124891372, 0.08745351315241663, 0.05830231186263489, 0.03886816439379898,
                                  0.02591204697726032, 0.017274607875139347],
                                 [0.0, 0.0012191302810517217, 0.0017586907877945829, 0.002533687324184502,
                                  0.003647963447187198, 0.0052507921021765274, 0.007556873721102695,
                                  0.010875096143973719, 0.01564990896096809, 0.02252085427562468, 0.03240822764221593,
                                  0.04663634272203422, 0.0671109097823202, 0.09657428611724307, 0.1389727865857899,
                                  0.19998525173890172, 0.13332348631807492, 0.08888230285521133, 0.059254837837129225,
                                  0.03950318099919944, 0.026335390357816298],
                                 [0.0, 0.0008647201301140475, 0.0012510222334198844, 0.0018046974235958735,
                                  0.002599967554209633, 0.003743392687447008, 0.0053881506882279525,
                                  0.007754558464490183, 0.011159584236520807, 0.016059304215032905, 0.02310999066491111,
                                  0.033256013697870686, 0.04785633048202437, 0.06886650389020785, 0.09910063016817042,
                                  0.14260825816676106, 0.20521678459631715, 0.13681117450144556, 0.09120742775210561,
                                  0.06080492029776227, 0.040536568149365654],
                                 [0.0, 0.0006186669274271216, 0.0009006962930324566, 0.001303070263894357,
                                  0.0018797807786244602, 0.0027081376465382194, 0.003899134297363516,
                                  0.005612321469314153, 0.008077182223280249, 0.011623872052443205,
                                  0.016727441945005864, 0.024071467980229418, 0.03463961022251, 0.0498473644387237,
                                  0.07173165352335982, 0.103223652510413, 0.14854138930439428, 0.21375470596430413,
                                  0.14250312144649216, 0.09500205813702098, 0.06333467257562896],
                                 [0.0, 0.0004471956759629265, 0.0006602040709169362, 0.0009611688179175701,
                                  0.0013905580770118067, 0.00200598879212833, 0.0028899613339336935,
                                  0.004160921203395802, 0.005989131335571843, 0.008619482226930104,
                                  0.012404295928271704, 0.017850518232883073, 0.02568762034781021, 0.036965305029289755,
                                  0.0531941040775971, 0.07654769888326386, 0.11015406284234866, 0.15851442120278988,
                                  0.22810614370833696, 0.15207074554413993, 0.10138047266949983],
                                 [0.0, 0.0003263695397449969, 0.0004974849881618116, 0.0007344472052357686,
                                  0.0010692568906745554, 0.001546933044446253, 0.0022315719139185527,
                                  0.0032149514346361562, 0.0046288368758404495, 0.0066626380614277225,
                                  0.009588784606183794, 0.013799218876041724, 0.01985789516557007, 0.028576317239916448,
                                  0.041122231996756846, 0.05917603782804783, 0.08515585708818764, 0.1225414188791134,
                                  0.17634013295353038, 0.2537577805465771, 0.16917183486598852],
                                 [0.0, 0.00023800534732939567, 0.00039273086270095553, 0.0005986395321518602,
                                  0.0008837837156798293, 0.0012866708745297257, 0.0018614738053087734,
                                  0.002685321564068818, 0.0038686534639625096, 0.0055700268504566604,
                                  0.008017364597729257, 0.011538489938682865, 0.01660503960639682, 0.02389563776660324,
                                  0.03438679274793018, 0.049483691587436894, 0.07120841119423083, 0.10247075521941651,
                                  0.1474579925277206, 0.2121956987706381, 0.3053548200270263],
                                 [0.0, 0.00016268906929111683, 0.00034257291826340864, 0.0005652770378363499,
                                  0.0008616516133701822, 0.0012720737999852674, 0.001851969299337421,
                                  0.002679311708375044, 0.003865116708514801, 0.005568345088756736,
                                  0.008017216312058924, 0.011539791092276154, 0.016607921691679437,
                                  0.023900458286637293, 0.03439417834651126, 0.04949461881238387, 0.07122433518308206,
                                  0.10249380319151997, 0.14749124776422756, 0.21224361296207783, 0.30542380911381545]]]
        constraint_cost_tensor = [0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
                                  1.0, 1.0, 1.0, 1.0, 1.0, 1.0]
        experiment_config = ExperimentConfig(
            output_dir=f"{constants.LOGGING.DEFAULT_LOG_DIR}fp_test",
            title="Linear programming for normal-form games to approximate a Nash equilibrium",
            random_seeds=[399], agent_type=AgentType.LINEAR_PROGRAMMING_NORMAL_FORM,
            log_every=1, br_log_every=5000,
            hparams={
                agents_constants.COMMON.EVAL_BATCH_SIZE: HParam(value=1,
                                                                name=agents_constants.COMMON.EVAL_BATCH_SIZE,
                                                                descr="number of iterations to evaluate theta"),
                agents_constants.COMMON.SAVE_EVERY: HParam(
                    value=10000, name=agents_constants.COMMON.SAVE_EVERY, descr="how frequently to save the model"),
                agents_constants.COMMON.CONFIDENCE_INTERVAL: HParam(
                    value=0.95, name=agents_constants.COMMON.CONFIDENCE_INTERVAL,
                    descr="confidence interval"),
                agents_constants.COMMON.MAX_ENV_STEPS: HParam(
                    value=50, name=agents_constants.COMMON.MAX_ENV_STEPS,
                    descr="maximum number of steps in the environment (for envs with infinite horizon generally)"),
                agents_constants.COMMON.RUNNING_AVERAGE: HParam(
                    value=40, name=agents_constants.COMMON.RUNNING_AVERAGE,
                    descr="the number of samples to include when computing the running avg"),
                agents_constants.COMMON.GAMMA: HParam(
                    value=1, name=agents_constants.COMMON.GAMMA,
                    descr="the discount factor gamma"),
                agents_constants.LP_FOR_CMDPs.STATES: HParam(
                    value=states, name=agents_constants.LP_FOR_CMDPs.STATES,
                    descr="the state space"),
                agents_constants.LP_FOR_CMDPs.ACTIONS: HParam(
                    value=actions, name=agents_constants.LP_FOR_CMDPs.ACTIONS,
                    descr="the action space"),
                agents_constants.LP_FOR_CMDPs.COST_TENSOR: HParam(
                    value=cost_tensor, name=agents_constants.LP_FOR_CMDPs.COST_TENSOR,
                    descr="the cost tensor"),
                agents_constants.LP_FOR_CMDPs.TRANSITION_TENSOR: HParam(
                    value=transition_tensor, name=agents_constants.LP_FOR_CMDPs.TRANSITION_TENSOR,
                    descr="the transition tensor"),
                agents_constants.LP_FOR_CMDPs.CONSTRAINT_COST_TENSORS: HParam(
                    value=[constraint_cost_tensor],
                    name=agents_constants.LP_FOR_CMDPs.CONSTRAINT_COST_TENSORS,
                    descr="the constraint cost tensor"),
                agents_constants.LP_FOR_CMDPs.CONSTRAINT_COST_THRESHOLDS: HParam(
                    value=[0.2],
                    name=agents_constants.LP_FOR_CMDPs.CONSTRAINT_COST_THRESHOLDS,
                    descr="the constraint cost thresholds")
            },
            player_type=PlayerType.SELF_PLAY, player_idx=1
        )
        return experiment_config

    def test_create_agent(self, mocker: pytest_mock.MockFixture, experiment_config: ExperimentConfig) -> None:
        """
        Tests creation of the LP-CMDP agent

        :return: None
        """
        simulation_env_config = mocker.MagicMock()
        experiment_config.agent_type = AgentType.LINEAR_PROGRAMMING_CMDP
        LinearProgrammingCMDPAgent(simulation_env_config=simulation_env_config,
                                   experiment_config=experiment_config)

    def test_run_agent(self, mocker: pytest_mock.MockFixture, experiment_config: ExperimentConfig,
                       example_simulation_config: SimulationEnvConfig) -> None:
        """
        Tests running the agent

        :param mocker: object for mocking API calls
        :param experiment_config: the example experiment config
        :param example_simulation_config: the example_simulation_config fixture

        :return: None
        """
        # Mock metastore facade
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.save_training_job', return_value=True)
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.save_experiment_execution',
                     return_value=True)
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.update_training_job', return_value=True)
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.update_experiment_execution',
                     return_value=True)
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.save_simulation_trace', return_value=True)
        mocker.patch('csle_common.metastore.metastore_facade.MetastoreFacade.save_tabular_policy', return_value=True)
        experiment_config.agent_type = AgentType.LINEAR_PROGRAMMING_CMDP
        agent = LinearProgrammingCMDPAgent(simulation_env_config=example_simulation_config,
                                           experiment_config=experiment_config)
        experiment_execution = agent.train()
        assert experiment_execution is not None
        assert experiment_execution.descr != ""
        assert experiment_execution.id is not None
        assert experiment_execution.config == experiment_config
        for seed in experiment_config.random_seeds:
            assert seed in experiment_execution.result.all_metrics
