"""API client for OmnibusX SDK."""

import json

import httpx


class ApiClient:
    """Client for interacting with the OmnibusX API."""

    def __init__(
        self, base_url: str, token: str, user_email: str = "", enable_https: bool = True
    ) -> None:
        """Initialize the client with base URL and authentication token."""
        self.base_url = base_url
        self.token = token
        self.user_email = user_email
        self.client = httpx.Client(
            base_url=self.base_url,
            headers={
                "Authorization": f"Bearer {self.token}",
                "User-Agent": "OmnibusX-SDK/1.0",
                "OmnibusX-Email": self.user_email,
                # Note: Content-Type is set per-request, not as default
            },
            verify=enable_https,
            timeout=30,
        )

    def get(
        self, endpoint: str, params: dict | None = None, group_id: str = ""
    ) -> dict:
        """Send a GET request to the specified endpoint."""
        headers = {"OmnibusX-GroupId": group_id} if group_id else {}
        response = self.client.get(endpoint, params=params, headers=headers)
        response.raise_for_status()
        return response.json()

    def post(
        self, endpoint: str, data: dict | None = None, group_id: str = ""
    ) -> dict:
        """Send a POST request to the specified endpoint."""
        headers = {"Content-Type": "application/json"}
        if group_id:
            headers["OmnibusX-GroupId"] = group_id
        response = self.client.post(endpoint, json=data, headers=headers)
        response.raise_for_status()
        return response.json()

    def post_upload(
        self,
        endpoint: str,
        file_data: bytes,
        params: dict,
        extra_headers: dict | None = None,
        group_id: str = "",
    ) -> dict:
        """Send a POST request with binary file data as multipart/form-data."""
        # Important: Do NOT set Content-Type header here
        # Let httpx automatically set it with the proper multipart boundary
        headers = {
            "OmnibusX-GroupId": group_id,
        }
        if extra_headers:
            headers.update(extra_headers)

        # Prepare multipart form data
        # Separate data and files like the GUI client does
        data = {
            "params": json.dumps(params),  # Form field with JSON string
        }
        files = {
            "file": ("chunk", file_data, "application/octet-stream"),  # File upload
        }

        response = self.client.post(endpoint, data=data, files=files, headers=headers)
        response.raise_for_status()
        return response.json()
