# Copyright 2021 Vincent Texier <vit@free.fr>
#
# This software is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
import logging
import sys
from datetime import datetime
from typing import Optional

import qrcode
from PyQt5.QtCore import QEvent, QLocale, QMutex, QPoint, pyqtSignal
from PyQt5.QtGui import QDesktopServices, QFont, QPixmap, QWheelEvent
from PyQt5.QtWebEngineWidgets import QWebEngineView
from PyQt5.QtWidgets import QApplication, QListView, QMainWindow, QSizePolicy, QWidget

from tikka.domains.application import Application
from tikka.domains.entities.account import Account
from tikka.domains.entities.address import DisplayAddress
from tikka.domains.entities.constants import AMOUNT_UNIT_KEY, DATA_PATH
from tikka.domains.entities.events import (
    AccountEvent,
    ConnectionsEvent,
    TransferEvent,
    UnitEvent,
)
from tikka.domains.entities.identity import IdentityStatus
from tikka.domains.entities.profile import Profile
from tikka.domains.entities.smith import SmithStatus
from tikka.slots.pyqt.entities.constants import (
    ADDRESS_MONOSPACE_FONT_NAME,
    ICON_ACCOUNT_NO_WALLET,
    ICON_ACCOUNT_WALLET_LOCKED,
    ICON_ACCOUNT_WALLET_UNLOCKED,
    ICON_IDENTITY_MEMBER,
    ICON_IDENTITY_MEMBER_NOT_VALIDATED,
    ICON_IDENTITY_NOT_MEMBER,
    SELECTED_UNIT_PREFERENCES_KEY,
)
from tikka.slots.pyqt.entities.qrcode_image import QRCodeImage
from tikka.slots.pyqt.entities.worker import AsyncQWorker
from tikka.slots.pyqt.models.transfers import TransferItemDelegate, TransfersListModel
from tikka.slots.pyqt.resources.gui.widgets.account_rc import Ui_AccountWidget
from tikka.slots.pyqt.widgets.account_menu import AccountPopupMenu
from tikka.slots.pyqt.widgets.account_profile_view import create_profile_html
from tikka.slots.pyqt.widgets.transfer_history_menu import TransferHistoryPopupMenu
from tikka.slots.pyqt.windows.transfer import TransferWindow


class TransfersListView(QListView):
    """
    TransfersListView class
    """

    scroll_offset_changed = pyqtSignal(int)

    def __init__(
        self,
        application: Application,
        address: str,
        page_size: int,
        parent: Optional[QWidget] = None,
    ):
        """
        Init TransfersListView instance

        :param application: Application instance
        :param address: Account address
        :param page_size: Nb of transfers per page
        :param parent: QWidget instance, default None
        """
        super().__init__(parent)

        self.application = application
        self.address = address
        self.scroll_offset = 0
        self.page_size = page_size

    def wheelEvent(self, event: QWheelEvent):
        """
        Override Wheel event

        :param event: QEvent instance
        :return:
        """
        delta = event.angleDelta().y()
        if not self.verticalScrollBar().isVisible():
            if delta < 0:  # 🔽 Scroll vers le bas (charger plus)
                if (
                    self.scroll_offset + self.page_size
                    < self.application.transfers.count(self.address)
                ):
                    self.scroll_offset += self.page_size
                    self.model().init_data(self.scroll_offset, self.page_size)  # type: ignore
                    self.scroll_offset_changed.emit(self.scroll_offset)

            elif delta > 0:  # 🔼 Scroll vers le haut (charger page précédente si dispo)
                self.scroll_offset -= self.page_size
                if self.scroll_offset < 0:
                    self.scroll_offset = 0
                self.model().init_data(self.scroll_offset, self.page_size)  # type: ignore
                self.scroll_offset_changed.emit(self.scroll_offset)
        else:
            # wheel backward and scroll bar maximum down...
            if (
                delta < 0
                and self.verticalScrollBar().value()
                == self.verticalScrollBar().maximum()
            ):

                if (
                    self.scroll_offset + self.page_size
                    < self.application.transfers.count(self.address)
                ):
                    self.scroll_offset += self.page_size
                    self.model().init_data(self.scroll_offset, self.page_size)  # type: ignore
                    self.scroll_offset_changed.emit(self.scroll_offset)

            # wheel forward and scroll bar maximum top...
            elif (
                delta > 0
                and self.verticalScrollBar().value()
                == self.verticalScrollBar().minimum()
            ):
                self.scroll_offset -= self.page_size
                if self.scroll_offset < 0:
                    self.scroll_offset = 0
                self.model().init_data(self.scroll_offset, self.page_size)  # type: ignore
                self.scroll_offset_changed.emit(self.scroll_offset)

        super().wheelEvent(event)


class AccountWidget(QWidget, Ui_AccountWidget):
    """
    AccountWidget class
    """

    def __init__(
        self,
        application: Application,
        account: Account,
        mutex: QMutex,
        parent: Optional[QWidget] = None,
    ) -> None:
        """
        Init AccountWidget instance

        :param application: Application instance
        :param account: Account instance
        :param mutex: QMutex instance
        :param parent: QWidget instance
        """
        super().__init__(parent=parent)
        self.setupUi(self)

        self.application = application
        self._ = self.application.translator.gettext

        self.account = account
        self.mutex = mutex
        self.history_limit = 100
        self.history_page_size = 100
        self.history_scroll_offset = 0
        # self.profile: Optional[Profile] = Profile("5GrwvaEF5zXb26Fz9rcQpDWS57CtERHpNehXCPcNoHGKutQY", {'title': 'Texier Vincent', 'description': 'Développeur\nqlKhflqhflqhfl!qjsfjhdjshqdljqhsflqhsflhqfhskdfjh!kqsjdhf!kjsqhde lqhflqhef lhqzef h hqzkhd lqzehf lhqzfluh fqhzfl hlqzhfl hlizqhf lihlzihdl ihalzeifhr lihalzirh \nqlhflqz \n iqhzlqhzflhqzfd \n oIZQHDLIZQHD\n lQZIHDLQZHD\n lQZIHDLQZHD\n lQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\nlQZIHDLQZHD\n\n\n\n\n\n\n\n\n\n\n\n\nlQZIHDLQZHD\nlQZIHDLQZHD', 'socials': [{'type': 'web', 'url': 'http://vit.free.fr'}], 'avatar': {'_content_type': 'image/png', '_content': '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'}, 'time': 1584271238, 'issuer': '7F6oyFQywURCACWZZGtG97Girh9EL1kg2WBwftEZxDoJ', 'hash': '2538AEB75DBFB3D7BF62421EEE655AD55048BD7092885A7A8E44AD69C98A7F71', 'signature': 'NEQgejt9d6ELP2AFt20vm2zMXuZPdIFsd+zf/m9vXD62UZ8wEXhpeRcftzsPwL65lVSWOE9fzHgtOPJZRckPDg==', 'city': 'Rambouillet, 78120', 'geoPoint': {'lon': 1.824095200031692, 'lat': 48.625434}, 'version': 2}
        self.profile: Optional[Profile] = self.application.profiles.get(
            self.account.address
        )

        self.monospace_font = QFont(ADDRESS_MONOSPACE_FONT_NAME)
        self.monospace_font.setStyleHint(QFont.Monospace)
        self.addressValueLabel.setFont(self.monospace_font)
        self.addressValueLabel.setText(self.account.address)
        if self.account.legacy_v1 is True:
            self.v1AddressValueLabel.setText(
                self.account.get_v1_address(
                    self.application.currencies.get_current().ss58_format
                )
            )

        # creating a pix map of qr code
        qr_code_pixmap = qrcode.make(
            self.account.address, image_factory=QRCodeImage
        ).pixmap()

        # set qrcode to the label
        self.QRCodeAddressLabel.setPixmap(qr_code_pixmap)

        self.transfers_model = TransfersListModel(
            application, self.account.address, self.locale()
        )

        # self.transfersListView.hide()
        # self.transfersListView = TransfersListView(
        #     application, self.account.address, self.history_page_size, self.groupBox
        # )
        #
        # self.transfersListView.setMouseTracking(False)
        # self.transfersListView.setEditTriggers(QAbstractItemView.NoEditTriggers)
        # self.transfersListView.setAlternatingRowColors(True)
        # self.transfersListView.setSelectionMode(QAbstractItemView.NoSelection)
        # self.transfersListView.setObjectName("transfersListView")
        # self.verticalLayout_4.addWidget(self.transfersListView)

        self.transfersListView.setModel(self.transfers_model)
        self.transfersListView.setItemDelegate(TransferItemDelegate())
        self.transfersListView.setResizeMode(QListView.Adjust)

        # profile webview
        self.profile_web_view = QWebEngineView(self.tabWidgetPage2)

        self.tabWidgetPage2.layout().addWidget(self.profile_web_view)
        self.profile_web_view.setSizePolicy(
            QSizePolicy.Expanding, QSizePolicy.Expanding
        )
        # Intercepter les changements d'URL pour ouvrir les liens externes
        self.profile_web_view.urlChanged.connect(self.on_profile_web_view_url_changed)

        ##############################
        # ASYNC METHODS
        ##############################
        # Create a QWorker object
        self.fetch_from_network_async_qworker = AsyncQWorker(
            self.fetch_account_from_network, self.mutex
        )
        self.fetch_from_network_async_qworker.finished.connect(
            self._on_finished_fetch_from_network
        )
        self.fetch_transfers_from_network_async_qworker = AsyncQWorker(
            self.fetch_transfers_from_network, self.mutex
        )
        self.fetch_transfers_from_network_async_qworker.finished.connect(
            self._on_finished_fetch_transfers_from_network
        )
        self.fetch_profile_from_network_async_qworker = AsyncQWorker(
            self.fetch_profile_from_network, self.mutex
        )
        self.fetch_profile_from_network_async_qworker.finished.connect(
            self._on_finished_fetch_profile_from_network
        )

        # events
        self.refreshButton.clicked.connect(self._on_refresh_button_clicked)
        self.transferToButton.clicked.connect(self.transfer)
        self.customContextMenuRequested.connect(self.on_context_menu)
        # self.transfersListView.scroll_offset_changed.connect(self._update_ui)
        self.transfersListView.customContextMenuRequested.connect(
            self.on_transfer_history_context_menu
        )
        # application events
        self.application.event_dispatcher.add_event_listener(
            AccountEvent.EVENT_TYPE_UPDATE, lambda e: self._update_ui()
        )
        self.application.event_dispatcher.add_event_listener(
            UnitEvent.EVENT_TYPE_CHANGED, lambda e: self._on_unit_changed_event()
        )
        self.application.event_dispatcher.add_event_listener(
            TransferEvent.EVENT_TYPE_SENT, lambda e: self._on_transfer_sent()
        )
        self.application.event_dispatcher.add_event_listener(
            ConnectionsEvent.EVENT_TYPE_INDEXER_CONNECTED,
            lambda e: self._on_indexer_connected(),
        )

        self.transfers_model.init_data(
            self.history_scroll_offset, self.history_page_size
        )
        self._update_ui()

    def cleanup(self):
        """
        Clean up WebEngine resources

        Fix message: Release of profile requested but WebEnginePage still not deleted. Expect troubles !

        :return:
        """
        if hasattr(self, "profile_web_view"):
            # Clear the profile web view
            self.profile_web_view.deleteLater()
            del self.profile_web_view

    def closeEvent(self, event):
        """
        Triggered when user close main window or tab

        :param event:
        :return:
        """
        self.cleanup()
        super().closeEvent(event)

    def transfer(self):
        """
        When user click on transfer to button

        :return:
        """
        TransferWindow(
            self.application, self.mutex, None, self.account, parent=self.parent()
        ).exec_()

    def _on_refresh_button_clicked(self, _: QEvent):
        """
        Triggered when user click on Refresh button

        :param _: QEvent instance
        :return:
        """
        self.errorLabel.setText("")
        self.fetch_from_network_async_qworker.start()
        self.fetch_transfers_from_network_async_qworker.start()
        self.fetch_profile_from_network_async_qworker.start()

    def _on_indexer_connected(self):
        """ "
        Triggered when the indexer is connected
        """
        self.fetch_transfers_from_network_async_qworker.start()

    def _on_transfer_sent(self):
        """
        Triggered after a successful transfer event

        :return:
        """
        # update account balance from network
        self.fetch_from_network_async_qworker.start()
        self.fetch_transfers_from_network_async_qworker.start()

    def _on_unit_changed_event(self):
        """
        Triggered when user change money units

        :return:
        """
        self._update_ui()
        self.transfers_model.init_data(
            self.history_scroll_offset, self.history_page_size
        )

    def fetch_account_from_network(self):
        """
        Get last account data from the network

        :return:
        """
        if not self.application.connections.node.is_connected():
            return

        self.refreshButton.setEnabled(False)

        try:
            self.application.accounts.network_update_account(self.account)
        except Exception as exception:
            self.errorLabel.setText(self._(str(exception)))
            logging.exception(exception)
        else:
            try:
                identity = self.application.identities.network_update_identity(
                    self.account.address
                )
            except Exception as exception:
                self.errorLabel.setText(self._(str(exception)))
                logging.exception(exception)
            else:
                if identity is not None:
                    try:
                        self.application.smiths.network_update_smith(identity.index)
                    except Exception as exception:
                        self.errorLabel.setText(self._(str(exception)))
                        logging.exception(exception)

    def _on_finished_fetch_from_network(self):
        """
        Triggered when async request fetch_account_from_network is finished

        :return:
        """
        self.account = self.application.accounts.get_by_address(self.account.address)
        self.profile = self.application.profiles.get(self.account.address)
        self._update_ui()
        self.refreshButton.setEnabled(True)
        logging.debug("Account widget update")

    def fetch_transfers_from_network(self):
        """
        Get transfers data from the network

        :return:
        """
        if not self.application.connections.indexer.is_connected():
            return

        self.refreshButton.setEnabled(False)

        try:
            self.application.transfers.network_fetch_history_for_account(
                self.account, self.history_limit
            )
        except Exception as exception:
            self.errorLabel.setText(self._(str(exception)))
            logging.exception(exception)

        try:
            self.account.total_transfers_count = (
                self.application.transfers.network_fetch_total_count_for_address(
                    self.account.address
                )
            )
        except Exception as exception:
            self.errorLabel.setText(self._(str(exception)))
            logging.exception(exception)

    def _on_finished_fetch_transfers_from_network(self):
        """
        Triggered when async request fetch_transfers_from_network is finished

        :return:
        """
        self.application.accounts.update(self.account)

        self.account = self.application.accounts.get_by_address(self.account.address)
        self.transfers_model.init_data(
            self.history_scroll_offset, self.history_page_size
        )
        self._update_ui()
        self.refreshButton.setEnabled(True)
        logging.debug(
            "Finished fetch account transfers %s %s",
            self.account.name,
            self.account.address,
        )

    def fetch_profile_from_network(self):
        """
        Get last account data from the network

        :return:
        """
        if not self.application.connections.datapod.is_connected():
            return

        self.refreshButton.setEnabled(False)
        try:
            self.profile = self.application.profiles.network_update(self.account)
        except Exception as exception:
            self.errorLabel.setText(self._(str(exception)))
            logging.exception(exception)

    def _on_finished_fetch_profile_from_network(self):
        """
        Triggered when async request fetch_account_from_network is finished

        :return:
        """
        self._update_ui()
        self.refreshButton.setEnabled(True)
        logging.debug("Account profile widget update")

    def _update_ui(self):
        """
        Update UI from self.account

        :return:
        """
        display_identity_status = {
            IdentityStatus.UNCONFIRMED.value: self._("Unconfirmed"),
            IdentityStatus.UNVALIDATED.value: self._("Unvalidated"),
            IdentityStatus.MEMBER.value: self._("Member"),
            IdentityStatus.NOT_MEMBER.value: self._("Not member"),
            IdentityStatus.REVOKED.value: self._("Revoked"),
        }

        display_identity_icon = {
            IdentityStatus.UNCONFIRMED.value: QPixmap(
                ICON_IDENTITY_MEMBER_NOT_VALIDATED
            ),
            IdentityStatus.UNVALIDATED.value: QPixmap(
                ICON_IDENTITY_MEMBER_NOT_VALIDATED
            ),
            IdentityStatus.MEMBER.value: QPixmap(ICON_IDENTITY_MEMBER),
            IdentityStatus.NOT_MEMBER.value: QPixmap(ICON_IDENTITY_NOT_MEMBER),
            IdentityStatus.REVOKED.value: QPixmap(ICON_IDENTITY_NOT_MEMBER),
        }

        display_smith_status = {
            SmithStatus.INVITED.value: self._("Invited"),
            SmithStatus.PENDING.value: self._("Pending"),
            SmithStatus.SMITH.value: self._("Smith"),
            SmithStatus.EXCLUDED.value: self._("Excluded"),
        }

        unit_preference = self.application.repository.preferences.get(
            SELECTED_UNIT_PREFERENCES_KEY
        )
        if unit_preference is not None:
            amount = self.application.amounts.get_amount(unit_preference)
        else:
            amount = self.application.amounts.get_amount(AMOUNT_UNIT_KEY)

        if self.account.balance is None:
            self.balanceValueLabel.setText("?")
            self.balanceReservedValueLabel.setText("")
        else:
            self.balanceValueLabel.setText(
                self.locale().toCurrencyString(
                    amount.value(self.account.balance), amount.symbol()
                )
            )
            display_reserved_balance = self.locale().toCurrencyString(
                amount.value(self.account.balance_reserved), amount.symbol()
            )
            self.balanceReservedValueLabel.setText(f"[-{display_reserved_balance}]")

        if self.application.wallets.exists(self.account.address):
            if self.application.wallets.is_unlocked(self.account.address):
                self.lockStatusIcon.setPixmap(QPixmap(ICON_ACCOUNT_WALLET_UNLOCKED))
            else:
                self.lockStatusIcon.setPixmap(QPixmap(ICON_ACCOUNT_WALLET_LOCKED))
        else:
            self.lockStatusIcon.setPixmap(QPixmap(ICON_ACCOUNT_NO_WALLET))

        identity = self.application.identities.get_by_address(self.account.address)
        if identity is not None:
            self.identityStatusIconLabel.setPixmap(
                display_identity_icon[identity.status.value]
            )
            identity_name = identity.name or ""
            self.identityNameValueLabel.setText(f"{identity_name}#{identity.index}")
            self.identityValueLabel.setText(
                display_identity_status[identity.status.value]
            )
            smith = self.application.smiths.get(identity.index)
            if smith is not None:
                self.smithValuelabel.setText(display_smith_status[smith.status.value])
            else:
                self.smithValuelabel.setText(self._("No"))
            if self.account.name is None:
                self.accountNameValueLabel.setText("")
            else:
                self.accountNameValueLabel.setText(f" - {self.account.name}")
        else:
            self.identityStatusIconLabel.setPixmap(QPixmap(ICON_IDENTITY_NOT_MEMBER))
            self.identityNameValueLabel.setText("")
            self.identityValueLabel.setText(self._("No"))
            self.smithValuelabel.setText(self._("No"))
            if self.account.name is None:
                self.accountNameValueLabel.setText("")
            else:
                self.accountNameValueLabel.setText(self.account.name)

        if self.account.root is None and self.account.path is None:
            self.derivationValueLabel.setText(self._("Root"))
        else:
            root_account = self.application.accounts.get_by_address(self.account.root)
            if root_account is not None and root_account.name is not None:
                self.derivationValueLabel.setFont(QFont())
                self.derivationValueLabel.setText(root_account.name + self.account.path)
            else:
                self.derivationValueLabel.setFont(self.monospace_font)
                self.derivationValueLabel.setText(
                    DisplayAddress(self.account.root).shorten + self.account.path
                )

        # transfers
        self.transfersCountValueLabel.setText(
            f"1...{self.history_limit}/{self.account.total_transfers_count}"
        )
        # current_page = int(
        #     (self.transfersListView.scroll_offset / self.history_page_size) + 1
        # )
        # self.transfersCountValueLabel.setText(
        #     str(
        #         f"Page {current_page}/{self.history_page_size} |"
        #         f"1...{self.history_limit}/{self.account.total_transfers_count}"
        #     )
        # )

        # profile
        self.profile_web_view.setHtml(self.get_profile_as_html())

    def get_profile_as_html(self) -> str:
        """
        Display profile data as HTML in web view

        :return:
        """
        if self.profile is not None:
            updated_on_time_stamp = self.profile.data.get("time", 0)
            updated_on_localized_datetime_string = self.locale().toString(
                datetime.fromtimestamp(updated_on_time_stamp),
                QLocale.dateTimeFormat(self.locale(), QLocale.ShortFormat),
            )
            updated_on_localized_string = self._(
                "Updated at {localized_string}"
            ).format(localized_string=updated_on_localized_datetime_string)

            return create_profile_html(self.profile.data, updated_on_localized_string)

        return self._("No profile.")

    def on_profile_web_view_url_changed(self, url):
        """
        Triggered when user click on web link in profile web view

        :return:
        """
        if url.isValid() and url.scheme() in ["http", "https"]:
            QDesktopServices.openUrl(url)
            # Reload page to stay on original content
            self.profile_web_view.sender().setHtml(self.get_profile_as_html())

    def on_context_menu(self, position: QPoint):
        """
        When right button on account tab

        :return:
        """
        # display popup menu at click position
        menu = AccountPopupMenu(self.application, self.account, self.mutex, self)
        menu.exec_(self.mapToGlobal(position))

    def on_transfer_history_context_menu(self, position: QPoint):
        """
        When right button on transfer history listview

        :return:
        """
        # get selected transfer
        transfer = self.transfersListView.currentIndex().internalPointer()
        if transfer is not None:
            # display popup menu at click position
            menu = TransferHistoryPopupMenu(
                self.application, self.account, transfer, self.mutex, self
            )
            menu.exec_(self.transfersListView.mapToGlobal(position))


if __name__ == "__main__":
    qapp = QApplication(sys.argv)

    application_ = Application(DATA_PATH)
    account_ = Account("5GrwvaEF5zXb26Fz9rcQpDWS57CtERHpNehXCPcNoHGKutQY")
    account_.balance = 1000
    account_.balance_available = 900
    account_.balance_reserved = 100

    main_window = QMainWindow()
    main_window.show()

    main_window.setCentralWidget(
        AccountWidget(application_, account_, QMutex(), main_window)
    )

    sys.exit(qapp.exec_())
