#***************************************************************************************************
# Copyright 2015, 2019, 2025 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
# Under the terms of Contract DE-NA0003525 with NTESS, the U.S. Government retains certain rights
# in this software.
# Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
# in compliance with the License.  You may obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0 or in the LICENSE file in the root pyGSTi directory.
#***************************************************************************************************
"""
Variables for working with the a qutrit model containing Idle, X(pi/2) and Y(pi/2) and Molmer-Sorenson gates.
"""
import sys as _sys

from numpy import pi as _pi

from ...circuits import circuitconstruction as _strc
from ...models import qutrit as _qutrit
from .. import stdtarget as _stdtarget

description = "Idle, symmetric X(pi/2), symmetric Y(pi/2), and Molmer-Sorenson gates"

gates = ['Gi', 'Gx', 'Gy', 'Gm']
prepStrs = _strc.to_circuits([
    (), ('Gx',), ('Gy',), ('Gm',),
    ('Gx', 'Gx'), ('Gm', 'Gx'), ('Gm', 'Gy'),
    ('Gy', 'Gy', 'Gy'), ('Gx', 'Gx', 'Gx')], line_labels=('*',))

effectStrs = _strc.to_circuits([
    (), ('Gx',), ('Gy',), ('Gm',),
    ('Gx', 'Gx'), ('Gy', 'Gm'), ('Gx', 'Gm')], line_labels=('*',))

germs = _strc.to_circuits([
    ('Gi',),
    ('Gy',),
    ('Gx',),
    ('Gm',),
    ('Gi', 'Gy'),
    ('Gi', 'Gx'),
    ('Gi', 'Gm'),
    ('Gy', 'Gm'),
    ('Gx', 'Gm'),
    ('Gx', 'Gy'),
    ('Gi', 'Gi', 'Gy'),
    ('Gi', 'Gi', 'Gx'),
    ('Gx', 'Gx', 'Gy'),
    ('Gi', 'Gi', 'Gm'),
    ('Gi', 'Gy', 'Gy'),
    ('Gi', 'Gy', 'Gx'),
    ('Gi', 'Gy', 'Gm'),
    ('Gi', 'Gx', 'Gy'),
    ('Gi', 'Gx', 'Gx'),
    ('Gi', 'Gx', 'Gm'),
    ('Gi', 'Gm', 'Gy'),
    ('Gi', 'Gm', 'Gx'),
    ('Gi', 'Gm', 'Gm'),
    ('Gy', 'Gy', 'Gx'),
    ('Gy', 'Gy', 'Gm'),
    ('Gy', 'Gx', 'Gx'),
    ('Gy', 'Gx', 'Gm'),
    ('Gy', 'Gm', 'Gx'),
    ('Gy', 'Gm', 'Gm'),
    ('Gx', 'Gx', 'Gm'),
    ('Gx', 'Gm', 'Gm'),
    ('Gx', 'Gy', 'Gy'),
    ('Gx', 'Gy', 'Gi'),
    ('Gx', 'Gi', 'Gy'),
    ('Gx', 'Gi', 'Gi'),
    ('Gy', 'Gi', 'Gi'),
    ('Gx', 'Gm', 'Gy'),
    ('Gx', 'Gy', 'Gy', 'Gi'),
    ('Gm', 'Gm', 'Gx', 'Gm'),
    ('Gx', 'Gx', 'Gy', 'Gx'),
    ('Gm', 'Gy', 'Gm', 'Gx'),
    ('Gx', 'Gi', 'Gy', 'Gy'),
    ('Gy', 'Gy', 'Gm', 'Gm'),
    ('Gi', 'Gx', 'Gy', 'Gm'),
    ('Gx', 'Gi', 'Gi', 'Gm'),
    ('Gx', 'Gx', 'Gy', 'Gy'),
    ('Gx', 'Gm', 'Gy', 'Gx'),
    ('Gm', 'Gy', 'Gm', 'Gi'),
    ('Gx', 'Gi', 'Gm', 'Gi', 'Gy'),
    ('Gy', 'Gm', 'Gx', 'Gy', 'Gy'),
    ('Gx', 'Gy', 'Gm', 'Gy', 'Gy'),
    ('Gm', 'Gx', 'Gi', 'Gx', 'Gx'),
    ('Gx', 'Gi', 'Gy', 'Gi', 'Gy'),
    ('Gx', 'Gx', 'Gy', 'Gm', 'Gy'),
    ('Gm', 'Gy', 'Gm', 'Gm', 'Gy'),
    ('Gy', 'Gi', 'Gi', 'Gy', 'Gx'),
    ('Gy', 'Gy', 'Gx', 'Gx', 'Gy'),
    ('Gx', 'Gy', 'Gm', 'Gx', 'Gi'),
    ('Gm', 'Gi', 'Gm', 'Gy', 'Gi'),
    ('Gy', 'Gm', 'Gx', 'Gx', 'Gx'),
    ('Gy', 'Gi', 'Gm', 'Gx', 'Gi'),
    ('Gy', 'Gy', 'Gx', 'Gm', 'Gx'),
    ('Gx', 'Gx', 'Gy', 'Gx', 'Gy', 'Gy'),
    ('Gi', 'Gy', 'Gx', 'Gx', 'Gy', 'Gy'),
    ('Gy', 'Gx', 'Gx', 'Gy', 'Gx', 'Gm'),
    ('Gx', 'Gi', 'Gi', 'Gy', 'Gy', 'Gy'),
    ('Gm', 'Gy', 'Gi', 'Gx', 'Gy', 'Gi'),
    ('Gx', 'Gx', 'Gy', 'Gx', 'Gi', 'Gx'),
    ('Gm', 'Gy', 'Gx', 'Gm', 'Gm', 'Gy')], line_labels=('*',))


germs_lite = _strc.to_circuits([
    ('Gi',),
    ('Gy',),
    ('Gx',),
    ('Gm',),
    ('Gi', 'Gy'),
    ('Gi', 'Gx'),
    ('Gi', 'Gm'),
    ('Gx', 'Gy'),
    ('Gy', 'Gm'),
    ('Gx', 'Gm'),
    ('Gi', 'Gi', 'Gy'),
    ('Gi', 'Gi', 'Gx'),
    ('Gx', 'Gx', 'Gy'),
    ('Gi', 'Gi', 'Gm'),
    ('Gi', 'Gy', 'Gy'),
    ('Gi', 'Gy', 'Gx'),
    ('Gi', 'Gy', 'Gm'),
    ('Gi', 'Gx', 'Gy'),
    ('Gi', 'Gx', 'Gx'),
    ('Gi', 'Gx', 'Gm'),
    ('Gi', 'Gm', 'Gy'),
    ('Gi', 'Gm', 'Gx'),
    ('Gi', 'Gm', 'Gm'),
    ('Gy', 'Gy', 'Gx'),
    ('Gy', 'Gy', 'Gm'),
    ('Gy', 'Gx', 'Gx'),
    ('Gy', 'Gx', 'Gm'),
    ('Gy', 'Gm', 'Gx'),
    ('Gy', 'Gm', 'Gm'),
    ('Gx', 'Gx', 'Gm'),
    ('Gx', 'Gm', 'Gm')], line_labels=('*',))


legacy_germs = _strc.to_circuits([
    ('Gi',),
    ('Gx',),
    ('Gy',),
    ('Gm',),
    ('Gx', 'Gy'),
    ('Gx', 'Gm'),
    ('Gx', 'Gy', 'Gi'),
    ('Gx', 'Gi', 'Gy'),
    ('Gx', 'Gi', 'Gi'),
    ('Gy', 'Gi', 'Gi'),
    ('Gx', 'Gy', 'Gm'),
    ('Gx', 'Gm', 'Gy'),
    ('Gi', 'Gx', 'Gm'),
    ('Gy', 'Gm', 'Gm'),
    ('Gx', 'Gy', 'Gy'),
    ('Gi', 'Gm', 'Gx'),
    ('Gx', 'Gx', 'Gi', 'Gy'),
    ('Gx', 'Gy', 'Gy', 'Gi'),
    ('Gy', 'Gm', 'Gm', 'Gm'),
    ('Gy', 'Gy', 'Gm', 'Gm'),
    ('Gx', 'Gm', 'Gy', 'Gx'),
    ('Gx', 'Gm', 'Gm', 'Gm'),
    ('Gx', 'Gi', 'Gy', 'Gy'),
    ('Gy', 'Gx', 'Gy', 'Gi'),
    ('Gx', 'Gy', 'Gm', 'Gy'),
    ('Gm', 'Gm', 'Gi', 'Gi'),
    ('Gy', 'Gx', 'Gy', 'Gm'),
    ('Gi', 'Gx', 'Gm', 'Gx'),
    ('Gx', 'Gx', 'Gy', 'Gx'),
    ('Gx', 'Gi', 'Gm', 'Gi'),
    ('Gm', 'Gy', 'Gm', 'Gx'),
    ('Gx', 'Gx', 'Gy', 'Gy'),
    ('Gm', 'Gy', 'Gm', 'Gi'),
    ('Gi', 'Gx', 'Gy', 'Gm'),
    ('Gm', 'Gi', 'Gx', 'Gi'),
    ('Gy', 'Gy', 'Gy', 'Gy'),
    ('Gi', 'Gy', 'Gy', 'Gm'),
    ('Gy', 'Gy', 'Gx', 'Gx', 'Gy'),
    ('Gm', 'Gi', 'Gm', 'Gy', 'Gi'),
    ('Gy', 'Gi', 'Gi', 'Gy', 'Gx'),
    ('Gx', 'Gy', 'Gm', 'Gy', 'Gy'),
    ('Gx', 'Gi', 'Gm', 'Gi', 'Gy'),
    ('Gy', 'Gm', 'Gx', 'Gy', 'Gy'),
    ('Gx', 'Gy', 'Gy', 'Gy', 'Gy'),
    ('Gm', 'Gy', 'Gm', 'Gm', 'Gy'),
    ('Gx', 'Gy', 'Gm', 'Gx', 'Gi'),
    ('Gx', 'Gx', 'Gy', 'Gm', 'Gy'),
    ('Gm', 'Gx', 'Gi', 'Gx', 'Gx'),
    ('Gy', 'Gi', 'Gm', 'Gx', 'Gi'),
    ('Gy', 'Gy', 'Gx', 'Gm', 'Gx'),
    ('Gm', 'Gx', 'Gi', 'Gy', 'Gx'),
    ('Gx', 'Gx', 'Gy', 'Gx', 'Gy', 'Gy'),
    ('Gx', 'Gi', 'Gi', 'Gy', 'Gy', 'Gy'),
    ('Gm', 'Gm', 'Gi', 'Gi', 'Gy', 'Gi'),
    ('Gy', 'Gx', 'Gx', 'Gy', 'Gx', 'Gm'),
    ('Gi', 'Gm', 'Gx', 'Gy', 'Gm', 'Gy'),
    ('Gm', 'Gy', 'Gi', 'Gx', 'Gy', 'Gi'),
    ('Gi', 'Gx', 'Gx', 'Gi', 'Gy', 'Gy'),
    ('Gy', 'Gi', 'Gx', 'Gx', 'Gy', 'Gm'),
    ('Gm', 'Gx', 'Gy', 'Gx', 'Gx', 'Gx'),
    ('Gi', 'Gy', 'Gx', 'Gx', 'Gy', 'Gy'),
    ('Gm', 'Gy', 'Gx', 'Gm', 'Gm', 'Gy')], line_labels=('*',))

#Missing GxGxGy (compared to non-legacy), which is needed to contain
# all of the std1Q_XYI `germs_lite` germs.
legacy_germs_lite = _strc.to_circuits([
    ('Gi',),
    ('Gy',),
    ('Gx',),
    ('Gm',),
    ('Gi', 'Gy'),
    ('Gi', 'Gx'),
    ('Gi', 'Gm'),
    ('Gy', 'Gx'),
    ('Gy', 'Gm'),
    ('Gx', 'Gm'),
    ('Gi', 'Gi', 'Gy'),
    ('Gi', 'Gi', 'Gx'),
    ('Gi', 'Gi', 'Gm'),
    ('Gi', 'Gy', 'Gy'),
    ('Gi', 'Gy', 'Gx'),
    ('Gi', 'Gy', 'Gm'),
    ('Gi', 'Gx', 'Gy'),
    ('Gi', 'Gx', 'Gx'),
    ('Gi', 'Gx', 'Gm'),
    ('Gi', 'Gm', 'Gy'),
    ('Gi', 'Gm', 'Gx'),
    ('Gi', 'Gm', 'Gm'),
    ('Gy', 'Gy', 'Gx'),
    ('Gy', 'Gy', 'Gm'),
    ('Gy', 'Gx', 'Gx'),
    ('Gy', 'Gx', 'Gm'),
    ('Gy', 'Gm', 'Gx'),
    ('Gy', 'Gm', 'Gm'),
    ('Gx', 'Gx', 'Gm'),
    ('Gx', 'Gm', 'Gm')], line_labels=('*',))


#Construct a target model: Identity, sym X(pi/2), sym Y(pi/2), Molmer-Sorenson
_target_model = _qutrit.create_qutrit_model(error_scale=0, x_angle=_pi / 2, y_angle=_pi / 2,
                                          ms_global=_pi / 2, ms_local=0, basis="qt")

_gscache = {("full", "auto"): _target_model}


def processor_spec():
    return target_model('static').create_processor_spec(None)


def target_model(parameterization_type="full", sim_type="auto"):
    """
    Returns a copy of the target model in the given parameterization.

    Parameters
    ----------
    parameterization_type : {"TP", "CPTP", "H+S", "S", ... }
        The gate and SPAM vector parameterization type. See
        :func:`Model.set_all_parameterizations` for all allowed values.

    sim_type : {"auto", "matrix", "map", "termorder:X" }
        The simulator type to be used for model calculations (leave as
        "auto" if you're not sure what this is).

    Returns
    -------
    Model
    """
    return _stdtarget._copy_target(_sys.modules[__name__], parameterization_type,
                                   sim_type, _gscache)


legacy_gs_target = _qutrit.create_qutrit_model(error_scale=0, x_angle=-_pi / 2, y_angle=_pi / 2,
                                             ms_global=-_pi / 2, ms_local=0, basis="qt")
#Note: negative signs from weird/incorrect conventions

global_fidPairs = [
    (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6), (8, 4)]

pergerm_fidPairsDict = {
    ('Gx',): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gm',): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
    ('Gi',): [
        (0, 6), (1, 0), (1, 1), (1, 3), (1, 4), (3, 3), (3, 4),
        (3, 5), (3, 6), (4, 0), (5, 6), (6, 3), (6, 5), (7, 2),
        (7, 3), (7, 6), (8, 5)],
    ('Gy',): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gi', 'Gy'): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx'): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gm'): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
    ('Gy', 'Gm'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gi', 'Gy', 'Gm'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gi', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gy', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gm', 'Gy'): [
        (0, 0), (0, 1), (0, 4), (1, 5), (3, 6), (4, 5), (5, 3),
        (5, 6), (6, 0), (7, 5), (7, 6), (8, 4)],
    ('Gi', 'Gy', 'Gy'): [
        (3, 4), (4, 0), (4, 1), (6, 1), (6, 5), (6, 6), (7, 2),
        (8, 6)],
    ('Gy', 'Gm', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gi', 'Gx'): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gi', 'Gi', 'Gy'): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gx', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gi', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gx', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gy', 'Gm'): [
        (0, 4), (2, 2), (2, 3), (3, 5), (4, 1), (4, 3), (4, 5),
        (5, 4), (5, 5), (6, 2), (6, 5), (7, 2), (7, 5), (7, 6),
        (8, 0)],
    ('Gi', 'Gm', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gi', 'Gi'): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gi', 'Gm', 'Gy'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gx', 'Gx', 'Gm'): [
        (0, 1), (1, 1), (2, 0), (2, 6), (5, 0), (5, 1), (6, 5),
        (6, 6)],
    ('Gi', 'Gm', 'Gm'): [
        (0, 5), (2, 2), (2, 5), (3, 6), (4, 4), (5, 0), (5, 2),
        (8, 6)],
    ('Gi', 'Gi', 'Gm'): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
    ('Gy', 'Gy', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gi', 'Gi'): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gy', 'Gm', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx', 'Gx'): [
        (3, 2), (3, 3), (4, 3), (5, 3), (7, 2), (7, 4), (8, 1),
        (8, 4)],
    ('Gx', 'Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gm', 'Gm'): [
        (0, 6), (1, 5), (3, 2), (3, 3), (3, 4), (4, 6), (5, 3),
        (6, 5), (6, 6), (7, 1)],
    ('Gx', 'Gy', 'Gi'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gy', 'Gm', 'Gm'): [
        (0, 2), (1, 1), (1, 2), (2, 2), (2, 4), (6, 3), (6, 6),
        (8, 3)],
    ('Gx', 'Gx', 'Gy', 'Gy'): [
        (0, 1), (1, 1), (2, 0), (2, 6), (5, 0), (5, 1), (6, 5),
        (6, 6)],
    ('Gx', 'Gi', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx', 'Gy', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gy', 'Gm', 'Gi'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gm', 'Gy', 'Gx'): [
        (0, 2), (2, 6), (3, 0), (3, 2), (3, 5), (3, 6), (6, 2),
        (6, 5), (6, 6), (8, 0)],
    ('Gx', 'Gy', 'Gy', 'Gi'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gm', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gy', 'Gm', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gx', 'Gy', 'Gx'): [
        (0, 0), (0, 1), (0, 4), (1, 5), (3, 6), (4, 5), (5, 3),
        (5, 6), (6, 0), (7, 5), (7, 6), (8, 4)],
    ('Gx', 'Gi', 'Gi', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gm', 'Gx', 'Gx', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gi', 'Gm', 'Gx', 'Gi'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gy', 'Gx', 'Gm', 'Gx'): [
        (0, 1), (1, 0), (2, 3), (3, 3), (3, 6), (4, 1), (5, 5),
        (6, 2), (6, 5), (7, 1), (7, 2), (7, 5), (8, 6)],
    ('Gx', 'Gx', 'Gy', 'Gm', 'Gy'): [
        (1, 1), (1, 5), (1, 6), (3, 3), (4, 5), (5, 0), (5, 5),
        (8, 5)],
    ('Gx', 'Gi', 'Gy', 'Gi', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gx', 'Gi', 'Gx', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gi', 'Gm', 'Gy', 'Gi'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gm', 'Gx', 'Gy', 'Gy'): [
        (0, 0), (0, 1), (0, 4), (1, 5), (3, 6), (4, 5), (5, 3),
        (5, 6), (6, 0), (7, 5), (7, 6), (8, 4)],
    ('Gx', 'Gi', 'Gm', 'Gi', 'Gy'): [
        (0, 0), (0, 1), (0, 4), (1, 5), (3, 6), (4, 5), (5, 3),
        (5, 6), (6, 0), (7, 5), (7, 6), (8, 4)],
    ('Gx', 'Gy', 'Gm', 'Gx', 'Gi'): [
        (0, 2), (2, 6), (3, 0), (3, 2), (3, 5), (3, 6), (6, 2),
        (6, 5), (6, 6), (8, 0)],
    ('Gy', 'Gi', 'Gi', 'Gy', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gy', 'Gm', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gy', 'Gx', 'Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gm', 'Gy', 'Gm', 'Gm', 'Gy'): [
        (0, 1), (1, 1), (2, 0), (2, 6), (5, 0), (5, 1), (6, 5),
        (6, 6)],
    ('Gm', 'Gy', 'Gi', 'Gx', 'Gy', 'Gi'): [
        (0, 2), (2, 6), (3, 0), (3, 2), (3, 5), (3, 6), (6, 2),
        (6, 5), (6, 6), (8, 0)],
    ('Gy', 'Gx', 'Gx', 'Gy', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gy', 'Gx', 'Gx', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gx', 'Gy', 'Gx', 'Gi', 'Gx'): [
        (0, 0), (0, 2), (0, 4), (0, 5), (2, 0), (3, 2), (3, 3),
        (3, 6), (5, 4), (5, 5), (5, 6), (6, 2), (6, 3), (6, 4),
        (7, 4), (7, 5), (8, 1), (8, 3)],
    ('Gm', 'Gy', 'Gx', 'Gm', 'Gm', 'Gy'): [
        (0, 2), (2, 6), (3, 0), (3, 2), (3, 5), (3, 6), (6, 2),
        (6, 5), (6, 6), (8, 0)],
    ('Gx', 'Gx', 'Gy', 'Gx', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gi', 'Gi', 'Gy', 'Gy', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
}


global_fidPairs_lite = [
    (0, 1), (3, 0), (4, 0), (4, 5), (4, 6), (5, 2), (5, 3), (5, 5),
    (6, 6), (8, 0), (8, 1), (8, 6)]

pergerm_fidPairsDict_lite = {
    ('Gx',): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gm',): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
    ('Gi',): [
        (0, 6), (1, 0), (1, 1), (1, 3), (1, 4), (3, 3), (3, 4),
        (3, 5), (3, 6), (4, 0), (5, 6), (6, 3), (6, 5), (7, 2),
        (7, 3), (7, 6), (8, 5)],
    ('Gy',): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gi', 'Gy'): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx'): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gm'): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
    ('Gy', 'Gm'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gi', 'Gy', 'Gm'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gi', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gy', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gy', 'Gy'): [
        (3, 4), (4, 0), (4, 1), (6, 1), (6, 5), (6, 6), (7, 2),
        (8, 6)],
    ('Gy', 'Gm', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gi', 'Gx'): [
        (0, 4), (1, 4), (2, 1), (2, 4), (2, 5), (3, 1), (3, 3),
        (3, 5), (5, 3), (6, 0), (6, 5), (8, 2), (8, 4)],
    ('Gi', 'Gi', 'Gy'): [
        (0, 6), (1, 0), (3, 2), (3, 6), (4, 2), (5, 5), (5, 6),
        (6, 2), (7, 0), (7, 2), (8, 1)],
    ('Gy', 'Gx', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gy', 'Gm'): [
        (0, 4), (2, 2), (2, 3), (3, 5), (4, 1), (4, 3), (4, 5),
        (5, 4), (5, 5), (6, 2), (6, 5), (7, 2), (7, 5), (7, 6),
        (8, 0)],
    ('Gi', 'Gm', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gy', 'Gx', 'Gm'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gm', 'Gm'): [
        (0, 6), (1, 5), (3, 2), (3, 3), (3, 4), (4, 6), (5, 3),
        (6, 5), (6, 6), (7, 1)],
    ('Gi', 'Gm', 'Gy'): [
        (0, 3), (1, 1), (1, 2), (3, 2), (4, 1), (4, 4), (5, 0),
        (6, 0), (7, 2), (7, 3)],
    ('Gx', 'Gx', 'Gm'): [
        (0, 1), (1, 1), (2, 0), (2, 6), (5, 0), (5, 1), (6, 5),
        (6, 6)],
    ('Gi', 'Gm', 'Gm'): [
        (0, 5), (2, 2), (2, 5), (3, 6), (4, 4), (5, 0), (5, 2),
        (8, 6)],
    ('Gy', 'Gy', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gx', 'Gx'): [
        (3, 2), (3, 3), (4, 3), (5, 3), (7, 2), (7, 4), (8, 1),
        (8, 4)],
    ('Gy', 'Gm', 'Gx'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gx', 'Gx', 'Gy'): [
        (0, 0), (0, 4), (3, 6), (4, 5), (5, 3), (7, 5), (7, 6),
        (8, 4)],
    ('Gi', 'Gi', 'Gm'): [
        (1, 4), (2, 4), (4, 0), (4, 2), (4, 3), (5, 1), (6, 2),
        (6, 6)],
}
