import json
from flask import Request, Response, request, jsonify
from logging import Logger
from typing import Any

from .iam_common import IamServer, _iam_lock, _get_iam_server
from .iam_pomes import (
    user_login, user_logout,
    user_token, token_exchange, login_callback
)

# the logger for IAM service operations
# (used exclusively at the HTTP endpoints - all other functions receive the logger as parameter)
__IAM_LOGGER: Logger | None = None


def logger_register(logger: Logger) -> None:
    """
    Register the logger for HTTP services.

    :param logger: the logger to be registered
    """
    global __IAM_LOGGER
    __IAM_LOGGER = logger


# @flask_app.route(rule=<login_endpoint>,  # JUSBR_ENDPOINT_LOGIN
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,  # KEYCLOAK_ENDPOINT_LOGIN
#                  methods=["GET"])
def service_login() -> Response:
    """
    Entry point for the IAM server's login service.

    These are the expected request parameters:
        - user-id: optional, identifies the reference user (aliases: 'user_id', 'login')
        - redirect-uri: a parameter to be added to the query part of the returned URL

    If provided, the user identification will be validated against the authorization data
    returned by *iam_server* upon login. On success, the following JSON, containing the appropriate
    URL for invoking the IAM server's authentication page, is returned:
        {
            "login-url": <login-url>
        }

    :return: *Response* with the URL for invoking the IAM server's authentication page, or *BAD REQUEST* if error
    """
    # declare the return variable
    result: Response | None = None

    # log the request
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=__IAM_LOGGER)
        if iam_server:
            # obtain the login URL
            login_url: str = user_login(iam_server=iam_server,
                                        args=request.args,
                                        errors=errors,
                                        logger=__IAM_LOGGER)
            if login_url:
                result = jsonify({"login-url": login_url})
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400

    # log the response
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=f"Response {result}, {result.get_data(as_text=True)}")

    return result


# @flask_app.route(rule=<logout_endpoint>,  # JUSBR_ENDPOINT_LOGOUT
#                  methods=["GET"])
# @flask_app.route(rule=<login_endpoint>,   # KEYCLOAK_ENDPOINT_LOGOUT
#                  methods=["GET"])
def service_logout() -> Response:
    """
    Entry point for the JusBR logout service.

    The user is identified by the attribute *user-id*, *user_id*, or "login", provided as a request parameter.
    If successful, remove all data relating to the user from the *IAM* server's registry.
    Otherwise, this operation fails silently, unless an error has ocurred.

    :return: *Response NO CONTENT*, or *BAD REQUEST* if error
    """
    # declare the return variable
    result: Response | None

    # log the request
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=__IAM_LOGGER)
        if iam_server:
            # logout the user
            user_logout(iam_server=iam_server,
                        args=request.args,
                        errors=errors,
                        logger=__IAM_LOGGER)
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = Response(status=204)

    # log the response
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=f"Response {result}")

    return result


# @flask_app.route(rule=<callback_endpoint>,  # JUSBR_ENDPOINT_CALLBACK
#                  methods=["GET", "POST"])
# @flask_app.route(rule=<callback_endpoint>,  # KEYCLOAK_ENDPOINT_CALLBACK
#                  methods=["POST"])
def service_callback() -> Response:
    """
    Entry point for the callback from JusBR on authentication operation.

    This callback is invoked from a front-end application after a successful login at the
    *IAM* server's login page, forwarding the data received. In a typical OAuth2 flow faction,
    this data is then used to effectively obtain the token from the *IAM* server.

    The relevant expected request arguments are:
        - *state*: used to enhance security during the authorization process, typically to provide *CSRF* protection
        - *code*: the temporary authorization code provided by the IAM server, to be exchanged for the token

    On success, the returned *Response* will contain the following JSON:
        {
            "user-id": <reference-user-identification>,
            "token": <token>
        }

    :return: *Response* containing the reference user identification and the token, or *BAD REQUEST*
    """
    # log the request
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=_log_init(request=request))

    errors: list[str] = []
    token_data: tuple[str, str] | None = None
    with _iam_lock:
        # retrieve the IAM server
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=__IAM_LOGGER)
        if iam_server:
            # process the callback operation
            token_data = login_callback(iam_server=iam_server,
                                        args=request.args,
                                        errors=errors,
                                        logger=__IAM_LOGGER)
    result: Response
    if errors:
        result = jsonify({"errors": "; ".join(errors)})
        result.status_code = 400
    else:
        result = jsonify({"user-id": token_data[0],
                          "token": token_data[1]})
    # log the response
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=f"Response {result}, {result.get_data(as_text=True)}")

    return result


# @flask_app.route(rule=<token_endpoint>,  # JUSBR_ENDPOINT_TOKEN
#                  methods=["GET"])
# @flask_app.route(rule=<token_endpoint>,  # KEYCLOAK_ENDPOINT_TOKEN
#                  methods=["GET"])
def service_token() -> Response:
    """
    Entry point for retrieving a token from the *IAM* server.

    The user is identified by the attribute *user-id*, *user_id*, or "login", provided as a request parameter.

    On success, the returned *Response* will contain the following JSON:
        {
            "user-id": <reference-user-identification>,
            "token": <token>
        }

    :return: *Response* containing the user reference identification and the token, or *BAD REQUEST*
    """
    # log the request
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=_log_init(request=request))

    # obtain the user's identification
    args: dict[str, Any] = request.args
    user_id: str = args.get("user-id") or args.get("user_id") or args.get("login")

    errors: list[str] = []
    token: str | None = None
    if user_id:
        with _iam_lock:
            # retrieve the IAM server
            iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                    errors=errors,
                                                    logger=__IAM_LOGGER)
            if iam_server:
                # retrieve the token
                errors: list[str] = []
                token: str = user_token(iam_server=iam_server,
                                        args=args,
                                        errors=errors,
                                        logger=__IAM_LOGGER)
    else:
        msg: str = "User identification not provided"
        errors.append(msg)
        if __IAM_LOGGER:
            __IAM_LOGGER.error(msg=msg)

    result: Response
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = jsonify({"user-id": user_id,
                          "token": token})
    # log the response
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=f"Response {result}, {result.get_data(as_text=True)}")

    return result


# @flask_app.route(rule=<callback_endpoint>,  # KEYCLOAK_ENDPOINT_EXCHANGE
#                  methods=["POST"])
def service_exchange() -> Response:
    """
    Entry point for requesting the *IAM* server to exchange the token.

    This is currently limited to the *KEYCLOAK* server. The token itself is stored in *KEYCLOAK*'s registry.
    The expected request parameters are:
        - user-id: identification for the reference user (aliases: 'user_id', 'login')
        - token: the token to be exchanged

    If the exchange is successful, the token data is stored in the *IAM* server's registry, and returned.
    Otherwise, *errors* will contain the appropriate error message.

    On success, the typical *Response* returned will contain the following attributes:
        {
            "token_type": "Bearer",
            "access_token": <str>,
            "expires_in": <number-of-seconds>,
            "refresh_token": <str>,
            "refesh_expires_in": <number-of-seconds>
        }

    :return: *Response* containing the token data, or *BAD REQUEST*
    """
    # log the request
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=_log_init(request=request))

    errors: list[str] = []
    with _iam_lock:
        # retrieve the IAM server (currently, only 'IAM_KEYCLOAK' is supported)
        iam_server: IamServer = _get_iam_server(endpoint=request.endpoint,
                                                errors=errors,
                                                logger=__IAM_LOGGER)
        # exchange the token
        token_data: dict[str, Any] | None = None
        if iam_server:
            errors: list[str] = []
            token_data = token_exchange(iam_server=iam_server,
                                        args=request.args,
                                        errors=errors,
                                        logger=__IAM_LOGGER)
    result: Response
    if errors:
        result = Response("; ".join(errors))
        result.status_code = 400
    else:
        result = jsonify(token_data)

    # log the response
    if __IAM_LOGGER:
        __IAM_LOGGER.debug(msg=f"Response {result}, {result.get_data(as_text=True)}")

    return result


def _log_init(request: Request) -> str:
    """
    Build the messages for logging the request entry.

    :param request: the Request object
    :return: the log message
    """

    params: str = json.dumps(obj=request.args,
                             ensure_ascii=False)
    return f"Request {request.method}:{request.path}, params {params}"
