import re
from ast import literal_eval
from collections import defaultdict
from dataclasses import dataclass
from fractions import Fraction
from functools import lru_cache, partial, wraps
from sys import maxunicode
from typing import (
    Callable,
    DefaultDict,
    Dict,
    Iterable,
    List,
    Optional,
    Tuple,
    TypeVar,
    Union,
)
from unicodedata import (
    bidirectional,
    category,
    decimal,
    digit,
    numeric,
    unidata_version,
)

import z3  # type: ignore

from crosshair.z3util import z3IntVal


@dataclass
class CharMask:
    """
    Represents a mask over unicode codepoints.

    Each range's minimum is less than it's maximum. Ranges cover valid codepoints.
    inv: all(0 <= minb < maxb <= maxunicode + 1 for (minb, maxb) in self.all_bounds())

    Ranges are listed in-order and are not overlapping or adjacent.
    inv: all(self.bounds_at(i)[1] < self.bounds_at(i + 1)[0] for i in range(len(self.parts) - 1))
    """

    parts: List[Union[int, Tuple[int, int]]]

    def maybe_add_bounds(self, minimum: int, maximum: int) -> None:
        """
        Append a new range to the bounds, if possible.

        `minimum` must be greater than or equal to the minimum of the highest existing
        range.
        pre: len(self.parts) == 0 or minimum >= self.bounds_at(len(self.parts) - 1)[0]

        New bounds must be valid.
        pre: 0 <= minimum <= maxunicode + 1
        pre: 0 <= maximum <= maxunicode + 1
        """
        if minimum < maximum:
            if self.parts:
                last_min, last_max = self.bounds_at(len(self.parts) - 1)
                if minimum <= last_max:
                    if maximum <= last_max:
                        return
                    assert minimum >= last_min
                    self.parts[-1] = (last_min, maximum)
                    return
            self.parts.append(minimum if minimum + 1 == maximum else (minimum, maximum))

    def interpret_smt_function(self, smt_fn: z3.ExprRef) -> z3.ExprRef:
        """Return an interpretation for an smt (Int->Bool) function for this mask."""
        codepoint = z3.Int("c")
        return z3.ForAll(
            [codepoint],
            smt_fn(codepoint) == self.smt_matches(codepoint),
            patterns=[smt_fn(codepoint)],  # (always expand)
        )

    def smt_matches(self, smt_ch: z3.ExprRef):
        # TODO: We could precompute and re-use the IntVal() call results below.
        # (for big masks, building this Z3 expr takes hundreds of ms!!)
        constraints = []
        for part in self.parts:
            if isinstance(part, int):
                constraints.append(smt_ch == z3IntVal(part))
            else:
                constraints.append(
                    z3.And(z3IntVal(part[0]) <= smt_ch, smt_ch < z3IntVal(part[1]))
                )
        if len(constraints) <= 1:
            return constraints[0] if constraints else z3.BoolVal(False)
        else:
            return z3.Or(*constraints)

    def covers(self, codepoint: int) -> bool:
        for minimum, maximum in self.all_bounds():
            if minimum <= codepoint < maximum:
                return True
        return False

    def all_bounds(self) -> Iterable[Tuple[int, int]]:
        parts = self.parts
        for i in range(len(parts)):
            yield self.bounds_at(i)

    def bounds_at(self, idx: int) -> Tuple[int, int]:
        """
        Return bounds for an allowed range at the given index.

        pre: 0 <= idx < len(self.parts)
        """
        item = self.parts[idx]
        return (item, item + 1) if isinstance(item, int) else item

    def invert(self) -> "CharMask":
        numbers = [b for i in range(len(self.parts)) for b in self.bounds_at(i)]
        numbers = [0] + numbers + [maxunicode + 1]
        inversion = CharMask([])
        for i in range(1, len(numbers), 2):
            inversion.maybe_add_bounds(numbers[i - 1], numbers[i])
        return inversion

    def union(self, other: "CharMask") -> "CharMask":
        ret = CharMask([])
        for minimum, maximum in sorted([*self.all_bounds(), *other.all_bounds()]):
            ret.maybe_add_bounds(minimum, maximum)
        return ret

    def intersect(self, other: "CharMask") -> "CharMask":
        myparts = self.parts
        otherparts = other.parts
        result = CharMask([])
        for myidx in range(len(myparts)):
            for otheridx in range(len(otherparts)):
                (mymin, mymax) = self.bounds_at(myidx)
                (omin, omax) = other.bounds_at(otheridx)
                result.maybe_add_bounds(max(mymin, omin), min(mymax, omax))
        return result

    def subtract(self, other: "CharMask") -> "CharMask":
        return self.intersect(other.invert())


def compute_categories() -> Dict[str, Tuple[Union[int, Tuple[int, int]], ...]]:
    category_data: DefaultDict[str, List[Union[int, Tuple[int, int]]]] = defaultdict(
        list
    )
    last_point = 0
    custom_categories = [("word", re.compile(r"\w"), CharMask([]))]
    last_category = category(chr(0))
    for cur_point in range(1, maxunicode + 1):
        ch = chr(cur_point)
        for name, regex, hits in custom_categories:
            if regex.fullmatch(ch):
                hits.maybe_add_bounds(cur_point, cur_point + 1)
        cur_category = category(ch)
        if cur_category != last_category:
            item: Union[int, Tuple[int, int]] = (
                last_point if cur_point == last_point + 1 else (last_point, cur_point)
            )
            category_data[last_category].append(item)
            last_point, last_category = cur_point, cur_category

    category_data[category(chr(maxunicode))].append((last_point, maxunicode + 1))
    for name, regex, hits in custom_categories:
        category_data[name] = hits.parts

    return {k: tuple(v) for (k, v) in sorted(category_data.items())}


_CATEGORY_RANGES_CACHE: Optional[Dict[str, CharMask]] = None


def get_unicode_categories() -> Dict[str, CharMask]:
    global _CATEGORY_RANGES_CACHE
    if _CATEGORY_RANGES_CACHE is None:
        if unidata_version in _PRECOMPUTED_CATEGORY_RANGES:
            ranges_str = _PRECOMPUTED_CATEGORY_RANGES[unidata_version]
            cats = literal_eval(ranges_str)
        else:
            cats = compute_categories()
        _CATEGORY_RANGES_CACHE = {
            cat: CharMask(ranges) for (cat, ranges) in cats.items()
        }
    return _CATEGORY_RANGES_CACHE


@lru_cache(maxsize=None)
def get_unicode_mask(*cat_names: str, invert: bool = False) -> CharMask:
    cats = get_unicode_categories()
    mask = cats[cat_names[0]]
    for cat_name in cat_names[1:]:
        mask = mask.union(cats[cat_name])
    if invert:
        mask = mask.invert()
    return mask


_T = TypeVar("_T")


@lru_cache(maxsize=None)
def get_char_fn_map(mapping_fn: Callable[[str], Optional[_T]]) -> Dict[int, _T]:
    ret: Dict[int, _T] = {}
    for codepoint in range(maxunicode + 1):
        ch = chr(codepoint)
        try:
            val = mapping_fn(ch)
        except ValueError:
            val = None
        if val is not None:
            ret[codepoint] = val
    return ret


@lru_cache(maxsize=None)
def get_char_fn_domain_mask(mapping_fn: Callable[[str], Optional[_T]]) -> CharMask:
    mask = CharMask([])
    for key in sorted(get_char_fn_map(mapping_fn).keys()):
        mask.maybe_add_bounds(key, key + 1)
    return mask


@lru_cache(maxsize=None)
def get_char_predicate_mask(predicate: Callable[[str], bool]) -> CharMask:
    mask = CharMask([])
    for codepoint in range(maxunicode + 1):
        ch = chr(codepoint)
        if predicate(ch):
            mask.maybe_add_bounds(codepoint, codepoint + 1)
    return mask


def make_mask(vals: Iterable[int]) -> CharMask:
    mask = CharMask([])
    for val in vals:
        mask.maybe_add_bounds(val, val + 1)
    return mask


_INTERPRETATION_CACHE: Dict[str, z3.ExprRef] = {}


def _cached_int_transform(name: str, transforms: Dict[int, int]) -> z3.ExprRef:
    if name in _INTERPRETATION_CACHE:
        return _INTERPRETATION_CACHE[name]
    else:
        smt_fn = z3.Function(name, z3.IntSort(), z3.IntSort())
        val_to_key = defaultdict(list)
        for k, v in transforms.items():
            val_to_key[v].append(k)
        cpvar = z3.Int("c")
        ifthens = [
            ([cpvar == k for k in keys], smt_fn(cpvar) == val)
            for (val, keys) in val_to_key.items()
        ]
        interpretation = z3.And(
            *[smt_fn(k) == val for (val, keys) in val_to_key.items() for k in keys]
        )
        _INTERPRETATION_CACHE[name] = interpretation
        return interpretation


def _cached_mask_interpretation(
    name: str, mask_getter: Callable[[], CharMask]
) -> Tuple[z3.ExprRef, Callable[[z3.ExprRef], z3.ExprRef]]:
    if name in _INTERPRETATION_CACHE:
        return _INTERPRETATION_CACHE[name]
    else:
        mask = mask_getter()
        smt_fn = z3.Function(name, z3.IntSort(), z3.BoolSort())
        interpretation = mask.interpret_smt_function(smt_fn)
        ret = (interpretation, smt_fn)
        _INTERPRETATION_CACHE[name] = ret
        return ret


def transform_fn(transformer):
    name = transformer.__name__

    @wraps(transformer)
    def wrapper(self) -> z3.ExprRef:
        if name in self._cached_smt_fns:
            return self._cached_smt_fns[name]
        self.solver.add(_cached_int_transform(name, transformer(self)))
        smt_fn = z3.Function(name, z3.IntSort(), z3.IntSort())
        self._cached_smt_fns[name] = smt_fn
        return smt_fn

    return wrapper


def mask_fn(mask_getter):
    name = mask_getter.__name__

    @wraps(mask_getter)
    def wrapper(self, *a) -> z3.ExprRef:
        if name in self._cached_smt_fns:
            return self._cached_smt_fns[name]
        constr, checker = _cached_mask_interpretation(name, partial(mask_getter, self))
        self.solver.add(constr)
        self._cached_smt_fns[name] = checker
        return checker

    return wrapper


@lru_cache(maxsize=None)
def fractionmap():
    ret = []
    for k, v in get_char_fn_map(numeric).items():
        frac = Fraction(v).limit_denominator()
        ret.append((k, (frac.numerator, frac.denominator)))
    return ret


@lru_cache(maxsize=None)
def nummap(mapfn: Callable[[str], int]) -> List[Tuple[int, int]]:
    map = get_char_fn_map(mapfn)
    return list(map.items())  # type: ignore


@lru_cache(maxsize=None)
def casemap(casefn: Callable[[str], str]) -> List[Tuple[int, str]]:
    map = get_char_fn_map(lambda ch: None if casefn(ch) == ch else casefn(ch))
    return list(map.items())  # type: ignore


class UnicodeMaskCache:
    def __init__(self, solver: z3.Solver):
        self.solver = solver
        self._cached_smt_fns: Dict[str, z3.FuncDeclRef] = {}

    @mask_fn
    def ascii(self):
        return CharMask([(0, 128)])

    @mask_fn
    def alnum(self):
        alpha = get_unicode_mask("Lm", "Lt", "Lu", "Ll", "Lo")
        return alpha.union(get_char_fn_domain_mask(numeric))

    @mask_fn
    def alpha(self):
        return get_unicode_mask("Lm", "Lt", "Lu", "Ll", "Lo")

    @mask_fn
    def decimal(self):
        return get_unicode_mask("Nd")

    @mask_fn
    def digit(self):
        return get_char_fn_domain_mask(digit)

    @mask_fn
    def numeric(self):
        return get_char_fn_domain_mask(numeric)

    @mask_fn
    def lower(self):
        return get_char_fn_domain_mask(lambda ch: True if ch.islower() else None)

    @mask_fn
    def printable(self):
        printable = get_unicode_mask(
            "Cc", "Co", "Cn", "Cf", "Cs", "Zs", "Zl", "Zp", "Zs"
        ).invert()
        # The ascii space char is printable too:
        printable = printable.union(CharMask([32]))
        return printable

    @mask_fn
    def space(self):
        def _is_space_char(ch):
            if category(ch) == "Zs":
                return True
            if bidirectional(ch) in ("WS", "B", "S"):
                return True
            return None

        return get_char_predicate_mask(_is_space_char)

    @mask_fn
    def newline(self):
        nls = (
            "\n",
            "\x0b",
            "\x0c",
            "\r",
            "\x1c",
            "\x1d",
            "\x1e",
            "\x85",
            "\u2028",
            "\u2029",
        )
        return CharMask(list(map(ord, nls)))

    @mask_fn
    def upper(self):
        return get_char_fn_domain_mask(lambda ch: True if ch.isupper() else None)

    @mask_fn
    def title(self):
        return get_char_fn_domain_mask(lambda ch: True if ch.istitle() else None)

    @mask_fn
    def word(self):
        return get_unicode_categories()["word"]

    @mask_fn
    def casefold_exists(self):
        return make_mask(k for k, v in casemap(str.casefold))

    @transform_fn
    def casefold_1st(self):
        return {k: ord(v[0]) for k, v in casemap(str.casefold)}

    @mask_fn
    def casefold_2nd_exists(self):
        return make_mask(k for k, v in casemap(str.casefold) if len(v) >= 2)

    @transform_fn
    def casefold_2nd(self):
        return {k: ord(v[1]) for k, v in casemap(str.casefold) if len(v) >= 2}

    @mask_fn
    def casefold_3rd_exists(self):
        return make_mask(k for k, v in casemap(str.casefold) if len(v) >= 3)

    @transform_fn
    def casefold_3rd(self):
        return {k: ord(v[2]) for k, v in casemap(str.casefold) if len(v) >= 3}

    @mask_fn
    def tolower_exists(self):
        return make_mask(k for k, v in casemap(str.lower))

    @transform_fn
    def tolower_1st(self):
        return {k: ord(v[0]) for k, v in casemap(str.lower)}

    @mask_fn
    def tolower_2nd_exists(self):
        return make_mask(k for k, v in casemap(str.lower) if len(v) >= 2)

    @transform_fn
    def tolower_2nd(self):
        return {k: ord(v[1]) for k, v in casemap(str.lower) if len(v) >= 2}

    @mask_fn
    def totitle_exists(self):
        return make_mask(k for k, v in casemap(str.title))

    @transform_fn
    def totitle_1st(self):
        return {k: ord(v[0]) for k, v in casemap(str.title)}

    @mask_fn
    def totitle_2nd_exists(self):
        return make_mask(k for k, v in casemap(str.title) if len(v) >= 2)

    @transform_fn
    def totitle_2nd(self):
        return {k: ord(v[1]) for k, v in casemap(str.title) if len(v) >= 2}

    @mask_fn
    def totitle_3rd_exists(self):
        return make_mask(k for k, v in casemap(str.title) if len(v) >= 3)

    @transform_fn
    def totitle_3rd(self):
        return {k: ord(v[2]) for k, v in casemap(str.title) if len(v) >= 3}

    @mask_fn
    def toupper_exists(self):
        return make_mask(k for k, v in casemap(str.upper))

    @transform_fn
    def toupper_1st(self):
        return {k: ord(v[0]) for k, v in casemap(str.upper)}

    @mask_fn
    def toupper_2nd_exists(self):
        return make_mask(k for k, v in casemap(str.upper) if len(v) >= 2)

    @transform_fn
    def toupper_2nd(self):
        return {k: ord(v[1]) for k, v in casemap(str.upper) if len(v) >= 2}

    @mask_fn
    def toupper_3rd_exists(self):
        return make_mask(k for k, v in casemap(str.upper) if len(v) >= 3)

    @transform_fn
    def toupper_3rd(self):
        return {k: ord(v[2]) for k, v in casemap(str.upper) if len(v) >= 3}

    @mask_fn
    def digit_exists(self):
        return make_mask(k for k, v in nummap(digit))

    @transform_fn
    def decimal_int(self):
        return dict(nummap(decimal))

    @mask_fn
    def decimal_exists(self):
        return make_mask(k for k, v in nummap(decimal))

    @transform_fn
    def digit_int(self):
        return dict(nummap(digit))

    @mask_fn
    def numeric_exists(self):
        return make_mask(k for k, v in fractionmap())

    @transform_fn
    def numeric_numerator(self):
        return {k: v[0] for k, v in fractionmap()}

    @transform_fn
    def numeric_denominator(self):
        return {k: v[1] for k, v in fractionmap()}


# fmt: off

# We store these values as repr strings, because we don't want to burden IDEs and dev
# tools with massive native data structures.
_PRECOMPUTED_CATEGORY_RANGES: Dict[str, str] = {
    "11.0.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1565,1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2208),2229,(2238,2259),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2902),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3133),3141,3145,(3150,3157),3159,(3163,3168),(3172,3174),(3184,3192),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3294),3295,(3300,3302),3312,(3315,3328),3332,3341,3345,3397,3401,(3408,3412),(3428,3430),(3456,3458),3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,(3717,3719),3721,(3723,3725),(3726,3732),3736,3744,3748,3750,(3752,3754),3756,3770,(3774,3776),3781,3783,(3790,3792),(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),5901,(5909,5920),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),6159,(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6847,6912),(6988,6992),(7037,7040),(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7418,7424),7674,(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8384,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),(11158,11160),11209,11263,11311,11359,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11855,11904),11930,(12020,12032),(12246,12272),(12284,12288),12352,(12439,12441),(12544,12549),12592,12687,(12731,12736),(12772,12784),12831,13055,(19894,19904),(40944,40960),(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42938,42999),(43052,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43878,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64450,64467),(64832,64848),(64912,64914),(64968,65008),(65022,65024),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65948,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),(66928,67072),(67383,67392),(67414,67424),(67432,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),(69247,69376),(69416,69424),(69466,69632),(69710,69714),(69744,69759),(69826,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69959,69968),(70007,70016),(70094,70096),70112,(70133,70144),70162,(70207,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70746,70748,(70751,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71352,71360),(71370,71424),(71451,71453),(71468,71472),(71488,71680),(71740,71840),(71923,71935),(71936,72192),(72264,72272),(72324,72326),(72355,72384),(72441,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73728),(74650,74752),74863,(74869,74880),(75076,77824),(78895,82944),(83527,92160),(92729,92736),92767,(92778,92782),(92784,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94021,94032),(94079,94095),(94112,94176),(94178,94208),(100338,100352),(101107,110592),(110879,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118784),(119030,119040),(119079,119081),(119273,119296),(119366,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122880),122887,(122905,122907),122914,122917,(122923,124928),(125125,125127),(125143,125184),(125259,125264),(125274,125278),(125280,126065),(126133,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127245,127248),(127340,127344),(127405,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128725,128736),(128749,128752),(128762,128768),(128884,128896),(128985,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129280),(129292,129296),129343,(129393,129395),(129399,129402),129403,(129443,129456),(129466,129472),(129475,129488),(129536,129632),(129646,131072),(173783,173824),(177973,177984),(178206,178208),(183970,183984),(191457,194560),(195102,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11359),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,43002,(43824,43867),(43872,43878),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),65392,(65438,65440),(92992,92996),(94099,94112),(94176,94178)),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2208,2229),(2230,2238),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3313,3315),(3333,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3719,3721),3722,3725,(3732,3736),(3737,3744),(3745,3748),3749,3751,(3754,3756),(3757,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6988),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7410),(7413,7415),(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12731),(12784,12800),(13312,19894),(19968,40944),(40960,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69376,69405),69415,(69424,69446),(69635,69688),(69763,69808),(69840,69865),(69891,69927),69956,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),(71424,71451),(71680,71724),71935,72192,(72203,72243),72250,72272,(72284,72324),(72326,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),(73728,74650),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94021),94032,(94208,100338),(100352,101107),(110592,110879),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173783),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11311),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,(65313,65339),(66560,66600),(66736,66772),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,(7410,7412),7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43453,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),(94033,94079),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2259,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,2902,(2914,2916),2946,3008,3021,3072,3076,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3770),(3771,3773),(3784,3790),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5941),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7674),(7675,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),43452,43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69446,69457),69633,(69688,69703),(69759,69762),(69811,69815),(69817,69819),(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),(70191,70194),70196,(70198,70200),70206,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(92912,92917),(92976,92983),(94095,94099),(113821,113823),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72784,72794),(73040,73050),(73120,73130),(92768,92778),(93008,93018),(120782,120832),(125264,125274)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(93019,93026),(93824,93847),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,12316,12336,12448,(65073,65075),65112,65123,65293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1566,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),(5741,5743),(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11855),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),70747,70749,70854,(71105,71128),(71233,71236),(71264,71277),(71484,71487),71739,(72255,72263),(72346,72349),(72350,72355),(72769,72774),(72816,72818),(73463,73465),(74864,74869),(92782,92784),92917,(92983,92988),92996,(93847,93851),113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8384),43064,65020,65129,65284,(65504,65506),(65509,65511),126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(64434,64450),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11160,11209),(11210,11263),(11493,11499),(11904,11930),(11931,12020),(12032,12246),(12272,12284),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13055),(13056,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),65021,65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65948),65952,(66000,66045),(67703,67705),68296,71487,(92988,92992),92997,113820,(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119273),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),126124,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127248,127340),(127344,127405),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128725),(128736,128749),(128752,128762),(128768,128884),(128896,128985),(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129280,129292),(129296,129343),(129344,129393),(129395,129399),129402,(129404,129443),(129456,129466),(129472,129475),(129488,129536),(129632,129646)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2208,2229),(2230,2238),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3302,3312),(3313,3315),(3333,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3719,3721),3722,3725,(3732,3736),(3737,3744),(3745,3748),3749,3751,(3754,3756),(3757,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6988),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7410),(7413,7415),(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11311),(11312,11359),(11360,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12731),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19894),(19968,40944),(40960,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42938),(42999,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43878),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69376,69416),(69424,69446),(69457,69461),(69635,69688),(69714,69744),(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),(71360,71370),(71424,71451),(71472,71484),(71680,71724),(71840,71923),71935,72192,(72203,72243),72250,72272,(72284,72324),(72326,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),(73728,74650),(74752,74863),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94021),94032,(94099,94112),(94176,94178),(94208,100338),(100352,101107),(110592,110879),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(124928,125125),(125127,125136),(125184,125252),(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(131072,173783),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102))}",
    "12.1.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78905),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1565,1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2208),2229,(2238,2259),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2902),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3133),3141,3145,(3150,3157),3159,(3163,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3294),3295,(3300,3302),3312,(3315,3328),3332,3341,3345,3397,3401,(3408,3412),(3428,3430),(3456,3458),3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,(3790,3792),(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),5901,(5909,5920),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),6159,(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6847,6912),(6988,6992),(7037,7040),(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7419,7424),7674,(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8384,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),(11158,11160),11311,11359,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11856,11904),11930,(12020,12032),(12246,12272),(12284,12288),12352,(12439,12441),(12544,12549),12592,12687,(12731,12736),(12772,12784),12831,(19894,19904),(40944,40960),(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42944,42946),(42951,42999),(43052,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43880,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64450,64467),(64832,64848),(64912,64914),(64968,65008),(65022,65024),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65948,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),(66928,67072),(67383,67392),(67414,67424),(67432,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),(69247,69376),(69416,69424),(69466,69600),(69623,69632),(69710,69714),(69744,69759),(69826,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69959,69968),(70007,70016),(70094,70096),70112,(70133,70144),70162,(70207,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70746,70748,(70752,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71353,71360),(71370,71424),(71451,71453),(71468,71472),(71488,71680),(71740,71840),(71923,71935),(71936,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72384),(72441,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77824),78895,(78905,82944),(83527,92160),(92729,92736),92767,(92778,92782),(92784,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94180,94208),(100344,100352),(101107,110592),(110879,110928),(110931,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118784),(119030,119040),(119079,119081),(119273,119296),(119366,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122880),122887,(122905,122907),122914,122917,(122923,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123584),(123642,123647),(123648,124928),(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127245,127248),(127341,127344),(127405,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128726,128736),(128749,128752),(128763,128768),(128884,128896),(128985,128992),(129004,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129280),129292,129394,(129399,129402),(129443,129445),(129451,129454),(129483,129485),(129620,129632),(129646,129648),(129652,129656),(129659,129664),(129667,129680),(129686,131072),(173783,173824),(177973,177984),(178206,178208),(183970,183984),(191457,194560),(195102,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11359),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42947,43002,(43824,43867),(43872,43880),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),65392,(65438,65440),(92992,92996),(94099,94112),(94176,94178),94179,(123191,123198),125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2208,2229),(2230,2238),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3313,3315),(3333,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6988),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12731),(12784,12800),(13312,19894),(19968,40944),(40960,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69376,69405),69415,(69424,69446),(69600,69623),(69635,69688),(69763,69808),(69840,69865),(69891,69927),69956,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),70751,(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71680,71724),71935,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),(73728,74650),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94027),94032,(94208,100344),(100352,101107),(110592,110879),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(123136,123181),123214,(123584,123628),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173783),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11311),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42946,(42948,42951),(65313,65339),(66560,66600),(66736,66772),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),(94033,94088),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2259,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,2902,(2914,2916),2946,3008,3021,3072,3076,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3790),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5941),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7674),(7675,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69446,69457),69633,(69688,69703),(69759,69762),(69811,69815),(69817,69819),(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),(70191,70194),70196,(70198,70200),70206,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(92912,92917),(92976,92983),94031,(94095,94099),(113821,113823),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),(123184,123191),(123628,123632),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72784,72794),(73040,73050),(73120,73130),(92768,92778),(93008,93018),(120782,120832),(123200,123210),(123632,123642),(125264,125274)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,12316,12336,12448,(65073,65075),65112,65123,65293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1566,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),70747,70749,70854,(71105,71128),(71233,71236),(71264,71277),(71484,71487),71739,72162,(72255,72263),(72346,72349),(72350,72355),(72769,72774),(72816,72818),(73463,73465),73727,(74864,74869),(92782,92784),92917,(92983,92988),92996,(93847,93851),94178,113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8384),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(64434,64450),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11160,11264),(11493,11499),(11904,11930),(11931,12020),(12032,12246),(12272,12284),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),65021,65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65948),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119273),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127248,127341),(127344,127405),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128726),(128736,128749),(128752,128763),(128768,128884),(128896,128985),(128992,129004),(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129280,129292),(129293,129394),(129395,129399),(129402,129443),(129445,129451),(129454,129483),(129485,129620),(129632,129646),(129648,129652),(129656,129659),(129664,129667),(129680,129686)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2208,2229),(2230,2238),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3302,3312),(3313,3315),(3333,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6988),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11311),(11312,11359),(11360,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12731),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19894),(19968,40944),(40960,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42944),(42946,42951),(42999,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43880),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69376,69416),(69424,69446),(69457,69461),(69600,69623),(69635,69688),(69714,69744),(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),70751,(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71424,71451),(71472,71484),(71680,71724),(71840,71923),71935,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101107),(110592,110879),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(123136,123181),(123191,123198),(123200,123210),123214,(123584,123628),(123632,123642),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(131072,173783),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102))}",
    "13.0.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78905),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1565,1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2208),2229,(2248,2259),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2901),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3133),3141,3145,(3150,3157),3159,(3163,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3294),3295,(3300,3302),3312,(3315,3328),3341,3345,3397,3401,(3408,3412),(3428,3430),3456,3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,(3790,3792),(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),5901,(5909,5920),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),6159,(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6849,6912),(6988,6992),(7037,7040),(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7419,7424),7674,(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8384,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),11158,11311,11359,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11859,11904),11930,(12020,12032),(12246,12272),(12284,12288),12352,(12439,12441),(12544,12549),12592,12687,(12772,12784),12831,(40957,40960),(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42944,42946),(42955,42997),(43053,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43884,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64450,64467),(64832,64848),(64912,64914),(64968,65008),(65022,65024),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65949,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),(66928,67072),(67383,67392),(67414,67424),(67432,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),69247,69290,(69294,69296),(69298,69376),(69416,69424),(69466,69552),(69580,69600),(69623,69632),(69710,69714),(69744,69759),(69826,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69960,69968),(70007,70016),70112,(70133,70144),70162,(70207,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70748,(70754,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71353,71360),(71370,71424),(71451,71453),(71468,71472),(71488,71680),(71740,71840),(71923,71935),(71943,71945),(71946,71948),71956,71959,71990,(71993,71995),(72007,72016),(72026,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72384),(72441,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73648),(73649,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77824),78895,(78905,82944),(83527,92160),(92729,92736),92767,(92778,92782),(92784,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94181,94192),(94194,94208),(100344,100352),(101590,101632),(101641,110592),(110879,110928),(110931,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118784),(119030,119040),(119079,119081),(119273,119296),(119366,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122880),122887,(122905,122907),122914,122917,(122923,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123584),(123642,123647),(123648,124928),(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127406,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128728,128736),(128749,128752),(128765,128768),(128884,128896),(128985,128992),(129004,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129200),(129202,129280),129401,129484,(129620,129632),(129646,129648),(129653,129656),(129659,129664),(129671,129680),(129705,129712),(129719,129728),(129731,129744),(129751,129792),129939,(129995,130032),(130042,131072),(173790,173824),(177973,177984),(178206,178208),(183970,183984),(191457,194560),(195102,196608),(201547,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11359),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42947,42952,42954,42998,43002,(43824,43867),(43872,43881),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),43881,65392,(65438,65440),(92992,92996),(94099,94112),(94176,94178),94179,(123191,123198),125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2208,2229),(2230,2248),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6988),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12736),(12784,12800),(13312,19904),(19968,40957),(40960,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69248,69290),(69296,69298),(69376,69405),69415,(69424,69446),(69552,69573),(69600,69623),(69635,69688),(69763,69808),(69840,69865),(69891,69927),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70751,70754),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71680,71724),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),73648,(73728,74650),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94027),94032,(94208,100344),(100352,101590),(101632,101641),(110592,110879),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(123136,123181),123214,(123584,123628),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173790),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11311),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42946,(42948,42952),42953,42997,(65313,65339),(66560,66600),(66736,66772),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),70094,(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,(71984,71990),(71991,71993),71997,72000,72002,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),(94033,94088),(94192,94194),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2259,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,(2901,2903),(2914,2916),2946,3008,3021,3072,3076,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3457,3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3790),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5941),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6847,6849),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7674),(7675,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),43052,(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69291,69293),(69446,69457),69633,(69688,69703),(69759,69762),(69811,69815),(69817,69819),(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),70095,(70191,70194),70196,(70198,70200),70206,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(71995,71997),71998,72003,(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(92912,92917),(92976,92983),94031,(94095,94099),94180,(113821,113823),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),(123184,123191),(123628,123632),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72016,72026),(72784,72794),(73040,73050),(73120,73130),(92768,92778),(93008,93018),(120782,120832),(123200,123210),(123632,123642),(125264,125274),(130032,130042)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69573,69580),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,12316,12336,12448,(65073,65075),65112,65123,65293,69293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1566,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),11858,(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),(70746,70748),70749,70854,(71105,71128),(71233,71236),(71264,71277),(71484,71487),71739,(72004,72007),72162,(72255,72263),(72346,72349),(72350,72355),(72769,72774),(72816,72818),(73463,73465),73727,(74864,74869),(92782,92784),92917,(92983,92988),92996,(93847,93851),94178,113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8384),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(43882,43884),(64434,64450),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11159,11264),(11493,11499),(11856,11858),(11904,11930),(11931,12020),(12032,12246),(12272,12284),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),65021,65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65949),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119273),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127245,127406),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128728),(128736,128749),(128752,128765),(128768,128884),(128896,128985),(128992,129004),(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129200,129202),(129280,129401),(129402,129484),(129485,129620),(129632,129646),(129648,129653),(129656,129659),(129664,129671),(129680,129705),(129712,129719),(129728,129731),(129744,129751),(129792,129939),(129940,129995)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2208,2229),(2230,2248),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,3294,(3296,3298),(3302,3312),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5901),(5902,5906),(5920,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6988),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11311),(11312,11359),(11360,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12736),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19904),(19968,40957),(40960,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42944),(42946,42955),(42997,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43882),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69248,69290),(69296,69298),(69376,69416),(69424,69446),(69457,69461),(69552,69580),(69600,69623),(69635,69688),(69714,69744),(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),(70751,70754),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71424,71451),(71472,71484),(71680,71724),(71840,71923),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72016,72026),(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72384,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),73648,(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101590),(101632,101641),(110592,110879),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(123136,123181),(123191,123198),(123200,123210),123214,(123584,123628),(123632,123642),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(130032,130042),(131072,173790),(173824,177973),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547))}",
    "14.0.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,(2192,2194),2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78905),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2160),2191,(2194,2200),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2901),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3132),3141,3145,(3150,3157),3159,(3163,3165),(3166,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3293),3295,(3300,3302),3312,(3315,3328),3341,3345,3397,3401,(3408,3412),(3428,3430),3456,3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,(3790,3792),(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),(5910,5919),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6863,6912),(6989,6992),7039,(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7419,7424),(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8385,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),11158,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11870,11904),11930,(12020,12032),(12246,12272),(12284,12288),12352,(12439,12441),(12544,12549),12592,12687,(12772,12784),12831,(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42955,42960),42962,42964,(42970,42994),(43053,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43884,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64451,64467),(64912,64914),(64968,64975),(64976,65008),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65949,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),66939,66955,66963,66966,66978,66994,67002,(67005,67072),(67383,67392),(67414,67424),(67432,67456),67462,67505,(67515,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),69247,69290,(69294,69296),(69298,69376),(69416,69424),(69466,69488),(69514,69552),(69580,69600),(69623,69632),(69710,69714),(69750,69759),(69827,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69960,69968),(70007,70016),70112,(70133,70144),70162,(70207,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70748,(70754,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71354,71360),(71370,71424),(71451,71453),(71468,71472),(71495,71680),(71740,71840),(71923,71935),(71943,71945),(71946,71948),71956,71959,71990,(71993,71995),(72007,72016),(72026,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72368),(72441,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73648),(73649,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77712),(77811,77824),78895,(78905,82944),(83527,92160),(92729,92736),92767,(92778,92782),92863,(92874,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94181,94192),(94194,94208),(100344,100352),(101590,101632),(101641,110576),110580,110588,110591,(110883,110928),(110931,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118528),(118574,118576),(118599,118608),(118724,118784),(119030,119040),(119079,119081),(119275,119296),(119366,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122624),(122655,122880),122887,(122905,122907),122914,122917,(122923,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123536),(123567,123584),(123642,123647),(123648,124896),124903,124908,124911,124927,(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127406,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128728,128733),(128749,128752),(128765,128768),(128884,128896),(128985,128992),(129004,129008),(129009,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129200),(129202,129280),(129620,129632),(129646,129648),(129653,129656),(129661,129664),(129671,129680),(129709,129712),(129723,129728),(129734,129744),(129754,129760),(129768,129776),(129783,129792),129939,(129995,130032),(130042,131072),(173792,173824),(177977,177984),(178206,178208),(183970,183984),(191457,194560),(195102,196608),(201547,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11360),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42945,42947,42952,42954,42961,42963,42965,42967,42969,42998,43002,(43824,43867),(43872,43881),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(122624,122634),(122635,122655),(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2249,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(42994,42997),(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),43881,65392,(65438,65440),(67456,67462),(67463,67505),(67506,67515),(92992,92996),(94099,94112),(94176,94178),94179,(110576,110580),(110581,110588),(110589,110591),(123191,123198),125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2249),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6989),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12736),(12784,12800),(13312,19904),(19968,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69248,69290),(69296,69298),(69376,69405),69415,(69424,69446),(69488,69506),(69552,69573),(69600,69623),(69635,69688),(69745,69747),69749,(69763,69808),(69840,69865),(69891,69927),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70751,70754),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71488,71495),(71680,71724),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),73648,(73728,74650),(74880,75076),(77712,77809),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92784,92863),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94027),94032,(94208,100344),(100352,101590),(101632,101641),(110592,110883),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),122634,(123136,123181),123214,(123536,123566),(123584,123628),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173792),(173824,177977),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11312),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42944,42946,(42948,42952),42953,42960,42966,42968,42997,(65313,65339),(66560,66600),(66736,66772),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),5909,5940,6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),70094,(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,(71984,71990),(71991,71993),71997,72000,72002,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),(94033,94088),(94192,94194),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2200,2208),(2250,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,(2901,2903),(2914,2916),2946,3008,3021,3072,3076,3132,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3457,3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3790),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5940),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),6159,(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6847,6863),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),43052,(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69291,69293),(69446,69457),(69506,69510),69633,(69688,69703),69744,(69747,69749),(69759,69762),(69811,69815),(69817,69819),69826,(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),70095,(70191,70194),70196,(70198,70200),70206,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(71995,71997),71998,72003,(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(92912,92917),(92976,92983),94031,(94095,94099),94180,(113821,113823),(118528,118574),(118576,118599),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),(123184,123191),123566,(123628,123632),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72016,72026),(72784,72794),(73040,73050),(73120,73130),(92768,92778),(92864,92874),(93008,93018),(120782,120832),(123200,123210),(123632,123642),(125264,125274),(130032,130042)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69573,69580),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,11869,12316,12336,12448,(65073,65075),65112,65123,65293,69293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,11862,11864,11866,11868,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1565,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7037,7039),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),(11858,11861),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69510,69514),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),(70746,70748),70749,70854,(71105,71128),(71233,71236),(71264,71277),71353,(71484,71487),71739,(72004,72007),72162,(72255,72263),(72346,72349),(72350,72355),(72769,72774),(72816,72818),(73463,73465),73727,(74864,74869),(77809,77811),(92782,92784),92917,(92983,92988),92996,(93847,93851),94178,113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,11861,11863,11865,11867,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8385),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),2184,8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(43882,43884),(64434,64451),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11159,11264),(11493,11499),(11856,11858),(11904,11930),(11931,12020),(12032,12246),(12272,12284),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),(64832,64848),64975,(65021,65024),65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65949),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(118608,118724),(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119275),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127245,127406),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128728),(128733,128749),(128752,128765),(128768,128884),(128896,128985),(128992,129004),129008,(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129200,129202),(129280,129620),(129632,129646),(129648,129653),(129656,129661),(129664,129671),(129680,129709),(129712,129723),(129728,129734),(129744,129754),(129760,129768),(129776,129783),(129792,129939),(129940,129995)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2250),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3302,3312),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6989),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12736),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19904),(19968,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42955),(42960,42962),42963,(42965,42970),(42994,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43882),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(67072,67383),(67392,67414),(67424,67432),(67456,67462),(67463,67505),(67506,67515),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69248,69290),(69296,69298),(69376,69416),(69424,69446),(69457,69461),(69488,69506),(69552,69580),(69600,69623),(69635,69688),(69714,69744),(69745,69747),69749,(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),(70751,70754),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71424,71451),(71472,71484),(71488,71495),(71680,71724),(71840,71923),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72016,72026),(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),73648,(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77712,77809),(77824,78895),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92784,92863),(92864,92874),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101590),(101632,101641),(110576,110580),(110581,110588),(110589,110591),(110592,110883),(110928,110931),(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(122624,122655),(123136,123181),(123191,123198),(123200,123210),123214,(123536,123566),(123584,123628),(123632,123642),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(130032,130042),(131072,173792),(173824,177977),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547))}",
    "15.0.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,(2192,2194),2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78912),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2160),2191,(2194,2200),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2901),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3132),3141,3145,(3150,3157),3159,(3163,3165),(3166,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3293),3295,(3300,3302),3312,(3316,3328),3341,3345,3397,3401,(3408,3412),(3428,3430),3456,3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,3791,(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),(5910,5919),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6863,6912),(6989,6992),7039,(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7419,7424),(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8385,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),11158,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11870,11904),11930,(12020,12032),(12246,12272),(12284,12288),12352,(12439,12441),(12544,12549),12592,12687,(12772,12784),12831,(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42955,42960),42962,42964,(42970,42994),(43053,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43884,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64451,64467),(64912,64914),(64968,64975),(64976,65008),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65949,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),66939,66955,66963,66966,66978,66994,67002,(67005,67072),(67383,67392),(67414,67424),(67432,67456),67462,67505,(67515,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),69247,69290,(69294,69296),(69298,69373),(69416,69424),(69466,69488),(69514,69552),(69580,69600),(69623,69632),(69710,69714),(69750,69759),(69827,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69960,69968),(70007,70016),70112,(70133,70144),70162,(70210,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70748,(70754,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71354,71360),(71370,71424),(71451,71453),(71468,71472),(71495,71680),(71740,71840),(71923,71935),(71943,71945),(71946,71948),71956,71959,71990,(71993,71995),(72007,72016),(72026,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72368),(72441,72448),(72458,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73472),73489,(73531,73534),(73562,73648),(73649,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77712),(77811,77824),(78934,82944),(83527,92160),(92729,92736),92767,(92778,92782),92863,(92874,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94181,94192),(94194,94208),(100344,100352),(101590,101632),(101641,110576),110580,110588,110591,(110883,110898),(110899,110928),(110931,110933),(110934,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118528),(118574,118576),(118599,118608),(118724,118784),(119030,119040),(119079,119081),(119275,119296),(119366,119488),(119508,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122624),(122655,122661),(122667,122880),122887,(122905,122907),122914,122917,(122923,122928),(122990,123023),(123024,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123536),(123567,123584),(123642,123647),(123648,124112),(124154,124896),124903,124908,124911,124927,(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127406,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128728,128732),(128749,128752),(128765,128768),(128887,128891),(128986,128992),(129004,129008),(129009,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129200),(129202,129280),(129620,129632),(129646,129648),(129661,129664),(129673,129680),129726,(129734,129742),(129756,129760),(129769,129776),(129785,129792),129939,(129995,130032),(130042,131072),(173792,173824),(177978,177984),(178206,178208),(183970,183984),(191457,194560),(195102,196608),(201547,201552),(205744,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11360),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42945,42947,42952,42954,42961,42963,42965,42967,42969,42998,43002,(43824,43867),(43872,43881),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(122624,122634),(122635,122655),(122661,122667),(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2249,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(42994,42997),(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),43881,65392,(65438,65440),(67456,67462),(67463,67505),(67506,67515),(92992,92996),(94099,94112),(94176,94178),94179,(110576,110580),(110581,110588),(110589,110591),(122928,122990),(123191,123198),124139,125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2249),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6989),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12736),(12784,12800),(13312,19904),(19968,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69248,69290),(69296,69298),(69376,69405),69415,(69424,69446),(69488,69506),(69552,69573),(69600,69623),(69635,69688),(69745,69747),69749,(69763,69808),(69840,69865),(69891,69927),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70751,70754),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71488,71495),(71680,71724),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),73474,(73476,73489),(73490,73524),73648,(73728,74650),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(82944,83527),(92160,92729),(92736,92767),(92784,92863),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94027),94032,(94208,100344),(100352,101590),(101632,101641),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),122634,(123136,123181),123214,(123536,123566),(123584,123628),(124112,124139),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547),(201552,205744)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11312),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42944,42946,(42948,42952),42953,42960,42966,42968,42997,(65313,65339),(66560,66600),(66736,66772),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),3315,(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),5909,5940,6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),70094,(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,(71984,71990),(71991,71993),71997,72000,72002,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),73475,(73524,73526),(73534,73536),73537,(94033,94088),(94192,94194),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2200,2208),(2250,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,(2901,2903),(2914,2916),2946,3008,3021,3072,3076,3132,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3457,3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3791),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5940),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),6159,(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6847,6863),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),43052,(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69291,69293),(69373,69376),(69446,69457),(69506,69510),69633,(69688,69703),69744,(69747,69749),(69759,69762),(69811,69815),(69817,69819),69826,(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),70095,(70191,70194),70196,(70198,70200),70206,70209,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(71995,71997),71998,72003,(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(73472,73474),(73526,73531),73536,73538,78912,(78919,78934),(92912,92917),(92976,92983),94031,(94095,94099),94180,(113821,113823),(118528,118574),(118576,118599),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),123023,(123184,123191),123566,(123628,123632),(124140,124144),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72016,72026),(72784,72794),(73040,73050),(73120,73130),(73552,73562),(92768,92778),(92864,92874),(93008,93018),(120782,120832),(123200,123210),(123632,123642),(124144,124154),(125264,125274),(130032,130042)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69573,69580),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119488,119508),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,11869,12316,12336,12448,(65073,65075),65112,65123,65293,69293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,11862,11864,11866,11868,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1565,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7037,7039),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),(11858,11861),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69510,69514),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),(70746,70748),70749,70854,(71105,71128),(71233,71236),(71264,71277),71353,(71484,71487),71739,(72004,72007),72162,(72255,72263),(72346,72349),(72350,72355),(72448,72458),(72769,72774),(72816,72818),(73463,73465),(73539,73552),73727,(74864,74869),(77809,77811),(92782,92784),92917,(92983,92988),92996,(93847,93851),94178,113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,11861,11863,11865,11867,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8385),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),2184,8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(43882,43884),(64434,64451),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11159,11264),(11493,11499),(11856,11858),(11904,11930),(11931,12020),(12032,12246),(12272,12284),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),(64832,64848),64975,(65021,65024),65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65949),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(118608,118724),(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119275),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127245,127406),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128728),(128732,128749),(128752,128765),(128768,128887),(128891,128986),(128992,129004),129008,(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129200,129202),(129280,129620),(129632,129646),(129648,129661),(129664,129673),(129680,129726),(129727,129734),(129742,129756),(129760,129769),(129776,129785),(129792,129939),(129940,129995)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2250),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3302,3312),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6989),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12736),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19904),(19968,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42955),(42960,42962),42963,(42965,42970),(42994,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43882),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(67072,67383),(67392,67414),(67424,67432),(67456,67462),(67463,67505),(67506,67515),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69248,69290),(69296,69298),(69376,69416),(69424,69446),(69457,69461),(69488,69506),(69552,69580),(69600,69623),(69635,69688),(69714,69744),(69745,69747),69749,(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),(70751,70754),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71424,71451),(71472,71484),(71488,71495),(71680,71724),(71840,71923),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72016,72026),(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),73474,(73476,73489),(73490,73524),(73552,73562),73648,(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92784,92863),(92864,92874),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101590),(101632,101641),(110576,110580),(110581,110588),(110589,110591),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119488,119508),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(122624,122655),(122661,122667),(122928,122990),(123136,123181),(123191,123198),(123200,123210),123214,(123536,123566),(123584,123628),(123632,123642),(124112,124140),(124144,124154),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(130032,130042),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(194560,195102),(196608,201547),(201552,205744))}",
    "15.1.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,(2192,2194),2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78912),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2160),2191,(2194,2200),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2901),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3132),3141,3145,(3150,3157),3159,(3163,3165),(3166,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3293),3295,(3300,3302),3312,(3316,3328),3341,3345,3397,3401,(3408,3412),(3428,3430),3456,3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,3791,(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),(5910,5919),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6863,6912),(6989,6992),7039,(7156,7164),(7224,7227),(7242,7245),(7305,7312),(7355,7357),(7368,7376),(7419,7424),(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8385,8400),(8433,8448),(8588,8592),(9255,9280),(9291,9312),(11124,11126),11158,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11870,11904),11930,(12020,12032),(12246,12272),12352,(12439,12441),(12544,12549),12592,12687,(12772,12783),12831,(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42955,42960),42962,42964,(42970,42994),(43053,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43884,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64451,64467),(64912,64914),(64968,64975),(64976,65008),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65949,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),66939,66955,66963,66966,66978,66994,67002,(67005,67072),(67383,67392),(67414,67424),(67432,67456),67462,67505,(67515,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,69216),69247,69290,(69294,69296),(69298,69373),(69416,69424),(69466,69488),(69514,69552),(69580,69600),(69623,69632),(69710,69714),(69750,69759),(69827,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69960,69968),(70007,70016),70112,(70133,70144),70162,(70210,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70656),70748,(70754,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71354,71360),(71370,71424),(71451,71453),(71468,71472),(71495,71680),(71740,71840),(71923,71935),(71943,71945),(71946,71948),71956,71959,71990,(71993,71995),(72007,72016),(72026,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72368),(72441,72448),(72458,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73472),73489,(73531,73534),(73562,73648),(73649,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77712),(77811,77824),(78934,82944),(83527,92160),(92729,92736),92767,(92778,92782),92863,(92874,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94181,94192),(94194,94208),(100344,100352),(101590,101632),(101641,110576),110580,110588,110591,(110883,110898),(110899,110928),(110931,110933),(110934,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,118528),(118574,118576),(118599,118608),(118724,118784),(119030,119040),(119079,119081),(119275,119296),(119366,119488),(119508,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122624),(122655,122661),(122667,122880),122887,(122905,122907),122914,122917,(122923,122928),(122990,123023),(123024,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123536),(123567,123584),(123642,123647),(123648,124112),(124154,124896),124903,124908,124911,124927,(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127406,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128728,128732),(128749,128752),(128765,128768),(128887,128891),(128986,128992),(129004,129008),(129009,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129200),(129202,129280),(129620,129632),(129646,129648),(129661,129664),(129673,129680),129726,(129734,129742),(129756,129760),(129769,129776),(129785,129792),129939,(129995,130032),(130042,131072),(173792,173824),(177978,177984),(178206,178208),(183970,183984),(191457,191472),(192094,194560),(195102,196608),(201547,201552),(205744,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11360),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42945,42947,42952,42954,42961,42963,42965,42967,42969,42998,43002,(43824,43867),(43872,43881),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(68800,68851),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(122624,122634),(122635,122655),(122661,122667),(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2249,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(42994,42997),(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),43881,65392,(65438,65440),(67456,67462),(67463,67505),(67506,67515),(92992,92996),(94099,94112),(94176,94178),94179,(110576,110580),(110581,110588),(110589,110591),(122928,122990),(123191,123198),124139,125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2249),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6989),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12736),(12784,12800),(13312,19904),(19968,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(69248,69290),(69296,69298),(69376,69405),69415,(69424,69446),(69488,69506),(69552,69573),(69600,69623),(69635,69688),(69745,69747),69749,(69763,69808),(69840,69865),(69891,69927),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70751,70754),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71488,71495),(71680,71724),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),73474,(73476,73489),(73490,73524),73648,(73728,74650),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(82944,83527),(92160,92729),(92736,92767),(92784,92863),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93952,94027),94032,(94208,100344),(100352,101590),(101632,101641),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),122634,(123136,123181),123214,(123536,123566),(123584,123628),(124112,124139),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(191472,192094),(194560,195102),(196608,201547),(201552,205744)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11312),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42944,42946,(42948,42952),42953,42960,42966,42968,42997,(65313,65339),(66560,66600),(66736,66772),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(68736,68787),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),3315,(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),5909,5940,6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),70094,(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,(71456,71458),71462,(71724,71727),71736,(71984,71990),(71991,71993),71997,72000,72002,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),73475,(73524,73526),(73534,73536),73537,(94033,94088),(94192,94194),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2200,2208),(2250,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,(2901,2903),(2914,2916),2946,3008,3021,3072,3076,3132,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3457,3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3791),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5940),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),6159,(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6847,6863),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),43052,(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(69291,69293),(69373,69376),(69446,69457),(69506,69510),69633,(69688,69703),69744,(69747,69749),(69759,69762),(69811,69815),(69817,69819),69826,(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),70095,(70191,70194),70196,(70198,70200),70206,70209,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,(71453,71456),(71458,71462),(71463,71468),(71727,71736),(71737,71739),(71995,71997),71998,72003,(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(73472,73474),(73526,73531),73536,73538,78912,(78919,78934),(92912,92917),(92976,92983),94031,(94095,94099),94180,(113821,113823),(118528,118574),(118576,118599),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),123023,(123184,123191),123566,(123628,123632),(124140,124144),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71472,71482),(71904,71914),(72016,72026),(72784,72794),(73040,73050),(73120,73130),(73552,73562),(92768,92778),(92864,92874),(93008,93018),(120782,120832),(123200,123210),(123632,123642),(124144,124154),(125264,125274),(130032,130042)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69573,69580),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119488,119508),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,11869,12316,12336,12448,(65073,65075),65112,65123,65293,69293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,11862,11864,11866,11868,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1565,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(7002,7009),(7037,7039),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),(11858,11861),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69510,69514),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70731,70736),(70746,70748),70749,70854,(71105,71128),(71233,71236),(71264,71277),71353,(71484,71487),71739,(72004,72007),72162,(72255,72263),(72346,72349),(72350,72355),(72448,72458),(72769,72774),(72816,72818),(73463,73465),(73539,73552),73727,(74864,74869),(77809,77811),(92782,92784),92917,(92983,92988),92996,(93847,93851),94178,113823,(121479,121484),(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,11861,11863,11865,11867,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8385),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),2184,8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(43882,43884),(64434,64451),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9255),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11159,11264),(11493,11499),(11856,11858),(11904,11930),(11931,12020),(12032,12246),(12272,12288),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12772),12783,(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),(64832,64848),64975,(65021,65024),65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65949),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(118608,118724),(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119275),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127245,127406),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128728),(128732,128749),(128752,128765),(128768,128887),(128891,128986),(128992,129004),129008,(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129200,129202),(129280,129620),(129632,129646),(129648,129661),(129664,129673),(129680,129726),(129727,129734),(129742,129756),(129760,129769),(129776,129785),(129792,129939),(129940,129995)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2250),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3302,3312),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6989),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7305),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12736),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19904),(19968,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42955),(42960,42962),42963,(42965,42970),(42994,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43882),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(67072,67383),(67392,67414),(67424,67432),(67456,67462),(67463,67505),(67506,67515),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(69216,69247),(69248,69290),(69296,69298),(69376,69416),(69424,69446),(69457,69461),(69488,69506),(69552,69580),(69600,69623),(69635,69688),(69714,69744),(69745,69747),69749,(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70656,70709),(70727,70731),(70736,70746),(70751,70754),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71424,71451),(71472,71484),(71488,71495),(71680,71724),(71840,71923),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72016,72026),(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),73474,(73476,73489),(73490,73524),(73552,73562),73648,(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(82944,83527),(92160,92729),(92736,92767),(92768,92778),(92784,92863),(92864,92874),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101590),(101632,101641),(110576,110580),(110581,110588),(110589,110591),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(119488,119508),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(122624,122655),(122661,122667),(122928,122990),(123136,123181),(123191,123198),(123200,123210),123214,(123536,123566),(123584,123628),(123632,123642),(124112,124140),(124144,124154),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(130032,130042),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(191472,192094),(194560,195102),(196608,201547),(201552,205744))}",
    "16.0.0": "{'Cc':((0,32),(127,160)),'Cf':(173,(1536,1542),1564,1757,1807,(2192,2194),2274,6158,(8203,8208),(8234,8239),(8288,8293),(8294,8304),65279,(65529,65532),69821,69837,(78896,78912),(113824,113828),(119155,119163),917505,(917536,917632)),'Cn':((888,890),(896,900),907,909,930,1328,(1367,1369),(1419,1421),1424,(1480,1488),(1515,1519),(1525,1536),1806,(1867,1869),(1970,1984),(2043,2045),(2094,2096),2111,(2140,2142),2143,(2155,2160),2191,(2194,2199),2436,(2445,2447),(2449,2451),2473,2481,(2483,2486),(2490,2492),(2501,2503),(2505,2507),(2511,2519),(2520,2524),2526,(2532,2534),(2559,2561),2564,(2571,2575),(2577,2579),2601,2609,2612,2615,(2618,2620),2621,(2627,2631),(2633,2635),(2638,2641),(2642,2649),2653,(2655,2662),(2679,2689),2692,2702,2706,2729,2737,2740,(2746,2748),2758,2762,(2766,2768),(2769,2784),(2788,2790),(2802,2809),2816,2820,(2829,2831),(2833,2835),2857,2865,2868,(2874,2876),(2885,2887),(2889,2891),(2894,2901),(2904,2908),2910,(2916,2918),(2936,2946),2948,(2955,2958),2961,(2966,2969),2971,2973,(2976,2979),(2981,2984),(2987,2990),(3002,3006),(3011,3014),3017,(3022,3024),(3025,3031),(3032,3046),(3067,3072),3085,3089,3113,(3130,3132),3141,3145,(3150,3157),3159,(3163,3165),(3166,3168),(3172,3174),(3184,3191),3213,3217,3241,3252,(3258,3260),3269,3273,(3278,3285),(3287,3293),3295,(3300,3302),3312,(3316,3328),3341,3345,3397,3401,(3408,3412),(3428,3430),3456,3460,(3479,3482),3506,3516,(3518,3520),(3527,3530),(3531,3535),3541,3543,(3552,3558),(3568,3570),(3573,3585),(3643,3647),(3676,3713),3715,3717,3723,3748,3750,(3774,3776),3781,3783,3791,(3802,3804),(3808,3840),3912,(3949,3953),3992,4029,4045,(4059,4096),4294,(4296,4301),(4302,4304),4681,(4686,4688),4695,4697,(4702,4704),4745,(4750,4752),4785,(4790,4792),4799,4801,(4806,4808),4823,4881,(4886,4888),(4955,4957),(4989,4992),(5018,5024),(5110,5112),(5118,5120),(5789,5792),(5881,5888),(5910,5919),(5943,5952),(5972,5984),5997,6001,(6004,6016),(6110,6112),(6122,6128),(6138,6144),(6170,6176),(6265,6272),(6315,6320),(6390,6400),6431,(6444,6448),(6460,6464),(6465,6468),(6510,6512),(6517,6528),(6572,6576),(6602,6608),(6619,6622),(6684,6686),6751,(6781,6783),(6794,6800),(6810,6816),(6830,6832),(6863,6912),6989,(7156,7164),(7224,7227),(7242,7245),(7307,7312),(7355,7357),(7368,7376),(7419,7424),(7958,7960),(7966,7968),(8006,8008),(8014,8016),8024,8026,8028,8030,(8062,8064),8117,8133,(8148,8150),8156,(8176,8178),8181,8191,8293,(8306,8308),8335,(8349,8352),(8385,8400),(8433,8448),(8588,8592),(9258,9280),(9291,9312),(11124,11126),11158,(11508,11513),11558,(11560,11565),(11566,11568),(11624,11631),(11633,11647),(11671,11680),11687,11695,11703,11711,11719,11727,11735,11743,(11870,11904),11930,(12020,12032),(12246,12272),12352,(12439,12441),(12544,12549),12592,12687,(12774,12783),12831,(42125,42128),(42183,42192),(42540,42560),(42744,42752),(42958,42960),42962,42964,(42973,42994),(43053,43056),(43066,43072),(43128,43136),(43206,43214),(43226,43232),(43348,43359),(43389,43392),43470,(43482,43486),43519,(43575,43584),(43598,43600),(43610,43612),(43715,43739),(43767,43777),(43783,43785),(43791,43793),(43799,43808),43815,43823,(43884,43888),(44014,44016),(44026,44032),(55204,55216),(55239,55243),(55292,55296),(64110,64112),(64218,64256),(64263,64275),(64280,64285),64311,64317,64319,64322,64325,(64451,64467),(64912,64914),(64968,64975),(64976,65008),(65050,65056),65107,65127,(65132,65136),65141,(65277,65279),65280,(65471,65474),(65480,65482),(65488,65490),(65496,65498),(65501,65504),65511,(65519,65529),(65534,65536),65548,65575,65595,65598,(65614,65616),(65630,65664),(65787,65792),(65795,65799),(65844,65847),65935,(65949,65952),(65953,66000),(66046,66176),(66205,66208),(66257,66272),(66300,66304),(66340,66349),(66379,66384),(66427,66432),66462,(66500,66504),(66518,66560),(66718,66720),(66730,66736),(66772,66776),(66812,66816),(66856,66864),(66916,66927),66939,66955,66963,66966,66978,66994,67002,(67005,67008),(67060,67072),(67383,67392),(67414,67424),(67432,67456),67462,67505,(67515,67584),(67590,67592),67593,67638,(67641,67644),(67645,67647),67670,(67743,67751),(67760,67808),67827,(67830,67835),(67868,67871),(67898,67903),(67904,67968),(68024,68028),(68048,68050),68100,(68103,68108),68116,68120,(68150,68152),(68155,68159),(68169,68176),(68185,68192),(68256,68288),(68327,68331),(68343,68352),(68406,68409),(68438,68440),(68467,68472),(68498,68505),(68509,68521),(68528,68608),(68681,68736),(68787,68800),(68851,68858),(68904,68912),(68922,68928),(68966,68969),(68998,69006),(69008,69216),69247,69290,(69294,69296),(69298,69314),(69317,69372),(69416,69424),(69466,69488),(69514,69552),(69580,69600),(69623,69632),(69710,69714),(69750,69759),(69827,69837),(69838,69840),(69865,69872),(69882,69888),69941,(69960,69968),(70007,70016),70112,(70133,70144),70162,(70210,70272),70279,70281,70286,70302,(70314,70320),(70379,70384),(70394,70400),70404,(70413,70415),(70417,70419),70441,70449,70452,70458,(70469,70471),(70473,70475),(70478,70480),(70481,70487),(70488,70493),(70500,70502),(70509,70512),(70517,70528),70538,(70540,70542),70543,70582,70593,(70595,70597),70598,70603,70614,(70617,70625),(70627,70656),70748,(70754,70784),(70856,70864),(70874,71040),(71094,71096),(71134,71168),(71237,71248),(71258,71264),(71277,71296),(71354,71360),(71370,71376),(71396,71424),(71451,71453),(71468,71472),(71495,71680),(71740,71840),(71923,71935),(71943,71945),(71946,71948),71956,71959,71990,(71993,71995),(72007,72016),(72026,72096),(72104,72106),(72152,72154),(72165,72192),(72264,72272),(72355,72368),(72441,72448),(72458,72640),(72674,72688),(72698,72704),72713,72759,(72774,72784),(72813,72816),(72848,72850),72872,(72887,72960),72967,72970,(73015,73018),73019,73022,(73032,73040),(73050,73056),73062,73065,73103,73106,(73113,73120),(73130,73440),(73465,73472),73489,(73531,73534),(73563,73648),(73649,73664),(73714,73727),(74650,74752),74863,(74869,74880),(75076,77712),(77811,77824),(78934,78944),(82939,82944),(83527,90368),(90426,92160),(92729,92736),92767,(92778,92782),92863,(92874,92880),(92910,92912),(92918,92928),(92998,93008),93018,93026,(93048,93053),(93072,93504),(93562,93760),(93851,93952),(94027,94031),(94088,94095),(94112,94176),(94181,94192),(94194,94208),(100344,100352),(101590,101631),(101641,110576),110580,110588,110591,(110883,110898),(110899,110928),(110931,110933),(110934,110948),(110952,110960),(111356,113664),(113771,113776),(113789,113792),(113801,113808),(113818,113820),(113828,117760),(118010,118016),(118452,118528),(118574,118576),(118599,118608),(118724,118784),(119030,119040),(119079,119081),(119275,119296),(119366,119488),(119508,119520),(119540,119552),(119639,119648),(119673,119808),119893,119965,(119968,119970),(119971,119973),(119975,119977),119981,119994,119996,120004,120070,(120075,120077),120085,120093,120122,120127,120133,(120135,120138),120145,(120486,120488),(120780,120782),(121484,121499),121504,(121520,122624),(122655,122661),(122667,122880),122887,(122905,122907),122914,122917,(122923,122928),(122990,123023),(123024,123136),(123181,123184),(123198,123200),(123210,123214),(123216,123536),(123567,123584),(123642,123647),(123648,124112),(124154,124368),(124411,124415),(124416,124896),124903,124908,124911,124927,(125125,125127),(125143,125184),(125260,125264),(125274,125278),(125280,126065),(126133,126209),(126270,126464),126468,126496,126499,(126501,126503),126504,126515,126520,126522,(126524,126530),(126531,126535),126536,126538,126540,126544,126547,(126549,126551),126552,126554,126556,126558,126560,126563,(126565,126567),126571,126579,126584,126589,126591,126602,(126620,126625),126628,126634,(126652,126704),(126706,126976),(127020,127024),(127124,127136),(127151,127153),127168,127184,(127222,127232),(127406,127462),(127491,127504),(127548,127552),(127561,127568),(127570,127584),(127590,127744),(128728,128732),(128749,128752),(128765,128768),(128887,128891),(128986,128992),(129004,129008),(129009,129024),(129036,129040),(129096,129104),(129114,129120),(129160,129168),(129198,129200),(129212,129216),(129218,129280),(129620,129632),(129646,129648),(129661,129664),(129674,129679),(129735,129742),(129757,129759),(129770,129776),(129785,129792),129939,(130042,131072),(173792,173824),(177978,177984),(178206,178208),(183970,183984),(191457,191472),(192094,194560),(195102,196608),(201547,201552),(205744,917505),(917506,917536),(917632,917760),(918000,983040),(1048574,1048576),(1114110,1114112)),'Co':((57344,63744),(983040,1048574),(1048576,1114110)),'Cs':((55296,57344),),'Ll':((97,123),181,(223,247),(248,256),257,259,261,263,265,267,269,271,273,275,277,279,281,283,285,287,289,291,293,295,297,299,301,303,305,307,309,(311,313),314,316,318,320,322,324,326,(328,330),331,333,335,337,339,341,343,345,347,349,351,353,355,357,359,361,363,365,367,369,371,373,375,378,380,(382,385),387,389,392,(396,398),402,405,(409,412),414,417,419,421,424,(426,428),429,432,436,438,(441,443),(445,448),454,457,460,462,464,466,468,470,472,474,(476,478),479,481,483,485,487,489,491,493,(495,497),499,501,505,507,509,511,513,515,517,519,521,523,525,527,529,531,533,535,537,539,541,543,545,547,549,551,553,555,557,559,561,(563,570),572,(575,577),578,583,585,587,589,(591,660),(661,688),881,883,887,(891,894),912,(940,975),(976,978),(981,984),985,987,989,991,993,995,997,999,1001,1003,1005,(1007,1012),1013,1016,(1019,1021),(1072,1120),1121,1123,1125,1127,1129,1131,1133,1135,1137,1139,1141,1143,1145,1147,1149,1151,1153,1163,1165,1167,1169,1171,1173,1175,1177,1179,1181,1183,1185,1187,1189,1191,1193,1195,1197,1199,1201,1203,1205,1207,1209,1211,1213,1215,1218,1220,1222,1224,1226,1228,(1230,1232),1233,1235,1237,1239,1241,1243,1245,1247,1249,1251,1253,1255,1257,1259,1261,1263,1265,1267,1269,1271,1273,1275,1277,1279,1281,1283,1285,1287,1289,1291,1293,1295,1297,1299,1301,1303,1305,1307,1309,1311,1313,1315,1317,1319,1321,1323,1325,1327,(1376,1417),(4304,4347),(4349,4352),(5112,5118),(7296,7305),7306,(7424,7468),(7531,7544),(7545,7579),7681,7683,7685,7687,7689,7691,7693,7695,7697,7699,7701,7703,7705,7707,7709,7711,7713,7715,7717,7719,7721,7723,7725,7727,7729,7731,7733,7735,7737,7739,7741,7743,7745,7747,7749,7751,7753,7755,7757,7759,7761,7763,7765,7767,7769,7771,7773,7775,7777,7779,7781,7783,7785,7787,7789,7791,7793,7795,7797,7799,7801,7803,7805,7807,7809,7811,7813,7815,7817,7819,7821,7823,7825,7827,(7829,7838),7839,7841,7843,7845,7847,7849,7851,7853,7855,7857,7859,7861,7863,7865,7867,7869,7871,7873,7875,7877,7879,7881,7883,7885,7887,7889,7891,7893,7895,7897,7899,7901,7903,7905,7907,7909,7911,7913,7915,7917,7919,7921,7923,7925,7927,7929,7931,7933,(7935,7944),(7952,7958),(7968,7976),(7984,7992),(8000,8006),(8016,8024),(8032,8040),(8048,8062),(8064,8072),(8080,8088),(8096,8104),(8112,8117),(8118,8120),8126,(8130,8133),(8134,8136),(8144,8148),(8150,8152),(8160,8168),(8178,8181),(8182,8184),8458,(8462,8464),8467,8495,8500,8505,(8508,8510),(8518,8522),8526,8580,(11312,11360),11361,(11365,11367),11368,11370,11372,11377,(11379,11381),(11382,11388),11393,11395,11397,11399,11401,11403,11405,11407,11409,11411,11413,11415,11417,11419,11421,11423,11425,11427,11429,11431,11433,11435,11437,11439,11441,11443,11445,11447,11449,11451,11453,11455,11457,11459,11461,11463,11465,11467,11469,11471,11473,11475,11477,11479,11481,11483,11485,11487,11489,(11491,11493),11500,11502,11507,(11520,11558),11559,11565,42561,42563,42565,42567,42569,42571,42573,42575,42577,42579,42581,42583,42585,42587,42589,42591,42593,42595,42597,42599,42601,42603,42605,42625,42627,42629,42631,42633,42635,42637,42639,42641,42643,42645,42647,42649,42651,42787,42789,42791,42793,42795,42797,(42799,42802),42803,42805,42807,42809,42811,42813,42815,42817,42819,42821,42823,42825,42827,42829,42831,42833,42835,42837,42839,42841,42843,42845,42847,42849,42851,42853,42855,42857,42859,42861,42863,(42865,42873),42874,42876,42879,42881,42883,42885,42887,42892,42894,42897,(42899,42902),42903,42905,42907,42909,42911,42913,42915,42917,42919,42921,42927,42933,42935,42937,42939,42941,42943,42945,42947,42952,42954,42957,42961,42963,42965,42967,42969,42971,42998,43002,(43824,43867),(43872,43881),(43888,43968),(64256,64263),(64275,64280),(65345,65371),(66600,66640),(66776,66812),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(68800,68851),(68976,68998),(71872,71904),(93792,93824),(119834,119860),(119886,119893),(119894,119912),(119938,119964),(119990,119994),119995,(119997,120004),(120005,120016),(120042,120068),(120094,120120),(120146,120172),(120198,120224),(120250,120276),(120302,120328),(120354,120380),(120406,120432),(120458,120486),(120514,120539),(120540,120546),(120572,120597),(120598,120604),(120630,120655),(120656,120662),(120688,120713),(120714,120720),(120746,120771),(120772,120778),120779,(122624,122634),(122635,122655),(122661,122667),(125218,125252)),'Lm':((688,706),(710,722),(736,741),748,750,884,890,1369,1600,(1765,1767),(2036,2038),2042,2074,2084,2088,2249,2417,3654,3782,4348,6103,6211,6823,(7288,7294),(7468,7531),7544,(7579,7616),8305,8319,(8336,8349),(11388,11390),11631,11823,12293,(12337,12342),12347,(12445,12447),(12540,12543),40981,(42232,42238),42508,42623,(42652,42654),(42775,42784),42864,42888,(42994,42997),(43000,43002),43471,43494,43632,43741,(43763,43765),(43868,43872),43881,65392,(65438,65440),(67456,67462),(67463,67505),(67506,67515),68942,68975,(92992,92996),(93504,93507),(93547,93549),(94099,94112),(94176,94178),94179,(110576,110580),(110581,110588),(110589,110591),(122928,122990),(123191,123198),124139,125259),'Lo':(170,186,443,(448,452),660,(1488,1515),(1519,1523),(1568,1600),(1601,1611),(1646,1648),(1649,1748),1749,(1774,1776),(1786,1789),1791,1808,(1810,1840),(1869,1958),1969,(1994,2027),(2048,2070),(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2249),(2308,2362),2365,2384,(2392,2402),(2418,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2544,2546),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),2929,2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3423,3426),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3585,3633),(3634,3636),(3648,3654),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),(3804,3808),3840,(3904,3912),(3913,3949),(3976,3981),(4096,4139),4159,(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4352,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4992,5008),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5873,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6108,(6176,6211),(6212,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6480,6510),(6512,6517),(6528,6572),(6576,6602),(6656,6679),(6688,6741),(6917,6964),(6981,6989),(7043,7073),(7086,7088),(7098,7142),(7168,7204),(7245,7248),(7258,7288),(7401,7405),(7406,7412),(7413,7415),7418,(8501,8505),(11568,11624),(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),12294,12348,(12353,12439),12447,(12449,12539),12543,(12549,12592),(12593,12687),(12704,12736),(12784,12800),(13312,19904),(19968,40981),(40982,42125),(42192,42232),(42240,42508),(42512,42528),(42538,42540),42606,(42656,42726),42895,42999,(43003,43010),(43011,43014),(43015,43019),(43020,43043),(43072,43124),(43138,43188),(43250,43256),43259,(43261,43263),(43274,43302),(43312,43335),(43360,43389),(43396,43443),(43488,43493),(43495,43504),(43514,43519),(43520,43561),(43584,43587),(43588,43596),(43616,43632),(43633,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43741),(43744,43755),43762,(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43968,44003),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65382,65392),(65393,65438),(65440,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(66176,66205),(66208,66257),(66304,66336),(66349,66369),(66370,66378),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66640,66718),(66816,66856),(66864,66916),(67008,67060),(67072,67383),(67392,67414),(67424,67432),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67680,67703),(67712,67743),(67808,67827),(67828,67830),(67840,67862),(67872,67898),(67968,68024),(68030,68032),68096,(68112,68116),(68117,68120),(68121,68150),(68192,68221),(68224,68253),(68288,68296),(68297,68325),(68352,68406),(68416,68438),(68448,68467),(68480,68498),(68608,68681),(68864,68900),(68938,68942),68943,(69248,69290),(69296,69298),(69314,69317),(69376,69405),69415,(69424,69446),(69488,69506),(69552,69573),(69600,69623),(69635,69688),(69745,69747),69749,(69763,69808),(69840,69865),(69891,69927),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),70106,70108,(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70528,70538),70539,70542,(70544,70582),70583,70609,70611,(70656,70709),(70727,70731),(70751,70754),(70784,70832),(70852,70854),70855,(71040,71087),(71128,71132),(71168,71216),71236,(71296,71339),71352,(71424,71451),(71488,71495),(71680,71724),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72640,72673),(72704,72713),(72714,72751),72768,(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73056,73062),(73063,73065),(73066,73098),73112,(73440,73459),73474,(73476,73489),(73490,73524),73648,(73728,74650),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(78944,82939),(82944,83527),(90368,90398),(92160,92729),(92736,92767),(92784,92863),(92880,92910),(92928,92976),(93027,93048),(93053,93072),(93507,93547),(93952,94027),94032,(94208,100344),(100352,101590),(101631,101641),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),122634,(123136,123181),123214,(123536,123566),(123584,123628),(124112,124139),(124368,124398),124400,(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(191472,192094),(194560,195102),(196608,201547),(201552,205744)),'Lt':(453,456,459,498,(8072,8080),(8088,8096),(8104,8112),8124,8140,8188),'Lu':((65,91),(192,215),(216,223),256,258,260,262,264,266,268,270,272,274,276,278,280,282,284,286,288,290,292,294,296,298,300,302,304,306,308,310,313,315,317,319,321,323,325,327,330,332,334,336,338,340,342,344,346,348,350,352,354,356,358,360,362,364,366,368,370,372,374,(376,378),379,381,(385,387),388,(390,392),(393,396),(398,402),(403,405),(406,409),(412,414),(415,417),418,420,(422,424),425,428,(430,432),(433,436),437,(439,441),444,452,455,458,461,463,465,467,469,471,473,475,478,480,482,484,486,488,490,492,494,497,500,(502,505),506,508,510,512,514,516,518,520,522,524,526,528,530,532,534,536,538,540,542,544,546,548,550,552,554,556,558,560,562,(570,572),(573,575),577,(579,583),584,586,588,590,880,882,886,895,902,(904,907),908,(910,912),(913,930),(931,940),975,(978,981),984,986,988,990,992,994,996,998,1000,1002,1004,1006,1012,1015,(1017,1019),(1021,1072),1120,1122,1124,1126,1128,1130,1132,1134,1136,1138,1140,1142,1144,1146,1148,1150,1152,1162,1164,1166,1168,1170,1172,1174,1176,1178,1180,1182,1184,1186,1188,1190,1192,1194,1196,1198,1200,1202,1204,1206,1208,1210,1212,1214,(1216,1218),1219,1221,1223,1225,1227,1229,1232,1234,1236,1238,1240,1242,1244,1246,1248,1250,1252,1254,1256,1258,1260,1262,1264,1266,1268,1270,1272,1274,1276,1278,1280,1282,1284,1286,1288,1290,1292,1294,1296,1298,1300,1302,1304,1306,1308,1310,1312,1314,1316,1318,1320,1322,1324,1326,(1329,1367),(4256,4294),4295,4301,(5024,5110),7305,(7312,7355),(7357,7360),7680,7682,7684,7686,7688,7690,7692,7694,7696,7698,7700,7702,7704,7706,7708,7710,7712,7714,7716,7718,7720,7722,7724,7726,7728,7730,7732,7734,7736,7738,7740,7742,7744,7746,7748,7750,7752,7754,7756,7758,7760,7762,7764,7766,7768,7770,7772,7774,7776,7778,7780,7782,7784,7786,7788,7790,7792,7794,7796,7798,7800,7802,7804,7806,7808,7810,7812,7814,7816,7818,7820,7822,7824,7826,7828,7838,7840,7842,7844,7846,7848,7850,7852,7854,7856,7858,7860,7862,7864,7866,7868,7870,7872,7874,7876,7878,7880,7882,7884,7886,7888,7890,7892,7894,7896,7898,7900,7902,7904,7906,7908,7910,7912,7914,7916,7918,7920,7922,7924,7926,7928,7930,7932,7934,(7944,7952),(7960,7966),(7976,7984),(7992,8000),(8008,8014),8025,8027,8029,8031,(8040,8048),(8120,8124),(8136,8140),(8152,8156),(8168,8173),(8184,8188),8450,8455,(8459,8462),(8464,8467),8469,(8473,8478),8484,8486,8488,(8490,8494),(8496,8500),(8510,8512),8517,8579,(11264,11312),11360,(11362,11365),11367,11369,11371,(11373,11377),11378,11381,(11390,11393),11394,11396,11398,11400,11402,11404,11406,11408,11410,11412,11414,11416,11418,11420,11422,11424,11426,11428,11430,11432,11434,11436,11438,11440,11442,11444,11446,11448,11450,11452,11454,11456,11458,11460,11462,11464,11466,11468,11470,11472,11474,11476,11478,11480,11482,11484,11486,11488,11490,11499,11501,11506,42560,42562,42564,42566,42568,42570,42572,42574,42576,42578,42580,42582,42584,42586,42588,42590,42592,42594,42596,42598,42600,42602,42604,42624,42626,42628,42630,42632,42634,42636,42638,42640,42642,42644,42646,42648,42650,42786,42788,42790,42792,42794,42796,42798,42802,42804,42806,42808,42810,42812,42814,42816,42818,42820,42822,42824,42826,42828,42830,42832,42834,42836,42838,42840,42842,42844,42846,42848,42850,42852,42854,42856,42858,42860,42862,42873,42875,(42877,42879),42880,42882,42884,42886,42891,42893,42896,42898,42902,42904,42906,42908,42910,42912,42914,42916,42918,42920,(42922,42927),(42928,42933),42934,42936,42938,42940,42942,42944,42946,(42948,42952),42953,(42955,42957),42960,42966,42968,42970,42972,42997,(65313,65339),(66560,66600),(66736,66772),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(68736,68787),(68944,68966),(71840,71872),(93760,93792),(119808,119834),(119860,119886),(119912,119938),119964,(119966,119968),119970,(119973,119975),(119977,119981),(119982,119990),(120016,120042),(120068,120070),(120071,120075),(120077,120085),(120086,120093),(120120,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120172,120198),(120224,120250),(120276,120302),(120328,120354),(120380,120406),(120432,120458),(120488,120513),(120546,120571),(120604,120629),(120662,120687),(120720,120745),120778,(125184,125218)),'Mc':(2307,2363,(2366,2369),(2377,2381),(2382,2384),(2434,2436),(2494,2497),(2503,2505),(2507,2509),2519,2563,(2622,2625),2691,(2750,2753),2761,(2763,2765),(2818,2820),2878,2880,(2887,2889),(2891,2893),2903,(3006,3008),(3009,3011),(3014,3017),(3018,3021),3031,(3073,3076),(3137,3141),(3202,3204),3262,(3264,3269),(3271,3273),(3274,3276),(3285,3287),3315,(3330,3332),(3390,3393),(3398,3401),(3402,3405),3415,(3458,3460),(3535,3538),(3544,3552),(3570,3572),(3902,3904),3967,(4139,4141),4145,4152,(4155,4157),(4182,4184),(4194,4197),(4199,4206),(4227,4229),(4231,4237),4239,(4250,4253),5909,5940,6070,(6078,6086),(6087,6089),(6435,6439),(6441,6444),(6448,6450),(6451,6457),(6681,6683),6741,6743,6753,(6755,6757),(6765,6771),6916,6965,6971,(6973,6978),(6979,6981),7042,7073,(7078,7080),7082,7143,(7146,7149),7150,(7154,7156),(7204,7212),(7220,7222),7393,7415,(12334,12336),(43043,43045),43047,(43136,43138),(43188,43204),(43346,43348),43395,(43444,43446),(43450,43452),(43454,43457),(43567,43569),(43571,43573),43597,43643,43645,43755,(43758,43760),43765,(44003,44005),(44006,44008),(44009,44011),44012,69632,69634,69762,(69808,69811),(69815,69817),69932,(69957,69959),70018,(70067,70070),(70079,70081),70094,(70188,70191),(70194,70196),70197,(70368,70371),(70402,70404),(70462,70464),(70465,70469),(70471,70473),(70475,70478),70487,(70498,70500),(70584,70587),70594,70597,(70599,70603),(70604,70606),70607,(70709,70712),(70720,70722),70725,(70832,70835),70841,(70843,70847),70849,(71087,71090),(71096,71100),71102,(71216,71219),(71227,71229),71230,71340,(71342,71344),71350,71454,(71456,71458),71462,(71724,71727),71736,(71984,71990),(71991,71993),71997,72000,72002,(72145,72148),(72156,72160),72164,72249,(72279,72281),72343,72751,72766,72873,72881,72884,(73098,73103),(73107,73109),73110,(73461,73463),73475,(73524,73526),(73534,73536),73537,(90410,90413),(94033,94088),(94192,94194),(119141,119143),(119149,119155)),'Me':((1160,1162),6846,(8413,8417),(8418,8421),(42608,42611)),'Mn':((768,880),(1155,1160),(1425,1470),1471,(1473,1475),(1476,1478),1479,(1552,1563),(1611,1632),1648,(1750,1757),(1759,1765),(1767,1769),(1770,1774),1809,(1840,1867),(1958,1969),(2027,2036),2045,(2070,2074),(2075,2084),(2085,2088),(2089,2094),(2137,2140),(2199,2208),(2250,2274),(2275,2307),2362,2364,(2369,2377),2381,(2385,2392),(2402,2404),2433,2492,(2497,2501),2509,(2530,2532),2558,(2561,2563),2620,(2625,2627),(2631,2633),(2635,2638),2641,(2672,2674),2677,(2689,2691),2748,(2753,2758),(2759,2761),2765,(2786,2788),(2810,2816),2817,2876,2879,(2881,2885),2893,(2901,2903),(2914,2916),2946,3008,3021,3072,3076,3132,(3134,3137),(3142,3145),(3146,3150),(3157,3159),(3170,3172),3201,3260,3263,3270,(3276,3278),(3298,3300),(3328,3330),(3387,3389),(3393,3397),3405,(3426,3428),3457,3530,(3538,3541),3542,3633,(3636,3643),(3655,3663),3761,(3764,3773),(3784,3791),(3864,3866),3893,3895,3897,(3953,3967),(3968,3973),(3974,3976),(3981,3992),(3993,4029),4038,(4141,4145),(4146,4152),(4153,4155),(4157,4159),(4184,4186),(4190,4193),(4209,4213),4226,(4229,4231),4237,4253,(4957,4960),(5906,5909),(5938,5940),(5970,5972),(6002,6004),(6068,6070),(6071,6078),6086,(6089,6100),6109,(6155,6158),6159,(6277,6279),6313,(6432,6435),(6439,6441),6450,(6457,6460),(6679,6681),6683,6742,(6744,6751),6752,6754,(6757,6765),(6771,6781),6783,(6832,6846),(6847,6863),(6912,6916),6964,(6966,6971),6972,6978,(7019,7028),(7040,7042),(7074,7078),(7080,7082),(7083,7086),7142,(7144,7146),7149,(7151,7154),(7212,7220),(7222,7224),(7376,7379),(7380,7393),(7394,7401),7405,7412,(7416,7418),(7616,7680),(8400,8413),8417,(8421,8433),(11503,11506),11647,(11744,11776),(12330,12334),(12441,12443),42607,(42612,42622),(42654,42656),(42736,42738),43010,43014,43019,(43045,43047),43052,(43204,43206),(43232,43250),43263,(43302,43310),(43335,43346),(43392,43395),43443,(43446,43450),(43452,43454),43493,(43561,43567),(43569,43571),(43573,43575),43587,43596,43644,43696,(43698,43701),(43703,43705),(43710,43712),43713,(43756,43758),43766,44005,44008,44013,64286,(65024,65040),(65056,65072),66045,66272,(66422,66427),(68097,68100),(68101,68103),(68108,68112),(68152,68155),68159,(68325,68327),(68900,68904),(68969,68974),(69291,69293),(69372,69376),(69446,69457),(69506,69510),69633,(69688,69703),69744,(69747,69749),(69759,69762),(69811,69815),(69817,69819),69826,(69888,69891),(69927,69932),(69933,69941),70003,(70016,70018),(70070,70079),(70089,70093),70095,(70191,70194),70196,(70198,70200),70206,70209,70367,(70371,70379),(70400,70402),(70459,70461),70464,(70502,70509),(70512,70517),(70587,70593),70606,70608,70610,(70625,70627),(70712,70720),(70722,70725),70726,70750,(70835,70841),70842,(70847,70849),(70850,70852),(71090,71094),(71100,71102),(71103,71105),(71132,71134),(71219,71227),71229,(71231,71233),71339,71341,(71344,71350),71351,71453,71455,(71458,71462),(71463,71468),(71727,71736),(71737,71739),(71995,71997),71998,72003,(72148,72152),(72154,72156),72160,(72193,72203),(72243,72249),(72251,72255),72263,(72273,72279),(72281,72284),(72330,72343),(72344,72346),(72752,72759),(72760,72766),72767,(72850,72872),(72874,72881),(72882,72884),(72885,72887),(73009,73015),73018,(73020,73022),(73023,73030),73031,(73104,73106),73109,73111,(73459,73461),(73472,73474),(73526,73531),73536,73538,73562,78912,(78919,78934),(90398,90410),(90413,90416),(92912,92917),(92976,92983),94031,(94095,94099),94180,(113821,113823),(118528,118574),(118576,118599),(119143,119146),(119163,119171),(119173,119180),(119210,119214),(119362,119365),(121344,121399),(121403,121453),121461,121476,(121499,121504),(121505,121520),(122880,122887),(122888,122905),(122907,122914),(122915,122917),(122918,122923),123023,(123184,123191),123566,(123628,123632),(124140,124144),(124398,124400),(125136,125143),(125252,125259),(917760,918000)),'Nd':((48,58),(1632,1642),(1776,1786),(1984,1994),(2406,2416),(2534,2544),(2662,2672),(2790,2800),(2918,2928),(3046,3056),(3174,3184),(3302,3312),(3430,3440),(3558,3568),(3664,3674),(3792,3802),(3872,3882),(4160,4170),(4240,4250),(6112,6122),(6160,6170),(6470,6480),(6608,6618),(6784,6794),(6800,6810),(6992,7002),(7088,7098),(7232,7242),(7248,7258),(42528,42538),(43216,43226),(43264,43274),(43472,43482),(43504,43514),(43600,43610),(44016,44026),(65296,65306),(66720,66730),(68912,68922),(68928,68938),(69734,69744),(69872,69882),(69942,69952),(70096,70106),(70384,70394),(70736,70746),(70864,70874),(71248,71258),(71360,71370),(71376,71396),(71472,71482),(71904,71914),(72016,72026),(72688,72698),(72784,72794),(73040,73050),(73120,73130),(73552,73562),(90416,90426),(92768,92778),(92864,92874),(93008,93018),(93552,93562),(118000,118010),(120782,120832),(123200,123210),(123632,123642),(124144,124154),(124401,124411),(125264,125274),(130032,130042)),'Nl':((5870,5873),(8544,8579),(8581,8585),12295,(12321,12330),(12344,12347),(42726,42736),(65856,65909),66369,66378,(66513,66518),(74752,74863)),'No':((178,180),185,(188,191),(2548,2554),(2930,2936),(3056,3059),(3192,3199),(3416,3423),(3440,3449),(3882,3892),(4969,4989),(6128,6138),6618,8304,(8308,8314),(8320,8330),(8528,8544),8585,(9312,9372),(9450,9472),(10102,10132),11517,(12690,12694),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(43056,43062),(65799,65844),(65909,65913),(65930,65932),(66273,66300),(66336,66340),(67672,67680),(67705,67712),(67751,67760),(67835,67840),(67862,67868),(68028,68030),(68032,68048),(68050,68096),(68160,68169),(68221,68223),(68253,68256),(68331,68336),(68440,68448),(68472,68480),(68521,68528),(68858,68864),(69216,69247),(69405,69415),(69457,69461),(69573,69580),(69714,69734),(70113,70133),(71482,71484),(71914,71923),(72794,72813),(73664,73685),(93019,93026),(93824,93847),(119488,119508),(119520,119540),(119648,119673),(125127,125136),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(127232,127245)),'Pc':(95,(8255,8257),8276,(65075,65077),(65101,65104),65343),'Pd':(45,1418,1470,5120,6150,(8208,8214),11799,11802,(11834,11836),11840,11869,12316,12336,12448,(65073,65075),65112,65123,65293,68974,69293),'Pe':(41,93,125,3899,3901,5788,8262,8318,8334,8969,8971,9002,10089,10091,10093,10095,10097,10099,10101,10182,10215,10217,10219,10221,10223,10628,10630,10632,10634,10636,10638,10640,10642,10644,10646,10648,10713,10715,10749,11811,11813,11815,11817,11862,11864,11866,11868,12297,12299,12301,12303,12305,12309,12311,12313,12315,(12318,12320),64830,65048,65078,65080,65082,65084,65086,65088,65090,65092,65096,65114,65116,65118,65289,65341,65373,65376,65379),'Pf':(187,8217,8221,8250,11779,11781,11786,11789,11805,11809),'Pi':(171,8216,(8219,8221),8223,8249,11778,11780,11785,11788,11804,11808),'Po':((33,36),(37,40),42,44,(46,48),(58,60),(63,65),92,161,167,(182,184),191,894,903,(1370,1376),1417,1472,1475,1478,(1523,1525),(1545,1547),(1548,1550),1563,(1565,1568),(1642,1646),1748,(1792,1806),(2039,2042),(2096,2111),2142,(2404,2406),2416,2557,2678,2800,3191,3204,3572,3663,(3674,3676),(3844,3859),3860,3973,(4048,4053),(4057,4059),(4170,4176),4347,(4960,4969),5742,(5867,5870),(5941,5943),(6100,6103),(6104,6107),(6144,6150),(6151,6155),(6468,6470),(6686,6688),(6816,6823),(6824,6830),(6990,6992),(7002,7009),(7037,7040),(7164,7168),(7227,7232),(7294,7296),(7360,7368),7379,(8214,8216),(8224,8232),(8240,8249),(8251,8255),(8257,8260),(8263,8274),8275,(8277,8287),(11513,11517),(11518,11520),11632,(11776,11778),(11782,11785),11787,(11790,11799),(11800,11802),11803,(11806,11808),(11818,11823),(11824,11834),(11836,11840),11841,(11843,11856),(11858,11861),(12289,12292),12349,12539,(42238,42240),(42509,42512),42611,42622,(42738,42744),(43124,43128),(43214,43216),(43256,43259),43260,(43310,43312),43359,(43457,43470),(43486,43488),(43612,43616),(43742,43744),(43760,43762),44011,(65040,65047),65049,65072,(65093,65095),(65097,65101),(65104,65107),(65108,65112),(65119,65122),65128,(65130,65132),(65281,65284),(65285,65288),65290,65292,(65294,65296),(65306,65308),(65311,65313),65340,65377,(65380,65382),(65792,65795),66463,66512,66927,67671,67871,67903,(68176,68185),68223,(68336,68343),(68409,68416),(68505,68509),(69461,69466),(69510,69514),(69703,69710),(69819,69821),(69822,69826),(69952,69956),(70004,70006),(70085,70089),70093,70107,(70109,70112),(70200,70206),70313,(70612,70614),(70615,70617),(70731,70736),(70746,70748),70749,70854,(71105,71128),(71233,71236),(71264,71277),71353,(71484,71487),71739,(72004,72007),72162,(72255,72263),(72346,72349),(72350,72355),(72448,72458),72673,(72769,72774),(72816,72818),(73463,73465),(73539,73552),73727,(74864,74869),(77809,77811),(92782,92784),92917,(92983,92988),92996,(93549,93552),(93847,93851),94178,113823,(121479,121484),124415,(125278,125280)),'Ps':(40,91,123,3898,3900,5787,8218,8222,8261,8317,8333,8968,8970,9001,10088,10090,10092,10094,10096,10098,10100,10181,10214,10216,10218,10220,10222,10627,10629,10631,10633,10635,10637,10639,10641,10643,10645,10647,10712,10714,10748,11810,11812,11814,11816,11842,11861,11863,11865,11867,12296,12298,12300,12302,12304,12308,12310,12312,12314,12317,64831,65047,65077,65079,65081,65083,65085,65087,65089,65091,65095,65113,65115,65117,65288,65339,65371,65375,65378),'Sc':(36,(162,166),1423,1547,(2046,2048),(2546,2548),2555,2801,3065,3647,6107,(8352,8385),43064,65020,65129,65284,(65504,65506),(65509,65511),(73693,73697),123647,126128),'Sk':(94,96,168,175,180,184,(706,710),(722,736),(741,748),749,(751,768),885,(900,902),2184,8125,(8127,8130),(8141,8144),(8157,8160),(8173,8176),(8189,8191),(12443,12445),(42752,42775),(42784,42786),(42889,42891),43867,(43882,43884),(64434,64451),65342,65344,65507,(127995,128000)),'Sm':(43,(60,63),124,126,172,177,215,247,1014,(1542,1545),8260,8274,(8314,8317),(8330,8333),8472,(8512,8517),8523,(8592,8597),(8602,8604),8608,8611,8614,8622,(8654,8656),8658,8660,(8692,8960),(8992,8994),9084,(9115,9140),(9180,9186),9655,9665,(9720,9728),9839,(10176,10181),(10183,10214),(10224,10240),(10496,10627),(10649,10712),(10716,10748),(10750,11008),(11056,11077),(11079,11085),64297,65122,(65124,65127),65291,(65308,65311),65372,65374,65506,(65513,65517),(69006,69008),120513,120539,120571,120597,120629,120655,120687,120713,120745,120771,(126704,126706)),'So':(166,169,174,176,1154,(1421,1423),(1550,1552),1758,1769,(1789,1791),2038,2554,2928,(3059,3065),3066,3199,3407,3449,(3841,3844),3859,(3861,3864),(3866,3872),3892,3894,3896,(4030,4038),(4039,4045),(4046,4048),(4053,4057),(4254,4256),(5008,5018),5741,6464,(6622,6656),(7009,7019),(7028,7037),(8448,8450),(8451,8455),(8456,8458),8468,(8470,8472),(8478,8484),8485,8487,8489,8494,(8506,8508),8522,(8524,8526),8527,(8586,8588),(8597,8602),(8604,8608),(8609,8611),(8612,8614),(8615,8622),(8623,8654),(8656,8658),8659,(8661,8692),(8960,8968),(8972,8992),(8994,9001),(9003,9084),(9085,9115),(9140,9180),(9186,9258),(9280,9291),(9372,9450),(9472,9655),(9656,9665),(9666,9720),(9728,9839),(9840,10088),(10132,10176),(10240,10496),(11008,11056),(11077,11079),(11085,11124),(11126,11158),(11159,11264),(11493,11499),(11856,11858),(11904,11930),(11931,12020),(12032,12246),(12272,12288),12292,(12306,12308),12320,(12342,12344),(12350,12352),(12688,12690),(12694,12704),(12736,12774),12783,(12800,12831),(12842,12872),12880,(12896,12928),(12938,12977),(12992,13312),(19904,19968),(42128,42183),(43048,43052),(43062,43064),43065,(43639,43642),(64832,64848),64975,(65021,65024),65508,65512,(65517,65519),(65532,65534),(65847,65856),(65913,65930),(65932,65935),(65936,65949),65952,(66000,66045),(67703,67705),68296,71487,(73685,73693),(73697,73714),(92988,92992),92997,113820,(117760,118000),(118016,118452),(118608,118724),(118784,119030),(119040,119079),(119081,119141),(119146,119149),(119171,119173),(119180,119210),(119214,119275),(119296,119362),119365,(119552,119639),(120832,121344),(121399,121403),(121453,121461),(121462,121476),(121477,121479),123215,126124,126254,(126976,127020),(127024,127124),(127136,127151),(127153,127168),(127169,127184),(127185,127222),(127245,127406),(127462,127491),(127504,127548),(127552,127561),(127568,127570),(127584,127590),(127744,127995),(128000,128728),(128732,128749),(128752,128765),(128768,128887),(128891,128986),(128992,129004),129008,(129024,129036),(129040,129096),(129104,129114),(129120,129160),(129168,129198),(129200,129212),(129216,129218),(129280,129620),(129632,129646),(129648,129661),(129664,129674),(129679,129735),(129742,129757),(129759,129770),(129776,129785),(129792,129939),(129940,130032)),'Zl':(8232,),'Zp':(8233,),'Zs':(32,160,5760,(8192,8203),8239,8287,12288),'word':((48,58),(65,91),95,(97,123),170,(178,180),181,(185,187),(188,191),(192,215),(216,247),(248,706),(710,722),(736,741),748,750,(880,885),(886,888),(890,894),895,902,(904,907),908,(910,930),(931,1014),(1015,1154),(1162,1328),(1329,1367),1369,(1376,1417),(1488,1515),(1519,1523),(1568,1611),(1632,1642),(1646,1648),(1649,1748),1749,(1765,1767),(1774,1789),1791,1808,(1810,1840),(1869,1958),1969,(1984,2027),(2036,2038),2042,(2048,2070),2074,2084,2088,(2112,2137),(2144,2155),(2160,2184),(2185,2191),(2208,2250),(2308,2362),2365,2384,(2392,2402),(2406,2416),(2417,2433),(2437,2445),(2447,2449),(2451,2473),(2474,2481),2482,(2486,2490),2493,2510,(2524,2526),(2527,2530),(2534,2546),(2548,2554),2556,(2565,2571),(2575,2577),(2579,2601),(2602,2609),(2610,2612),(2613,2615),(2616,2618),(2649,2653),2654,(2662,2672),(2674,2677),(2693,2702),(2703,2706),(2707,2729),(2730,2737),(2738,2740),(2741,2746),2749,2768,(2784,2786),(2790,2800),2809,(2821,2829),(2831,2833),(2835,2857),(2858,2865),(2866,2868),(2869,2874),2877,(2908,2910),(2911,2914),(2918,2928),(2929,2936),2947,(2949,2955),(2958,2961),(2962,2966),(2969,2971),2972,(2974,2976),(2979,2981),(2984,2987),(2990,3002),3024,(3046,3059),(3077,3085),(3086,3089),(3090,3113),(3114,3130),3133,(3160,3163),3165,(3168,3170),(3174,3184),(3192,3199),3200,(3205,3213),(3214,3217),(3218,3241),(3242,3252),(3253,3258),3261,(3293,3295),(3296,3298),(3302,3312),(3313,3315),(3332,3341),(3342,3345),(3346,3387),3389,3406,(3412,3415),(3416,3426),(3430,3449),(3450,3456),(3461,3479),(3482,3506),(3507,3516),3517,(3520,3527),(3558,3568),(3585,3633),(3634,3636),(3648,3655),(3664,3674),(3713,3715),3716,(3718,3723),(3724,3748),3749,(3751,3761),(3762,3764),3773,(3776,3781),3782,(3792,3802),(3804,3808),3840,(3872,3892),(3904,3912),(3913,3949),(3976,3981),(4096,4139),(4159,4170),(4176,4182),(4186,4190),4193,(4197,4199),(4206,4209),(4213,4226),4238,(4240,4250),(4256,4294),4295,4301,(4304,4347),(4348,4681),(4682,4686),(4688,4695),4696,(4698,4702),(4704,4745),(4746,4750),(4752,4785),(4786,4790),(4792,4799),4800,(4802,4806),(4808,4823),(4824,4881),(4882,4886),(4888,4955),(4969,4989),(4992,5008),(5024,5110),(5112,5118),(5121,5741),(5743,5760),(5761,5787),(5792,5867),(5870,5881),(5888,5906),(5919,5938),(5952,5970),(5984,5997),(5998,6001),(6016,6068),6103,6108,(6112,6122),(6128,6138),(6160,6170),(6176,6265),(6272,6277),(6279,6313),6314,(6320,6390),(6400,6431),(6470,6510),(6512,6517),(6528,6572),(6576,6602),(6608,6619),(6656,6679),(6688,6741),(6784,6794),(6800,6810),6823,(6917,6964),(6981,6989),(6992,7002),(7043,7073),(7086,7142),(7168,7204),(7232,7242),(7245,7294),(7296,7307),(7312,7355),(7357,7360),(7401,7405),(7406,7412),(7413,7415),7418,(7424,7616),(7680,7958),(7960,7966),(7968,8006),(8008,8014),(8016,8024),8025,8027,8029,(8031,8062),(8064,8117),(8118,8125),8126,(8130,8133),(8134,8141),(8144,8148),(8150,8156),(8160,8173),(8178,8181),(8182,8189),(8304,8306),(8308,8314),(8319,8330),(8336,8349),8450,8455,(8458,8468),8469,(8473,8478),8484,8486,8488,(8490,8494),(8495,8506),(8508,8512),(8517,8522),8526,(8528,8586),(9312,9372),(9450,9472),(10102,10132),(11264,11493),(11499,11503),(11506,11508),11517,(11520,11558),11559,11565,(11568,11624),11631,(11648,11671),(11680,11687),(11688,11695),(11696,11703),(11704,11711),(11712,11719),(11720,11727),(11728,11735),(11736,11743),11823,(12293,12296),(12321,12330),(12337,12342),(12344,12349),(12353,12439),(12445,12448),(12449,12539),(12540,12544),(12549,12592),(12593,12687),(12690,12694),(12704,12736),(12784,12800),(12832,12842),(12872,12880),(12881,12896),(12928,12938),(12977,12992),(13312,19904),(19968,42125),(42192,42238),(42240,42509),(42512,42540),(42560,42607),(42623,42654),(42656,42736),(42775,42784),(42786,42889),(42891,42958),(42960,42962),42963,(42965,42973),(42994,43010),(43011,43014),(43015,43019),(43020,43043),(43056,43062),(43072,43124),(43138,43188),(43216,43226),(43250,43256),43259,(43261,43263),(43264,43302),(43312,43335),(43360,43389),(43396,43443),(43471,43482),(43488,43493),(43494,43519),(43520,43561),(43584,43587),(43588,43596),(43600,43610),(43616,43639),43642,(43646,43696),43697,(43701,43703),(43705,43710),43712,43714,(43739,43742),(43744,43755),(43762,43765),(43777,43783),(43785,43791),(43793,43799),(43808,43815),(43816,43823),(43824,43867),(43868,43882),(43888,44003),(44016,44026),(44032,55204),(55216,55239),(55243,55292),(63744,64110),(64112,64218),(64256,64263),(64275,64280),64285,(64287,64297),(64298,64311),(64312,64317),64318,(64320,64322),(64323,64325),(64326,64434),(64467,64830),(64848,64912),(64914,64968),(65008,65020),(65136,65141),(65142,65277),(65296,65306),(65313,65339),(65345,65371),(65382,65471),(65474,65480),(65482,65488),(65490,65496),(65498,65501),(65536,65548),(65549,65575),(65576,65595),(65596,65598),(65599,65614),(65616,65630),(65664,65787),(65799,65844),(65856,65913),(65930,65932),(66176,66205),(66208,66257),(66273,66300),(66304,66340),(66349,66379),(66384,66422),(66432,66462),(66464,66500),(66504,66512),(66513,66518),(66560,66718),(66720,66730),(66736,66772),(66776,66812),(66816,66856),(66864,66916),(66928,66939),(66940,66955),(66956,66963),(66964,66966),(66967,66978),(66979,66994),(66995,67002),(67003,67005),(67008,67060),(67072,67383),(67392,67414),(67424,67432),(67456,67462),(67463,67505),(67506,67515),(67584,67590),67592,(67594,67638),(67639,67641),67644,(67647,67670),(67672,67703),(67705,67743),(67751,67760),(67808,67827),(67828,67830),(67835,67868),(67872,67898),(67968,68024),(68028,68048),(68050,68097),(68112,68116),(68117,68120),(68121,68150),(68160,68169),(68192,68223),(68224,68256),(68288,68296),(68297,68325),(68331,68336),(68352,68406),(68416,68438),(68440,68467),(68472,68498),(68521,68528),(68608,68681),(68736,68787),(68800,68851),(68858,68900),(68912,68922),(68928,68966),(68975,68998),(69216,69247),(69248,69290),(69296,69298),(69314,69317),(69376,69416),(69424,69446),(69457,69461),(69488,69506),(69552,69580),(69600,69623),(69635,69688),(69714,69744),(69745,69747),69749,(69763,69808),(69840,69865),(69872,69882),(69891,69927),(69942,69952),69956,69959,(69968,70003),70006,(70019,70067),(70081,70085),(70096,70107),70108,(70113,70133),(70144,70162),(70163,70188),(70207,70209),(70272,70279),70280,(70282,70286),(70287,70302),(70303,70313),(70320,70367),(70384,70394),(70405,70413),(70415,70417),(70419,70441),(70442,70449),(70450,70452),(70453,70458),70461,70480,(70493,70498),(70528,70538),70539,70542,(70544,70582),70583,70609,70611,(70656,70709),(70727,70731),(70736,70746),(70751,70754),(70784,70832),(70852,70854),70855,(70864,70874),(71040,71087),(71128,71132),(71168,71216),71236,(71248,71258),(71296,71339),71352,(71360,71370),(71376,71396),(71424,71451),(71472,71484),(71488,71495),(71680,71724),(71840,71923),(71935,71943),71945,(71948,71956),(71957,71959),(71960,71984),71999,72001,(72016,72026),(72096,72104),(72106,72145),72161,72163,72192,(72203,72243),72250,72272,(72284,72330),72349,(72368,72441),(72640,72673),(72688,72698),(72704,72713),(72714,72751),72768,(72784,72813),(72818,72848),(72960,72967),(72968,72970),(72971,73009),73030,(73040,73050),(73056,73062),(73063,73065),(73066,73098),73112,(73120,73130),(73440,73459),73474,(73476,73489),(73490,73524),(73552,73562),73648,(73664,73685),(73728,74650),(74752,74863),(74880,75076),(77712,77809),(77824,78896),(78913,78919),(78944,82939),(82944,83527),(90368,90398),(90416,90426),(92160,92729),(92736,92767),(92768,92778),(92784,92863),(92864,92874),(92880,92910),(92928,92976),(92992,92996),(93008,93018),(93019,93026),(93027,93048),(93053,93072),(93504,93549),(93552,93562),(93760,93847),(93952,94027),94032,(94099,94112),(94176,94178),94179,(94208,100344),(100352,101590),(101631,101641),(110576,110580),(110581,110588),(110589,110591),(110592,110883),110898,(110928,110931),110933,(110948,110952),(110960,111356),(113664,113771),(113776,113789),(113792,113801),(113808,113818),(118000,118010),(119488,119508),(119520,119540),(119648,119673),(119808,119893),(119894,119965),(119966,119968),119970,(119973,119975),(119977,119981),(119982,119994),119995,(119997,120004),(120005,120070),(120071,120075),(120077,120085),(120086,120093),(120094,120122),(120123,120127),(120128,120133),120134,(120138,120145),(120146,120486),(120488,120513),(120514,120539),(120540,120571),(120572,120597),(120598,120629),(120630,120655),(120656,120687),(120688,120713),(120714,120745),(120746,120771),(120772,120780),(120782,120832),(122624,122655),(122661,122667),(122928,122990),(123136,123181),(123191,123198),(123200,123210),123214,(123536,123566),(123584,123628),(123632,123642),(124112,124140),(124144,124154),(124368,124398),(124400,124411),(124896,124903),(124904,124908),(124909,124911),(124912,124927),(124928,125125),(125127,125136),(125184,125252),125259,(125264,125274),(126065,126124),(126125,126128),(126129,126133),(126209,126254),(126255,126270),(126464,126468),(126469,126496),(126497,126499),126500,126503,(126505,126515),(126516,126520),126521,126523,126530,126535,126537,126539,(126541,126544),(126545,126547),126548,126551,126553,126555,126557,126559,(126561,126563),126564,(126567,126571),(126572,126579),(126580,126584),(126585,126589),126590,(126592,126602),(126603,126620),(126625,126628),(126629,126634),(126635,126652),(127232,127245),(130032,130042),(131072,173792),(173824,177978),(177984,178206),(178208,183970),(183984,191457),(191472,192094),(194560,195102),(196608,201547),(201552,205744))}",
}

# fmt: on

if __name__ == "__main__":
    # Run this file directly to generate a line in the cached data above.
    print(
        f'    "{unidata_version}": {repr(repr(compute_categories()).replace(" ", ""))},'
    )


def _test_invert_symmetry(m: CharMask):
    """
    Check that double inversion is the same as the original.

    post: list(_.all_bounds()) == list(m.all_bounds())
    """
    return m.invert().invert()


def _test_intersection(left: CharMask, right: CharMask):
    """
    Check intersection behavior.

    post: _.covers(0) == (left.covers(0) and right.covers(0))
    post: _.covers(9) == (left.covers(9) and right.covers(9))
    """
    return left.intersect(right)


def _test_union(left: CharMask, right: CharMask):
    """
    Check union behavior.

    post: _.covers(0) == (left.covers(0) or right.covers(0))
    post: _.covers(9) == (left.covers(9) or right.covers(9))
    """
    return left.union(right)


def _test_set_operations(left: CharMask, right: CharMask):
    """
    Check: compliment of the intersection is the same as the union of both compliments.

    post: _[0] == _[1]
    """
    return (
        left.intersect(right).invert(),
        left.invert().union(right.invert()),
    )
