"""
Implements an R-tree data structure
"""

import itertools
import numpy as np

from ._cvectorgeo import bbox_intersection_area

MIN_CHILDREN  = 2

class Node(object):
    """ Represents an R-tree node """
    def __init__(self, bbox, parent, max_children=50):
        self.bbox = list(bbox)
        self.parent = parent
        self.max_children = max_children
        self.children = []

    def __len__(self):
        return len(self.children)

    def add(self, geom):
        raise NotImplementedError()

    def split(self):
        raise NotImplementedError()

class LeafNode(Node):
    """ Leaf node

    Parameters
    ----------
    bbox : 4-tuple
        (x-min, y-min, x-max, y-max)
    parent : Node
    max_children : int, optional
    """
    def __init__(self, bbox, parent, **kw):
        super(LeafNode, self).__init__(bbox, parent, **kw)

    def add(self, geom):
        """ Add a geometry and return a code and a list of any new nodes
        generated by splitting. """
        retval = 0
        newnodes = []
        bb = geom.bbox
        self.bbox = [min(self.bbox[0], bb[0]), min(self.bbox[1], bb[1]),
                     max(self.bbox[2], bb[2]), max(self.bbox[3], bb[3])]
        self.children.append(geom)

        if len(self.children) > self.max_children:
            retval = 1
            newnodes = self.split()
        return retval, newnodes

    def split(self, kind="linear"):
        """ Split and return new nodes. """
        bbs = [c.bbox for c in self.children]
        if kind == "linear":
            seeds = linear_pick_seeds(self.children)
            pick_next = linear_pick_next
        elif kind == "quadratic":
            seeds = quadratic_pick_seeds(self.children)
            pick_next = quadratic_pick_next
        else:
            raise KeyError("kind must be one of 'linear', 'quadratic'")
        node0 = LeafNode(bbs[seeds[0]], self.parent, max_children=self.max_children)
        node1 = LeafNode(bbs[seeds[1]], self.parent, max_children=self.max_children)

        # add each child to a node
        children = self.children
        newnodes = [node0, node1]
        while len(children) != 0:
            i, j = pick_next(node0.bbox, node1.bbox, children)
            newnodes[j].add(children.pop(i))

            # Ensure that each node gets a minimum number of children
            if len(node0) + len(children) == MIN_CHILDREN:
                for i in range(len(children)-1, -1, -1):
                    node0.add(children.pop(i))
            elif len(node1) + len(children) == MIN_CHILDREN:
                for i in range(len(children)-1, -1, -1):
                    node1.add(children.pop(i))

        return [node0, node1]

class NonLeafNode(Node):
    """ Non-leaf node

    Parameters
    ----------
    bbox : 4-tuple
        (x-min, y-min, x-max, y-max)
    parent : Node
    max_children : int, optional
    """
    def __init__(self, bbox, parent, **kw):
        super(NonLeafNode, self).__init__(bbox, parent, **kw)

    def add(self, geom):
        """ Add a geometry and return a code and a list of any new nodes
        generated by splitting. """
        retval = 0
        newnodes = []

        # choose a child node to add to
        v = [volume_expanded(c.bbox, geom.bbox) for c in self.children]
        i = v.index(min(v))
        target = self.children[i]
        flag, newchildnodes = target.add(geom)
        bb = target.bbox
        self.bbox = [min(self.bbox[0], bb[0]), min(self.bbox[1], bb[1]),
                     max(self.bbox[2], bb[2]), max(self.bbox[3], bb[3])]

        if flag == 1:
            # replace
            del self.children[i]
            self.children.extend(newchildnodes)
            if len(self.children) > self.max_children:
                newnodes = self.split()
                retval = 1

        return retval, newnodes

    def split(self, kind="linear"):
        """ Split and return new nodes. """
        bbs = [c.bbox for c in self.children]
        if kind == "linear":
            seeds = linear_pick_seeds(self.children)
            pick_next = linear_pick_next
        elif kind == "quadratic":
            seeds = quadratic_pick_seeds(self.children)
            pick_next = quadratic_pick_next
        else:
            raise KeyError("kind must be one of 'linear', 'quadratic'")
        node0 = NonLeafNode(bbs[seeds[0]], self.parent, max_children=self.max_children)
        node1 = NonLeafNode(bbs[seeds[1]], self.parent, max_children=self.max_children)

        # add each child to a node
        children = self.children
        newnodes = [node0, node1]
        while len(children) != 0:
            i, j = pick_next(node0.bbox, node1.bbox, children)
            _bb = children[i].bbox
            _node = newnodes[j]
            _node.children.append(children.pop(i))
            _node.bbox = [min(_node.bbox[0], _bb[0]), min(_node.bbox[1], _bb[1]),
                          max(_node.bbox[2], _bb[2]), max(_node.bbox[3], _bb[3])]

            # Ensure that each node gets a minimum number of children
            if len(node0) + len(children) == MIN_CHILDREN:
                for i in range(len(children)-1, -1, -1):
                    _bb = children[i].bbox
                    node0.children.append(children.pop(i))
                    node0.bbox = [min(_node.bbox[0], _bb[0]), min(_node.bbox[1], _bb[1]),
                                  max(node0.bbox[2], _bb[2]), max(_node.bbox[3], _bb[3])]
            elif len(node1) + len(children) == MIN_CHILDREN:
                for i in range(len(children)-1, -1, -1):
                    _bb = children[i].bbox
                    node1.children.append(children.pop(i))
                    node1.bbox = [min(_node.bbox[0], _bb[0]), min(_node.bbox[1], _bb[1]),
                                  max(node1.bbox[2], _bb[2]), max(_node.bbox[3], _bb[3])]

        return [node0, node1]

def build_tree(geoms, max_children=50):
    """ Construct an R-tree

    Parameters
    ----------
    geoms : list of objects implementing a `bbox` attribute
    max_children : int, optional
    """
    bboxes = [g.bbox for g in geoms]
    xmin = min(b[0] for b in bboxes)
    xmax = max(b[2] for b in bboxes)
    ymin = min(b[1] for b in bboxes)
    ymax = max(b[3] for b in bboxes)

    root = LeafNode([xmin, ymin, xmax, ymax], None, max_children=max_children)
    for geom in geoms:
        flag, newnodes = root.add(geom)
        if flag == 1:
            root = NonLeafNode(root.bbox, None, max_children=max_children)
            root.children = newnodes
    return root

def linear_pick_seeds(geoms):
    """ Choose the seeds for splitting a list of geometries using the linear
    approximation of Guttman (1984). """
    fullbb = list(geoms[0].bbox)
    extreme_x0 = 0  # highest xmin
    extreme_x1 = 0  # lowest xmax
    extreme_y0 = 0  # highest ymin
    extreme_y1 = 0  # lowest ymax
    i = 1
    for geom in geoms[1:]:
        gbb = geom.bbox
        fullbb[0] = min(fullbb[0], gbb[0])
        fullbb[1] = min(fullbb[1], gbb[1])
        fullbb[2] = max(fullbb[2], gbb[2])
        fullbb[3] = max(fullbb[3], gbb[3])
        if geoms[extreme_x0].bbox[0] < gbb[0]:
            extreme_x0 = i
        if geoms[extreme_x1].bbox[2] > gbb[2]:
            extreme_x1 = i
        if geoms[extreme_y0].bbox[1] < gbb[1]:
            extreme_y0 = i
        if geoms[extreme_y1].bbox[3] > gbb[3]:
            extreme_y1 = i
        i += 1
    # normalize
    dx = (geoms[extreme_x0].bbox[0] - geoms[extreme_x1].bbox[2]) / (fullbb[2]-fullbb[0])
    dy = (geoms[extreme_y0].bbox[1] - geoms[extreme_y1].bbox[3]) / (fullbb[3]-fullbb[1])
    if dx > dy:
        return extreme_x0, extreme_x1
    else:
        return extreme_y0, extreme_y1

def linear_pick_next(bb0, bb1, geoms):
    """ Simply return the last geometry, as well as the index of the bbox that
    it should be placed in. """
    n = len(geoms)-1
    if volume_expanded(bb0, geoms[n].bbox) > volume_expanded(bb1, geoms[n].bbox):
        return n, 1
    else:
        return n, 0

def quadratic_pick_seeds(geoms):
    """ Choose the seeds for splitting a list of geometries using the quadratic
    approximation of Guttman (1984). """
    d = -1.0
    I, J = None, None
    for (i, j) in itertools.combinations(range(len(geoms)), 2):
        bbi = geoms[i].bbox
        bbj = geoms[j].bbox
        bbc = [min(bbi[0], bbj[0]), min(bbi[1], bbj[1]),
               max(bbi[2], bbj[2]), max(bbi[3], bbj[3])]
        d_ = area(bbc) - area(bbi) - area(bbj)
        if d_ > d:
            d = d_
            I, J = i, j
    return I, J

def quadratic_pick_next(bb0, bb1, geoms):
    """ Choose an index from *geoms* to place in one of *bb0*, *bb1*.
    Return the best entry to index next, and the bbox to place it in. """
    d0 = [volume_expanded(bb0, g.bbox) for g in geoms]
    d1 = [volume_expanded(bb1, g.bbox) for g in geoms]
    diff = [abs(a-b) for (a,b) in zip(d0, d1)]
    i = diff.index(max(diff))
    if d0[i] > d1[i]:
        return i, 1
    else:
        return i, 0

def area(bbox):
    return (bbox[2]-bbox[0])*(bbox[3]-bbox[1])

def volume_expanded(bb0, bb1):
    """ Return the volume by which the union of two bounding boxes larger than
    the bounding box of geom0. """
    union_bbox = [min(bb0[0], bb1[0]), min(bb0[1], bb1[1]),
                  max(bb0[2], bb1[2]), max(bb0[3], bb1[3])]
    return area(union_bbox)-area(bb0)

def depth(tree, d=0):
    if isinstance(tree, LeafNode):
        return d+1
    else:
        if len(tree.children) != 0:
            return max(depth(child, d+1) for child in tree.children)
        else:
            return d

# def bbox_intersection_area(bb0, bb1):
#     dx = max(min(bb0[2], bb1[2]) - max(bb0[0], bb1[0]), 0.0)
#     dy = max(min(bb0[3], bb1[3]) - max(bb0[1], bb1[1]), 0.0)
#     return dx*dy

def overlaps(bb0, bb1):
    return bbox_intersection_area(bb0, bb1) != 0.0

def search(tree, bbox):
    """ Return all geometries overlapping *bbox*. """
    if isinstance(tree, LeafNode):
        matches = []
        for geom in tree.children:
            if overlaps(geom.bbox, bbox):
                matches.append(geom)
    else:
        matches = []
        for node in tree.children:
            matches.extend(search(node, bbox))
    return matches

def contains_geom(tree, geom):
    """ Test whether tree contains *geom* """
    raise NotImplementedError()
    return

def itergeoms(tree):
    """ Iterator for all geometry ojects in tree """
    if isinstance(tree, LeafNode):
        for geom in tree.children:
            yield geom
    else:
        for child in tree.children:
            for geom in itergeoms(child):
                yield geom

def iternodes(tree):
    """ Iterator for all nodes in tree """
    for child in tree.children:
        if isinstance(child, Node):
            for _child in iternodes(child):
                yield _child
        else:
            yield child

def iterleaves(tree):
    """ Iterator for all leaf nodes in tree """
    for child in tree.children:
        if isinstance(child, NonLeafNode):
            for _child in iterleaves(child):
                yield _child
        else:
            yield child
