#!/usr/bin/env python3
"""
ChalkML Command-Line Interface
===============================

Terminal-based ML data manipulation with intuitive notation.

Examples:
    chalkml -rm col 01N results.csv
    chalkml -fillsmart col 03N mean data.csv
    chalkml -map col 05N "x*2" data.csv
    chalkml -if col 06N "col:05N>100" "High" "Low" data.csv
"""

import sys
from pathlib import Path
from typing import List, Optional

from .engine import get_chalkml_engine
from .knowledge_graph import get_knowledge_graph


def show_help():
    """Create the argument parser for ChalkML CLI"""
    parser = argparse.ArgumentParser(
        prog='chalkml',
        description='Terminal-based ML data manipulation engine',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Remove operations
  chalkml -rm col 01N data.csv              # Remove 1st column
  chalkml -rm col N02 data.csv              # Remove 2nd last column
  chalkml -rm col range 01N:05N data.csv    # Remove columns 1-5
  
  # Move & Copy
  chalkml -mv col 01N N01 data.csv          # Move 1st to last
  chalkml -cp col 02N 03N data.csv          # Copy 2nd to 3rd
  
  # Smart fill
  chalkml -fillsmart col 03N mean data.csv  # Fill with mean
  chalkml -fillsmart col 04N knn data.csv   # KNN imputation
  
  # Design patterns
  chalkml -map col 05N "x*2" data.csv       # Double values
  chalkml -reduce col 01N,02N,03N sum Total data.csv
  chalkml -stencil col 03N 5 rolling_mean data.csv
  
  # Control flow
  chalkml -if col 06N "col:05N>100" "High" "Low" data.csv
  chalkml -while col 03N "col:03N>0" "col:03N*0.9" data.csv

Position Notation:
  01N, 02N, 10N  = 1st, 2nd, 10th from left
  N01, N02, N10  = Last, 2nd last, 10th last from right
  
For more info: https://github.com/mankind-research/chalkml
        """
    )
    
    parser.add_argument('--version', action='version', version='ChalkML 1.0.0')
    parser.add_argument('-o', '--output', help='Output file path (default: overwrite input)')
    
    subparsers = parser.add_subparsers(dest='command', help='ChalkML commands')
    
    # Info command
    info_parser = subparsers.add_parser('-info', help='Show dataset information')
    info_parser.add_argument('file', help='CSV file path')
    
    # Undo command
    undo_parser = subparsers.add_parser('-undo', help='Undo last operation')
    
    # Remove command
    rm_parser = subparsers.add_parser('-rm', help='Remove columns or rows')
    rm_parser.add_argument('target', choices=['col', 'row'], help='Remove column or row')
    rm_parser.add_argument('position', nargs='?', help='Position (01N, N02) or "range" keyword')
    rm_parser.add_argument('range_expr', nargs='?', help='Range expression (01N:05N)')
    rm_parser.add_argument('file', help='CSV file path')
    
    # Move command
    mv_parser = subparsers.add_parser('-mv', help='Move column')
    mv_parser.add_argument('target', choices=['col'], help='Target type')
    mv_parser.add_argument('from_pos', help='Source position')
    mv_parser.add_argument('to_pos', help='Destination position')
    mv_parser.add_argument('file', help='CSV file path')
    
    # Copy command
    cp_parser = subparsers.add_parser('-cp', help='Copy column')
    cp_parser.add_argument('target', choices=['col'], help='Target type')
    cp_parser.add_argument('from_pos', help='Source position')
    cp_parser.add_argument('to_pos', help='Destination position')
    cp_parser.add_argument('file', help='CSV file path')
    
    # Rename command
    rn_parser = subparsers.add_parser('-rn', help='Rename column')
    rn_parser.add_argument('target', choices=['col'], help='Target type')
    rn_parser.add_argument('position', help='Column position')
    rn_parser.add_argument('new_name', help='New column name')
    rn_parser.add_argument('file', help='CSV file path')
    
    # Slice command
    slice_parser = subparsers.add_parser('-slice', help='Slice data')
    slice_parser.add_argument('target', choices=['col', 'row'], help='Target type')
    slice_parser.add_argument('range_expr', help='Range expression (01N:10N)')
    slice_parser.add_argument('file', help='CSV file path')
    
    # Fill command
    fill_parser = subparsers.add_parser('-fill', help='Fill empty cells')
    fill_parser.add_argument('target', choices=['col', 'row'], help='Target type')
    fill_parser.add_argument('position', help='Position')
    fill_parser.add_argument('value', help='Fill value')
    fill_parser.add_argument('file', help='CSV file path')
    
    # Merge command
    merge_parser = subparsers.add_parser('-merge', help='Merge columns')
    merge_parser.add_argument('target', choices=['col'], help='Target type')
    merge_parser.add_argument('positions', help='Comma-separated positions (01N,02N,03N)')
    merge_parser.add_argument('new_name', help='New column name')
    merge_parser.add_argument('file', help='CSV file path')
    merge_parser.add_argument('--sep', default=' ', help='Separator (default: space)')
    
    # Smart fill command
    fillsmart_parser = subparsers.add_parser('-fillsmart', help='Smart fill with strategies')
    fillsmart_parser.add_argument('target', choices=['col'], help='Target type')
    fillsmart_parser.add_argument('position', help='Column position')
    fillsmart_parser.add_argument('strategy', 
                                  choices=['forward', 'backward', 'interpolate', 'mean', 'median', 'mode', 'knn'],
                                  help='Fill strategy')
    fillsmart_parser.add_argument('file', help='CSV file path')
    
    # Derive command
    derive_parser = subparsers.add_parser('-derive', help='Derive new column')
    derive_parser.add_argument('new_name', help='New column name')
    derive_parser.add_argument('formula', help='Derivation formula (use col:pos)')
    derive_parser.add_argument('file', help='CSV file path')
    
    # One-hot encoding
    onehot_parser = subparsers.add_parser('-onehot', help='One-hot encode column')
    onehot_parser.add_argument('target', choices=['col'], help='Target type')
    onehot_parser.add_argument('position', help='Column position')
    onehot_parser.add_argument('file', help='CSV file path')
    onehot_parser.add_argument('--prefix', help='Prefix for encoded columns')
    
    # Binning
    bin_parser = subparsers.add_parser('-bin', help='Bin continuous column')
    bin_parser.add_argument('target', choices=['col'], help='Target type')
    bin_parser.add_argument('position', help='Column position')
    bin_parser.add_argument('file', help='CSV file path')
    bin_parser.add_argument('--bins', type=int, default=5, help='Number of bins (default: 5)')
    bin_parser.add_argument('--strategy', choices=['quantile', 'uniform'], default='quantile')
    
    # Scaling
    scale_parser = subparsers.add_parser('-scale', help='Scale numerical column')
    scale_parser.add_argument('target', choices=['col'], help='Target type')
    scale_parser.add_argument('position', help='Column position')
    scale_parser.add_argument('file', help='CSV file path')
    scale_parser.add_argument('--method', choices=['standard', 'minmax'], default='standard')
    
    # IF/ELSE
    if_parser = subparsers.add_parser('-if', help='Conditional IF/ELSE logic')
    if_parser.add_argument('target', choices=['col'], help='Target type')
    if_parser.add_argument('position', help='Target column position')
    if_parser.add_argument('condition', help='Condition (col:pos==value)')
    if_parser.add_argument('if_value', help='Value when condition is True')
    if_parser.add_argument('else_value', help='Value when condition is False')
    if_parser.add_argument('file', help='CSV file path')
    
    # WHILE
    while_parser = subparsers.add_parser('-while', help='WHILE loop')
    while_parser.add_argument('target', choices=['col'], help='Target type')
    while_parser.add_argument('position', help='Target column position')
    while_parser.add_argument('condition', help='Condition (col:pos>value)')
    while_parser.add_argument('operation', help='Operation to apply')
    while_parser.add_argument('file', help='CSV file path')
    while_parser.add_argument('--max-iter', type=int, default=100, help='Max iterations')
    
    # MAP pattern
    map_parser = subparsers.add_parser('-map', help='MAP pattern: transform each element')
    map_parser.add_argument('target', choices=['col'], help='Target type')
    map_parser.add_argument('position', help='Column position')
    map_parser.add_argument('function', help='Function to apply (use "x" as variable)')
    map_parser.add_argument('file', help='CSV file path')
    
    # REDUCE pattern
    reduce_parser = subparsers.add_parser('-reduce', help='REDUCE pattern: combine columns')
    reduce_parser.add_argument('target', choices=['col'], help='Target type')
    reduce_parser.add_argument('positions', help='Comma-separated positions')
    reduce_parser.add_argument('operation', choices=['sum', 'product', 'mean', 'max', 'min'])
    reduce_parser.add_argument('result_name', help='Result column name')
    reduce_parser.add_argument('file', help='CSV file path')
    
    # STENCIL pattern
    stencil_parser = subparsers.add_parser('-stencil', help='STENCIL pattern: sliding window')
    stencil_parser.add_argument('target', choices=['col'], help='Target type')
    stencil_parser.add_argument('position', help='Column position')
    stencil_parser.add_argument('window_size', type=int, help='Window size')
    stencil_parser.add_argument('operation', 
                                choices=['rolling_mean', 'rolling_sum', 'rolling_max', 'rolling_min', 'rolling_std'])
    stencil_parser.add_argument('file', help='CSV file path')
    
    # SCAN pattern
    scan_parser = subparsers.add_parser('-scan', help='SCAN pattern: cumulative operations')
    scan_parser.add_argument('target', choices=['col'], help='Target type')
    scan_parser.add_argument('position', help='Column position')
    scan_parser.add_argument('operation', choices=['cumsum', 'cumprod', 'cummax', 'cummin'])
    scan_parser.add_argument('file', help='CSV file path')
    
    return parser


def main(args: Optional[List[str]] = None):
    """Main CLI entry point"""
    parser = create_parser()
    
    # Handle no arguments
    if args is None:
        args = sys.argv[1:]
    
    if not args:
        parser.print_help()
        sys.exit(0)
    
    parsed_args = parser.parse_args(args)
    
    # Handle commands with no file requirement
    if parsed_args.command == '-undo':
        engine = get_chalkml_engine()
        success, message = engine.undo_last()
        print(message)
        sys.exit(0 if success else 1)
    
    # Get engine
    engine = get_chalkml_engine()
    
    try:
        success = False
        message = ""
        
        # Route to appropriate handler
        if parsed_args.command == '-info':
            success, message = engine.show_info(parsed_args.file)
        
        elif parsed_args.command == '-rm':
            if parsed_args.position == 'range':
                # Range removal
                success, message = engine.remove_range(
                    parsed_args.file, 
                    parsed_args.target, 
                    parsed_args.range_expr,
                    output_path=parsed_args.output
                )
            elif parsed_args.position:
                # Position removal
                if parsed_args.target == 'col':
                    success, message = engine.remove_column(
                        parsed_args.file, 
                        parsed_args.position,
                        output_path=parsed_args.output
                    )
                else:
                    success, message = engine.remove_row(
                        parsed_args.file, 
                        parsed_args.position,
                        output_path=parsed_args.output
                    )
            else:
                # No position - remove last
                if parsed_args.target == 'col':
                    success, message = engine.remove_column(
                        parsed_args.file, 
                        None,
                        output_path=parsed_args.output
                    )
                else:
                    success, message = engine.remove_row(
                        parsed_args.file, 
                        None,
                        output_path=parsed_args.output
                    )
        
        elif parsed_args.command == '-mv':
            success, message = engine.move_column(
                parsed_args.file,
                parsed_args.from_pos,
                parsed_args.to_pos,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-cp':
            success, message = engine.copy_column(
                parsed_args.file,
                parsed_args.from_pos,
                parsed_args.to_pos,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-rn':
            success, message = engine.rename_column(
                parsed_args.file,
                parsed_args.position,
                parsed_args.new_name,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-slice':
            success, message = engine.slice_data(
                parsed_args.file,
                parsed_args.target,
                parsed_args.range_expr,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-fill':
            success, message = engine.fill_empty(
                parsed_args.file,
                parsed_args.target,
                parsed_args.position,
                parsed_args.value,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-merge':
            positions = parsed_args.positions.split(',')
            success, message = engine.merge_columns(
                parsed_args.file,
                positions,
                parsed_args.new_name,
                separator=parsed_args.sep,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-fillsmart':
            success, message = engine.fill_smart(
                parsed_args.file,
                parsed_args.position,
                parsed_args.strategy,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-derive':
            success, message = engine.derive_column(
                parsed_args.file,
                parsed_args.new_name,
                parsed_args.formula,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-onehot':
            success, message = engine.one_hot_encode(
                parsed_args.file,
                parsed_args.position,
                prefix=parsed_args.prefix,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-bin':
            success, message = engine.bin_column(
                parsed_args.file,
                parsed_args.position,
                n_bins=parsed_args.bins,
                strategy=parsed_args.strategy,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-scale':
            success, message = engine.scale_column(
                parsed_args.file,
                parsed_args.position,
                method=parsed_args.method,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-if':
            success, message = engine.apply_if_else(
                parsed_args.file,
                parsed_args.position,
                parsed_args.condition,
                parsed_args.if_value,
                parsed_args.else_value,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-while':
            success, message = engine.apply_while(
                parsed_args.file,
                parsed_args.position,
                parsed_args.condition,
                parsed_args.operation,
                max_iterations=parsed_args.max_iter,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-map':
            success, message = engine.map_pattern(
                parsed_args.file,
                parsed_args.position,
                parsed_args.function,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-reduce':
            positions = parsed_args.positions.split(',')
            success, message = engine.reduce_pattern(
                parsed_args.file,
                positions,
                parsed_args.operation,
                parsed_args.result_name,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-stencil':
            success, message = engine.stencil_pattern(
                parsed_args.file,
                parsed_args.position,
                parsed_args.window_size,
                parsed_args.operation,
                output_path=parsed_args.output
            )
        
        elif parsed_args.command == '-scan':
            success, message = engine.scan_pattern(
                parsed_args.file,
                parsed_args.position,
                parsed_args.operation,
                output_path=parsed_args.output
            )
        
        else:
            print(f"Unknown command: {parsed_args.command}")
            parser.print_help()
            sys.exit(1)
        
        # Print result
        print(message)
        sys.exit(0 if success else 1)
    
    except Exception as e:
        print(f"Error: {str(e)}", file=sys.stderr)
        sys.exit(1)


if __name__ == '__main__':
    main()
