#!/usr/bin/env python3
"""
ChalkML Command-Line Interface (Simple Parser)
===============================================

Direct string parsing like the original terminal_widget.py

Syntax:
    chalkml -rm col 01N file.csv
    chalkml -fillsmart col 03N mean file.csv
    chalkml -map col 05N "x*2" file.csv
"""

import sys
from pathlib import Path

from .engine import get_chalkml_engine
from .git_engine import get_chalkml_git
from .quantum_engine import get_quantum_engine
from .relevance_engine import get_relevance_engine
from .redact_engine import get_redact_engine
from .scaffold_engine import get_scaffold_engine
from .bow_engine import get_bow_engine


def show_help():
    """Show ChalkML help"""
    help_text = """
ChalkML - Terminal-Based ML Data Manipulation
==============================================

POSITION NOTATION:
  01N, 02N, 10N  = 1st, 2nd, 10th from left
  N01, N02, N10  = Last, 2nd last, 10th last from right

COMMANDS:

  chalkml -info <file>
      Show dataset information

  chalkml -rm col [position] <file>
  chalkml -rm row [position] <file>
      Remove column or row
      Examples:
        chalkml -rm col 01N data.csv        # Remove 1st column
        chalkml -rm col N03 data.csv        # Remove 3rd from right
        chalkml -rm col range 01N:05N data.csv  # Remove columns 1-5

  chalkml -mv col from to <file>
      Move column
      Example: chalkml -mv col 01N N01 data.csv

  chalkml -cp col from to <file>
      Copy column
      Example: chalkml -cp col 02N 03N data.csv

  chalkml -rn col position "name" <file>
      Rename column
      Example: chalkml -rn col 03N "Age" data.csv

  chalkml -fillsmart col position strategy <file>
      Smart fill missing values
      Strategies: mean, median, mode, knn, forward, backward, interpolate
      Example: chalkml -fillsmart col 03N mean data.csv

  chalkml -derive "name" "formula" <file>
      Derive new column
      Example: chalkml -derive "BMI" "col:weight/(col:height**2)" data.csv

  chalkml -onehot col position <file>
      One-hot encode column
      Example: chalkml -onehot col 05N data.csv

  chalkml -scale col position <file> [--method standard|minmax]
      Scale numerical column
      Example: chalkml -scale col 03N data.csv --method standard

  chalkml -map col position "function" <file>
      MAP pattern: transform each element
      Example: chalkml -map col 05N "x*2" data.csv

  chalkml -reduce col positions operation result <file>
      REDUCE pattern: combine columns
      Example: chalkml -reduce col 01N,02N,03N sum Total data.csv

  chalkml -stencil col position window operation <file>
      STENCIL pattern: sliding window
      Example: chalkml -stencil col 03N 5 rolling_mean data.csv

  chalkml -scan col position operation <file>
      SCAN pattern: cumulative operations
      Example: chalkml -scan col 03N cumsum data.csv

  chalkml -if col position "condition" if_value else_value <file>
      IF/ELSE conditional logic
      Example: chalkml -if col 06N "col:05N>100" "High" "Low" data.csv

  chalkml -undo
      Undo last operation

GIT COMMANDS (Version Control for Data):

  chalkml init
      Initialize data repository (Git for Data)

  chalkml commit -m "message" <file>
      Commit current data state
      Example: chalkml commit -m "Filled missing values" data.csv

  chalkml log [--limit N]
      Show commit history
      Example: chalkml log --limit 20

  chalkml checkout <sha> <output>
      Time travel to specific commit
      Example: chalkml checkout a7f8c9d2 restored.csv

  chalkml diff <sha1> <sha2>
      Compare two commits
      Example: chalkml diff a7f8c9d2 b2e4f6a8

  chalkml branch <name>
      Create and switch to new branch
      Example: chalkml branch experimental

  chalkml stash <name> <file>
      Stash current state
      Example: chalkml stash "backup-v1" data.csv

  chalkml stash pop <name> <output>
      Restore stashed state
      Example: chalkml stash pop "backup-v1" data.csv

  chalkml stash list
      List all stashes

QUANTUM COMMANDS (Deterministic Compression):

  chalkml -quantum col all --analyze <file>
      Analyze correlations and suggest quantum groups
      Example: chalkml -quantum col all --analyze data.csv

  chalkml -quantum col all --output <file>
      Quantumize data (auto-group correlated features)
      Example: chalkml -quantum col all --output quantumized_data.csv data.csv

  chalkml -quantum col all --schema <schema.json> --input <file> --output <file>
      Apply existing schema to new data (PRODUCTION MODE)
      Example: chalkml -quantum col all --schema .chalkml/quantum_schemas/schema.json --input new.csv --output q_new.csv

  chalkml -quantum col all --decode --input <file> --output <file>
      Decode quantum features back to original columns
      Example: chalkml -quantum col all --decode --input quantumized.csv --output decoded.csv

For detailed documentation, visit: https://github.com/mankind-research/chalkml
"""
    print(help_text)


def main(args=None):
    """Main CLI entry point - uses simple string parsing like terminal_widget.py"""
    if args is None:
        args = sys.argv[1:]
    
    if not args or args[0] in ['-h', '--help', '-help']:
        show_help()
        return 0
    
    if args[0] == '--version':
        print("ChalkML 1.0.0")
        return 0
    
    # Get engines (with current directory)
    import os
    current_dir = os.getcwd()
    engine = get_chalkml_engine(workspace_path=current_dir)
    git_engine = get_chalkml_git(workspace_path=current_dir)
    quantum_engine = get_quantum_engine(workspace_path=current_dir)
    relevance_engine = get_relevance_engine(workspace_path=current_dir)
    redact_engine = get_redact_engine(workspace_path=current_dir)
    scaffold_engine = get_scaffold_engine(workspace_path=current_dir)
    bow_engine = get_bow_engine(workspace_path=current_dir)
    
    try:
        command = args[0]
        
        # Git commands (version control)
        if command == 'init':
            success, message = git_engine.init()
            print(message)
            return 0 if success else 1
        
        elif command == 'commit':
            if len(args) < 4 or args[1] != '-m':
                print("Usage: chalkml commit -m \"message\" <file>")
                return 1
            message = args[2]
            file_path = args[3]
            success, msg = git_engine.commit(message, file_path)
            print(msg)
            return 0 if success else 1
        
        elif command == 'log':
            limit = 10
            if '--limit' in args:
                limit_idx = args.index('--limit')
                limit = int(args[limit_idx + 1])
            success, message = git_engine.log(limit=limit)
            print(message)
            return 0 if success else 1
        
        elif command == 'checkout':
            if len(args) < 3:
                print("Usage: chalkml checkout <sha> <output_file>")
                return 1
            sha = args[1]
            output = args[2]
            success, message = git_engine.checkout(sha, output)
            print(message)
            return 0 if success else 1
        
        elif command == 'diff':
            if len(args) < 3:
                print("Usage: chalkml diff <sha1> <sha2>")
                return 1
            sha1 = args[1]
            sha2 = args[2]
            success, message = git_engine.diff(sha1, sha2)
            print(message)
            return 0 if success else 1
        
        elif command == 'branch':
            if len(args) < 2:
                print("Usage: chalkml branch <name>")
                return 1
            name = args[1]
            success, message = git_engine.branch(name, create=True)
            print(message)
            return 0 if success else 1
        
        elif command == 'stash':
            if len(args) < 2:
                # List stashes
                if len(args) == 1:
                    print("Usage: chalkml stash <name> <file>  OR  chalkml stash list")
                    return 1
                elif args[1] == 'list':
                    success, message = git_engine.stash_list()
                    print(message)
                    return 0 if success else 1
                elif args[1] == 'pop':
                    if len(args) < 4:
                        print("Usage: chalkml stash pop <name> <output_file>")
                        return 1
                    name = args[2]
                    output = args[3]
                    success, message = git_engine.stash_pop(name, output)
                    print(message)
                    return 0 if success else 1
            else:
                # Stash save
                if len(args) < 3:
                    print("Usage: chalkml stash <name> <file>")
                    return 1
                name = args[1]
                file_path = args[2]
                success, message = git_engine.stash(name, file_path)
                print(message)
                return 0 if success else 1
        
        # QUANTUM commands (deterministic compression)
        elif command == '-quantum':
            if len(args) < 2:
                print("Usage: chalkml -quantum col all [options]")
                return 1
            
            # Parse arguments
            # chalkml -quantum col all --analyze data.csv
            # chalkml -quantum col all --output quantumized.csv data.csv
            # chalkml -quantum col all --schema schema.json --input data.csv --output q_data.csv
            
            if '--analyze' in args:
                # Analysis mode
                analyze_idx = args.index('--analyze')
                if analyze_idx + 1 >= len(args):
                    print("Usage: chalkml -quantum col all --analyze <file>")
                    return 1
                input_file = args[analyze_idx + 1]
                success, message = quantum_engine.analyze_file(input_file)
                print(message)
                return 0 if success else 1
            
            elif '--decode' in args:
                # Decode mode
                if '--input' not in args or '--output' not in args:
                    print("Usage: chalkml -quantum col all --decode --input <file> --output <file>")
                    return 1
                input_idx = args.index('--input')
                output_idx = args.index('--output')
                input_file = args[input_idx + 1]
                output_file = args[output_idx + 1]
                
                # Load schema from input file's metadata or use default
                # For now, we'll need schema path
                if '--schema' not in args:
                    print("Error: --decode requires --schema <schema.json>")
                    return 1
                schema_idx = args.index('--schema')
                schema_path = args[schema_idx + 1]
                
                try:
                    from .quantum_engine import QuantumSchema
                    import pandas as pd
                    
                    schema = QuantumSchema.load(Path(schema_path))
                    q_df = pd.read_csv(input_file)
                    decoded_df = quantum_engine.decode_quantum_dataframe(q_df, schema)
                    decoded_df.to_csv(output_file, index=False)
                    
                    print(f"✅ Decoded: {input_file} → {output_file}")
                    print(f"   Columns: {len(q_df.columns)} → {len(decoded_df.columns)}")
                    return 0
                except Exception as e:
                    print(f"❌ Decode failed: {str(e)}")
                    return 1
            
            elif '--schema' in args:
                # Production mode: Apply existing schema
                if '--input' not in args or '--output' not in args:
                    print("Usage: chalkml -quantum col all --schema <schema.json> --input <file> --output <file>")
                    return 1
                
                schema_idx = args.index('--schema')
                input_idx = args.index('--input')
                output_idx = args.index('--output')
                
                schema_path = args[schema_idx + 1]
                input_file = args[input_idx + 1]
                output_file = args[output_idx + 1]
                
                success, message, _ = quantum_engine.quantumize_file(
                    input_path=input_file,
                    output_path=output_file,
                    schema_path=schema_path
                )
                print(message)
                return 0 if success else 1
            
            elif '--output' in args:
                # Training mode: Create new schema
                output_idx = args.index('--output')
                if output_idx + 1 >= len(args):
                    print("Usage: chalkml -quantum col all --output <output_file> <input_file>")
                    return 1
                
                output_file = args[output_idx + 1]
                
                # Input file is last argument
                input_file = args[-1]
                
                # Check for target column
                target_col = None
                if '--target' in args:
                    target_idx = args.index('--target')
                    target_col = args[target_idx + 1]
                
                success, message, schema_path = quantum_engine.quantumize_file(
                    input_path=input_file,
                    output_path=output_file,
                    schema_path=None,  # Create new
                    auto_group=True,
                    target_column=target_col
                )
                print(message)
                if schema_path:
                    print(f"\n📋 Schema saved: {schema_path}")
                    print("   Use this schema for production predictions:")
                    print(f"   chalkml -quantum col all --schema {schema_path} --input new.csv --output q_new.csv")
                return 0 if success else 1
            
            else:
                print("Error: Must specify --analyze, --output, --schema, or --decode")
                return 1
        
        # RELEVANCE commands (feature selection)
        elif command == '-relevance':
            if len(args) < 2:
                print("Usage: chalkml -relevance col all [options]")
                return 1
            
            if '--analyze' in args:
                # Analysis mode - show relevance metrics
                if '--target' not in args:
                    print("Error: --analyze requires --target <column>")
                    return 1
                
                target_idx = args.index('--target')
                target_col = args[target_idx + 1]
                input_file = args[-1]
                
                # Get alpha if specified
                alpha = 0.05
                if '--alpha' in args:
                    alpha_idx = args.index('--alpha')
                    alpha = float(args[alpha_idx + 1])
                
                try:
                    analysis = relevance_engine.analyze_file(input_file, target_col, alpha=alpha)
                    
                    print(f"\n📊 Feature Relevance Analysis: {input_file}")
                    print("=" * 60)
                    print(f"Target: {target_col}")
                    print(f"Significance level (α): {alpha}")
                    print(f"\nOriginal features: {analysis['original_count']}")
                    print(f"Selected features: {analysis['selected_count']}")
                    print(f"Reduction: {analysis['reduction_pct']:.1f}%")
                    
                    print(f"\n🔍 Top 10 Features by Mutual Information:")
                    mi_sorted = sorted(analysis['mutual_information'].items(), key=lambda x: x[1], reverse=True)
                    for i, (feat, score) in enumerate(mi_sorted[:10], 1):
                        print(f"   {i:2d}. {feat:30s} I(X;Y) = {score:.4f}")
                    
                    print(f"\n✅ Selected Features ({len(analysis['selected_features'])}):")
                    for feat in analysis['selected_features'][:20]:  # Show first 20
                        print(f"   - {feat}")
                    if len(analysis['selected_features']) > 20:
                        print(f"   ... and {len(analysis['selected_features']) - 20} more")
                    
                    return 0
                except Exception as e:
                    print(f"❌ Analysis failed: {str(e)}")
                    return 1
            
            elif '--output' in args:
                # Selection mode - create filtered file
                if '--target' not in args:
                    print("Error: --output requires --target <column>")
                    return 1
                
                target_idx = args.index('--target')
                output_idx = args.index('--output')
                target_col = args[target_idx + 1]
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                # Get alpha if specified
                alpha = 0.05
                if '--alpha' in args:
                    alpha_idx = args.index('--alpha')
                    alpha = float(args[alpha_idx + 1])
                
                try:
                    analysis = relevance_engine.select_features_file(
                        input_file, output_file, target_col, alpha=alpha
                    )
                    
                    print(f"✅ Feature selection complete")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Original features: {analysis['original_count']}")
                    print(f"   Selected features: {analysis['selected_count']}")
                    print(f"   Reduction: {analysis['reduction_pct']:.1f}%")
                    print(f"\n📋 Report saved to .chalkml/relevance_reports/")
                    
                    return 0
                except Exception as e:
                    print(f"❌ Feature selection failed: {str(e)}")
                    return 1
            
            else:
                print("Error: Must specify --analyze or --output")
                return 1
        
        # REDACT commands (privacy-preserving)
        elif command == '-redact':
            if len(args) < 2:
                print("Usage: chalkml -redact [method] [options]")
                return 1
            
            method = args[1]  # 'dp', 'k-anonymity', 'masking'
            
            if method == 'dp':
                # Differential privacy
                if '--epsilon' not in args or '--output' not in args:
                    print("Usage: chalkml -redact dp --epsilon <value> --output <file> <input>")
                    return 1
                
                epsilon_idx = args.index('--epsilon')
                output_idx = args.index('--output')
                epsilon = float(args[epsilon_idx + 1])
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                try:
                    result = redact_engine.redact_file(
                        input_file, output_file, method='dp', epsilon=epsilon
                    )
                    
                    print(f"✅ Differential Privacy applied (ε = {epsilon})")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['input_rows']}")
                    print(f"   Privacy level: {'Strong' if epsilon < 1 else 'Moderate' if epsilon < 5 else 'Weak'}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Differential privacy failed: {str(e)}")
                    return 1
            
            elif method == 'k-anonymity':
                # k-anonymity
                if '--k' not in args or '--qi' not in args or '--output' not in args:
                    print("Usage: chalkml -redact k-anonymity --k <value> --qi <col1,col2,...> --output <file> <input>")
                    return 1
                
                k_idx = args.index('--k')
                qi_idx = args.index('--qi')
                output_idx = args.index('--output')
                
                k = int(args[k_idx + 1])
                quasi_identifiers = args[qi_idx + 1].split(',')
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                try:
                    result = redact_engine.redact_file(
                        input_file, output_file, method='k-anonymity',
                        k=k, quasi_identifiers=quasi_identifiers
                    )
                    
                    metadata = result['metadata']
                    print(f"✅ k-Anonymity applied (k = {k})")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['input_rows']}")
                    print(f"   Quasi-identifiers: {', '.join(quasi_identifiers)}")
                    print(f"   Achieved k: {metadata.get('achieved_k', '?')}")
                    print(f"   Status: {'✓ Achieved' if metadata.get('achieved') else '✗ Not achieved'}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ k-Anonymity failed: {str(e)}")
                    return 1
            
            elif method == 'masking':
                # Data masking
                if '--output' not in args:
                    print("Usage: chalkml -redact masking --output <file> [--hash <cols>] [--tokenize <cols>] <input>")
                    return 1
                
                output_idx = args.index('--output')
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                hash_cols = None
                tokenize_cols = None
                
                if '--hash' in args:
                    hash_idx = args.index('--hash')
                    hash_cols = args[hash_idx + 1].split(',')
                
                if '--tokenize' in args:
                    tok_idx = args.index('--tokenize')
                    tokenize_cols = args[tok_idx + 1].split(',')
                
                try:
                    result = redact_engine.redact_file(
                        input_file, output_file, method='masking',
                        hash_columns=hash_cols, tokenize_columns=tokenize_cols
                    )
                    
                    print(f"✅ Data masking applied")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['input_rows']}")
                    if hash_cols:
                        print(f"   Hashed: {', '.join(hash_cols)}")
                    if tokenize_cols:
                        print(f"   Tokenized: {', '.join(tokenize_cols)}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Data masking failed: {str(e)}")
                    return 1
            
            else:
                print(f"Unknown redaction method: {method}")
                print("Available: dp, k-anonymity, masking")
                return 1
        
        # SCAFFOLD commands (mathematical data generation)
        elif command == '-scaffold':
            if len(args) < 2:
                print("Usage: chalkml -scaffold [method] [options]")
                return 1
            
            method = args[1]  # 'sequence', 'distribution', 'timeseries'
            
            if method == 'sequence':
                if '--type' not in args or '--output' not in args:
                    print("Usage: chalkml -scaffold sequence --type <seq_type> --count <n> --output <file>")
                    return 1
                
                type_idx = args.index('--type')
                output_idx = args.index('--output')
                seq_type = args[type_idx + 1]
                output_file = args[output_idx + 1]
                
                count = 100
                if '--count' in args:
                    count_idx = args.index('--count')
                    count = int(args[count_idx + 1])
                
                scale = 1.0
                if '--scale' in args:
                    scale_idx = args.index('--scale')
                    scale = float(args[scale_idx + 1])
                
                try:
                    result = scaffold_engine.scaffold_file(
                        output_file, method='sequence',
                        sequence_type=seq_type, count=count, scale=scale
                    )
                    
                    print(f"✅ Sequence generated: {seq_type}")
                    print(f"   Output: {output_file}")
                    print(f"   Count: {result['rows']}")
                    print(f"   Scale: {scale}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Sequence generation failed: {str(e)}")
                    return 1
            
            elif method == 'distribution':
                if '--dist' not in args or '--output' not in args:
                    print("Usage: chalkml -scaffold distribution --dist <type> --count <n> --output <file>")
                    return 1
                
                dist_idx = args.index('--dist')
                output_idx = args.index('--output')
                dist_type = args[dist_idx + 1]
                output_file = args[output_idx + 1]
                
                count = 100
                if '--count' in args:
                    count_idx = args.index('--count')
                    count = int(args[count_idx + 1])
                
                kwargs = {'dist_type': dist_type, 'count': count}
                
                if dist_type == 'normal':
                    if '--mean' in args:
                        mean_idx = args.index('--mean')
                        kwargs['mean'] = float(args[mean_idx + 1])
                    if '--std' in args:
                        std_idx = args.index('--std')
                        kwargs['std'] = float(args[std_idx + 1])
                
                elif dist_type == 'uniform':
                    if '--low' in args:
                        low_idx = args.index('--low')
                        kwargs['low'] = float(args[low_idx + 1])
                    if '--high' in args:
                        high_idx = args.index('--high')
                        kwargs['high'] = float(args[high_idx + 1])
                
                try:
                    result = scaffold_engine.scaffold_file(
                        output_file, method='distribution', **kwargs
                    )
                    
                    print(f"✅ Distribution generated: {dist_type}")
                    print(f"   Output: {output_file}")
                    print(f"   Count: {result['rows']}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Distribution generation failed: {str(e)}")
                    return 1
            
            elif method == 'timeseries':
                if '--output' not in args:
                    print("Usage: chalkml -scaffold timeseries --start <date> --freq <freq> --count <n> --output <file>")
                    return 1
                
                output_idx = args.index('--output')
                output_file = args[output_idx + 1]
                
                start = '2024-01-01'
                if '--start' in args:
                    start_idx = args.index('--start')
                    start = args[start_idx + 1]
                
                freq = '1D'
                if '--freq' in args:
                    freq_idx = args.index('--freq')
                    freq = args[freq_idx + 1]
                
                count = 365
                if '--count' in args:
                    count_idx = args.index('--count')
                    count = int(args[count_idx + 1])
                
                try:
                    result = scaffold_engine.scaffold_file(
                        output_file, method='timeseries',
                        start=start, freq=freq, count=count
                    )
                    
                    print(f"✅ Time series generated")
                    print(f"   Output: {output_file}")
                    print(f"   Start: {start}")
                    print(f"   Frequency: {freq}")
                    print(f"   Count: {result['rows']}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Time series generation failed: {str(e)}")
                    return 1
            
            else:
                print(f"Unknown scaffold method: {method}")
                print("Available: sequence, distribution, timeseries")
                return 1
        
        # BOW commands (blueprint-based standardization)
        elif command == '-bow':
            if len(args) < 2:
                print("Usage: chalkml -bow [operation] [options]")
                return 1
            
            operation = args[1]  # 'format', 'standard', 'layout'
            
            if operation == 'format':
                if '--column' not in args or '--type' not in args or '--output' not in args:
                    print("Usage: chalkml -bow format --column <col> --type <format> --output <file> <input>")
                    return 1
                
                col_idx = args.index('--column')
                type_idx = args.index('--type')
                output_idx = args.index('--output')
                
                column = args[col_idx + 1]
                format_type = args[type_idx + 1]
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                kwargs = {'column': column, 'format_type': format_type}
                
                # Parse optional args
                if '--pattern' in args:
                    pattern_idx = args.index('--pattern')
                    kwargs['pattern'] = args[pattern_idx + 1]
                
                if '--symbol' in args:
                    symbol_idx = args.index('--symbol')
                    kwargs['symbol'] = args[symbol_idx + 1]
                
                if '--precision' in args:
                    precision_idx = args.index('--precision')
                    kwargs['precision'] = int(args[precision_idx + 1])
                
                try:
                    result = bow_engine.bow_file(input_file, output_file, 'format', **kwargs)
                    
                    print(f"✅ Format applied: {format_type}")
                    print(f"   Column: {column}")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['rows']}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Format failed: {str(e)}")
                    return 1
            
            elif operation == 'standard':
                if '--standard' not in args or '--output' not in args:
                    print("Usage: chalkml -bow standard --standard <type> --output <file> <input>")
                    return 1
                
                standard_idx = args.index('--standard')
                output_idx = args.index('--output')
                
                standard = args[standard_idx + 1]
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                kwargs = {'standard': standard}
                
                # GAAP requires currency columns
                if standard.upper() == 'GAAP' and '--currency' in args:
                    currency_idx = args.index('--currency')
                    currency_cols = args[currency_idx + 1].split(',')
                    kwargs['currency_cols'] = currency_cols
                
                try:
                    result = bow_engine.bow_file(input_file, output_file, 'standard', **kwargs)
                    
                    print(f"✅ Standard applied: {standard}")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['rows']}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Standard application failed: {str(e)}")
                    return 1
            
            elif operation == 'layout':
                if '--layout' not in args or '--output' not in args:
                    print("Usage: chalkml -bow layout --layout <type> --output <file> <input>")
                    return 1
                
                layout_idx = args.index('--layout')
                output_idx = args.index('--output')
                
                layout = args[layout_idx + 1]
                output_file = args[output_idx + 1]
                input_file = args[-1]
                
                try:
                    result = bow_engine.bow_file(input_file, output_file, 'layout', layout=layout)
                    
                    print(f"✅ Layout applied: {layout}")
                    print(f"   Input: {input_file}")
                    print(f"   Output: {output_file}")
                    print(f"   Rows: {result['rows']}")
                    print(f"\n📋 Report: {result['report_path']}")
                    return 0
                except Exception as e:
                    print(f"❌ Layout application failed: {str(e)}")
                    return 1
            
            else:
                print(f"Unknown BOW operation: {operation}")
                print("Available: format, standard, layout")
                return 1
        
        # Regular commands below
        command = args[0]
        
        # -info command
        if command == '-info':
            if len(args) < 2:
                print("Usage: chalkml -info <file>")
                return 1
            success, message = engine.show_info(args[1])
            print(message)
            return 0 if success else 1
        
        # -undo command
        elif command == '-undo':
            success, message = engine.undo_last()
            print(message)
            return 0 if success else 1
        
        # -rm command
        elif command == '-rm':
            if len(args) < 3:
                print("Usage: chalkml -rm col|row [range] [position] <file>")
                return 1
            
            target = args[1]  # 'col' or 'row'
            
            # Check for range operation
            if len(args) >= 4 and args[2] == 'range':
                # chalkml -rm col range 01N:05N file.csv
                range_expr = args[3]
                file_path = args[4]
                success, message = engine.remove_range(file_path, target, range_expr)
            elif len(args) == 3:
                # No position: chalkml -rm col file.csv
                file_path = args[2]
                if target == 'col':
                    success, message = engine.remove_column(file_path, None)
                elif target == 'row':
                    success, message = engine.remove_row(file_path, None)
                else:
                    print(f"Unknown target: {target}. Use 'col' or 'row'")
                    return 1
            else:
                # With position: chalkml -rm col 01N file.csv
                position = args[2]
                file_path = args[3]
                if target == 'col':
                    success, message = engine.remove_column(file_path, position)
                elif target == 'row':
                    success, message = engine.remove_row(file_path, position)
                else:
                    print(f"Unknown target: {target}. Use 'col' or 'row'")
                    return 1
            
            print(message)
            return 0 if success else 1
        
        # -mv command
        elif command == '-mv':
            if len(args) < 5:
                print("Usage: chalkml -mv col from to <file>")
                return 1
            target = args[1]
            from_pos = args[2]
            to_pos = args[3]
            file_path = args[4]
            success, message = engine.move_column(file_path, from_pos, to_pos)
            print(message)
            return 0 if success else 1
        
        # -cp command
        elif command == '-cp':
            if len(args) < 5:
                print("Usage: chalkml -cp col from to <file>")
                return 1
            target = args[1]
            from_pos = args[2]
            to_pos = args[3]
            file_path = args[4]
            success, message = engine.copy_column(file_path, from_pos, to_pos)
            print(message)
            return 0 if success else 1
        
        # -rn command
        elif command == '-rn':
            if len(args) < 5:
                print("Usage: chalkml -rn col position \"name\" <file>")
                return 1
            target = args[1]
            position = args[2]
            new_name = args[3].strip('"\'')
            file_path = args[4]
            success, message = engine.rename_column(file_path, position, new_name)
            print(message)
            return 0 if success else 1
        
        # -fillsmart command
        elif command == '-fillsmart':
            if len(args) < 5:
                print("Usage: chalkml -fillsmart col position strategy <file>")
                print("Strategies: forward, backward, interpolate, mean, median, mode, knn")
                return 1
            target = args[1]
            position = args[2]
            strategy = args[3]
            file_path = args[4]
            success, message = engine.fill_smart(file_path, position, strategy)
            print(message)
            return 0 if success else 1
        
        # -derive command
        elif command == '-derive':
            if len(args) < 4:
                print("Usage: chalkml -derive \"newname\" \"formula\" <file>")
                return 1
            new_name = args[1].strip('"\'')
            formula = args[2].strip('"\'')
            file_path = args[3]
            success, message = engine.derive_column(file_path, new_name, formula)
            print(message)
            return 0 if success else 1
        
        # -onehot command
        elif command == '-onehot':
            if len(args) < 4:
                print("Usage: chalkml -onehot col position <file>")
                return 1
            target = args[1]
            position = args[2]
            file_path = args[3]
            prefix = None
            if '--prefix' in args:
                prefix_idx = args.index('--prefix')
                prefix = args[prefix_idx + 1]
            success, message = engine.one_hot_encode(file_path, position, prefix=prefix)
            print(message)
            return 0 if success else 1
        
        # -scale command
        elif command == '-scale':
            if len(args) < 4:
                print("Usage: chalkml -scale col position <file> [--method standard|minmax]")
                return 1
            target = args[1]
            position = args[2]
            file_path = args[3]
            method = 'standard'
            if '--method' in args:
                method_idx = args.index('--method')
                method = args[method_idx + 1]
            success, message = engine.scale_column(file_path, position, method=method)
            print(message)
            return 0 if success else 1
        
        # -map command
        elif command == '-map':
            if len(args) < 5:
                print("Usage: chalkml -map col position \"function\" <file>")
                return 1
            target = args[1]
            position = args[2]
            function = args[3].strip('"\'')
            file_path = args[4]
            success, message = engine.map_pattern(file_path, position, function)
            print(message)
            return 0 if success else 1
        
        # -reduce command
        elif command == '-reduce':
            if len(args) < 6:
                print("Usage: chalkml -reduce col positions operation result <file>")
                return 1
            target = args[1]
            positions = args[2].split(',')
            operation = args[3]
            result_name = args[4]
            file_path = args[5]
            success, message = engine.reduce_pattern(file_path, positions, operation, result_name)
            print(message)
            return 0 if success else 1
        
        # -stencil command
        elif command == '-stencil':
            if len(args) < 6:
                print("Usage: chalkml -stencil col position window operation <file>")
                return 1
            target = args[1]
            position = args[2]
            window_size = int(args[3])
            operation = args[4]
            file_path = args[5]
            success, message = engine.stencil_pattern(file_path, position, window_size, operation)
            print(message)
            return 0 if success else 1
        
        # -scan command
        elif command == '-scan':
            if len(args) < 5:
                print("Usage: chalkml -scan col position operation <file>")
                return 1
            target = args[1]
            position = args[2]
            operation = args[3]
            file_path = args[4]
            success, message = engine.scan_pattern(file_path, position, operation)
            print(message)
            return 0 if success else 1
        
        # -if command
        elif command == '-if':
            if len(args) < 7:
                print("Usage: chalkml -if col position \"condition\" if_value else_value <file>")
                return 1
            target = args[1]
            position = args[2]
            condition = args[3].strip('"\'')
            if_value = args[4]
            else_value = args[5]
            file_path = args[6]
            success, message = engine.apply_if_else(file_path, position, condition, if_value, else_value)
            print(message)
            return 0 if success else 1
        
        else:
            print(f"Unknown command: {command}")
            print("Use 'chalkml --help' for usage")
            return 1
    
    except Exception as e:
        print(f"Error: {str(e)}", file=sys.stderr)
        return 1


if __name__ == '__main__':
    sys.exit(main())
