"""
ChalkML Quick Start - Python API
=================================

This example demonstrates using ChalkML as a Python library.
"""

from chalkml import get_chalkml_engine
import pandas as pd

# Create sample data
df = pd.DataFrame({
    'name': ['Alice', 'Bob', 'Charlie', 'David'],
    'age': [25, None, 35, 40],
    'salary': [50000, 60000, 75000, None],
    'city': ['NYC', 'LA', 'Chicago', 'NYC'],
    'score': [85, 90, 92, 78]
})

df.to_csv('sample_data.csv', index=False)
print("✅ Sample data created\n")

# Initialize engine
engine = get_chalkml_engine()

# 1. Show dataset info
print("=" * 50)
print("Dataset Info")
print("=" * 50)
success, message = engine.show_info('sample_data.csv')
print(message)
print()

# 2. Fill missing age with mean
print("=" * 50)
print("Fill missing age with mean")
print("=" * 50)
success, message = engine.fill_smart('sample_data.csv', '02N', 'mean')
print(message)
print()

# 3. Fill missing salary with median
print("=" * 50)
print("Fill missing salary with median")
print("=" * 50)
success, message = engine.fill_smart('sample_data.csv', '03N', 'median')
print(message)
print()

# 4. Derive new column
print("=" * 50)
print("Feature Engineering: salary_per_age")
print("=" * 50)
success, message = engine.derive_column(
    'sample_data.csv',
    'salary_per_age',
    'col:03N/col:02N'
)
print(message)
print()

# 5. Apply MAP pattern
print("=" * 50)
print("MAP Pattern: Double the scores")
print("=" * 50)
success, message = engine.map_pattern(
    'sample_data.csv',
    '05N',
    'x * 2'
)
print(message)
print()

# 6. Apply REDUCE pattern
print("=" * 50)
print("REDUCE Pattern: Sum age and score")
print("=" * 50)
success, message = engine.reduce_pattern(
    'sample_data.csv',
    ['02N', '05N'],
    'sum',
    'age_score_sum'
)
print(message)
print()

# 7. IF/ELSE conditional
print("=" * 50)
print("IF/ELSE: Categorize salary")
print("=" * 50)
success, message = engine.apply_if_else(
    'sample_data.csv',
    '03N',
    'col:03N>60000',
    'High',
    'Low'
)
print(message)
print()

# Show final result
print("=" * 50)
print("Final Dataset")
print("=" * 50)
df_result = pd.read_csv('sample_data.csv')
print(df_result)
print()

print("✅ Examples complete!")
print("\nTo undo operations:")
print("  engine.undo_last()")
